﻿using System.Windows.Forms;
using System.Threading;
using System;
using SharpSvn;
using SharpSvn.UI;
using System.Collections.Generic;
using DependencyAnalyzer.Global;

namespace DependencyAnalyzer.Forms
{    
    public struct SvnData
    {
        public string revision;
        public string uri;
        public string localPath;
    }

    public partial class SvnSelectRevisionForm : Form
    {
        private Boolean cancelSvn;

        protected delegate void AddLogDelegate(string item);
        protected delegate void CloseFormDelegate();
        protected delegate void EnableDisableButtonsDelegate( bool status);

        private string localPath;
        private string uri;
        private List<string> revisions;
        private long revision;

        public SvnSelectRevisionForm(string localPath, string uri = null, List<string> revisions=null)
        {
            this.localPath = localPath;
            this.uri = uri;
            this.revisions = revisions;
            InitializeComponent();
        }

        public string getUri()
        {
            return uri;
        }

        public string getRevision()
        {
            return Convert.ToString(revision);
        }

        private void SvnForm_Load(object sender, EventArgs e)
        {
            if (uri != null)
            {
                txtUri.Text = uri;
                txtUri.ReadOnly = true;
            }
        }

        private void AddLog(string item)
        {
            if (txtLog.InvokeRequired)
            {
                BeginInvoke(new AddLogDelegate(AddLog), new object[] { item });
            }
            else
            {
                txtLog.AppendText(item + "\n");
            }
        }

        private void CloseForm()
        {
            if (txtLog.InvokeRequired)
            {
                BeginInvoke(new CloseFormDelegate(CloseForm), new object[] { });
            }
            else
            {
                cancelSvn = true;
                this.Close();
            }
        }

        private void EnableDisableButtons(bool status)
        {
            if (btnOk.InvokeRequired || btnSelectFromLog.InvokeRequired)
            {
                BeginInvoke(new EnableDisableButtonsDelegate(EnableDisableButtons), new object[] { status });
            }
            else
            {
                btnOk.Enabled = status;
                btnSelectFromLog.Enabled = status;
            }
        }

        private string GetHeadRevisionNumber()
        {
            using (SvnClient client = new SvnClient())
            {
                SvnInfoEventArgs info;
                Uri repos = new Uri(uri);

                client.GetInfo(repos, out info);

                return Convert.ToString(info.Revision);
            } 
        }

        private void UpdateSvn(object svnData)
        {
            SvnData data = (SvnData)svnData;

            //Disable btnOK
            EnableDisableButtons(false);

            using (SvnClient client = new SvnClient())
            {
                SvnUpdateArgs args = new SvnUpdateArgs();

                if (data.revision.ToUpper().Equals("HEAD"))
                {
                    //Check if the HEAD revision is not already part of the project
                    if (Functions.FindInArray(revisions, GetHeadRevisionNumber()))
                    {
                        MessageBox.Show("The HEAD revision is already part of the project!", "Revision error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        EnableDisableButtons(true);
                        return;
                    }
                }
                if (!data.revision.ToUpper().Equals("HEAD"))
                {
                    //First check if the revision is not already part of the project
                    if (Functions.FindInArray(revisions, data.revision))
                    {
                        MessageBox.Show("This revision is already part of the project!", "Revision error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        EnableDisableButtons(true);
                        return;
                    }
                    if (long.TryParse(data.revision, out revision))
                    {
                        //set the revision number if the user entered a valid number
                        args.Revision = new SvnRevision(revision);
                    }
                    else
                    {
                        MessageBox.Show("Revision error", "Invalid Revision number!", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        EnableDisableButtons(true);
                        return;
                    }
                }
                try
                {
                    SvnUIBindArgs uiArgs = new SvnUIBindArgs();
                    uiArgs.ParentWindow = this;
                    uiArgs.Synchronizer = this;
                    SvnUI.Bind(client, uiArgs);

                    //Create events for client
                    client.Processing += new EventHandler<SvnProcessingEventArgs>(client_Processing);
                    client.Notify += new EventHandler<SvnNotifyEventArgs>(client_Notify);
                    client.Cancel += new EventHandler<SvnCancelEventArgs>(client_Cancel);

                    //SvnUpdateResult provides info about what happened during a checkout
                    SvnUpdateResult result;

                    //Update the code in the specified directory             
                    if (client.Update(@data.localPath, args, out result))
                    {
                        revision = result.Revision;
                        DialogResult = DialogResult.OK;
                        CloseForm();
                    }
                }
                catch (SvnOperationCanceledException) { /* Do not show Cancel Exception */ }
                catch (SvnException se)
                {
                    AddLog(String.Format("SVN update error: {0}", se.Message));
                    MessageBox.Show(se.Message, "SVN update error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    EnableDisableButtons(true);
                }
            }
        }

        private void CheckOutSvn(object svnData)
        {
            SvnData data = (SvnData) svnData;

            //Disable btnOK
            EnableDisableButtons(false);

            using (SvnClient client = new SvnClient())
            {
                SvnCheckOutArgs args = new SvnCheckOutArgs();

                if (!data.revision.ToUpper().Equals("HEAD"))
                {
                    if (long.TryParse(data.revision, out revision))
                    {
                        //set the revision number if the user entered a valid number
                        args.Revision = new SvnRevision(revision);
                    }
                    else
                    {
                        MessageBox.Show("Revision error", "Invalid Revision number!", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        EnableDisableButtons(true);
                        return;
                    }
                }
                try
                {
                    SvnUIBindArgs uiArgs = new SvnUIBindArgs();
                    uiArgs.ParentWindow = this;
                    uiArgs.Synchronizer = this;
                    SvnUI.Bind(client, uiArgs);

                    //Create events for client
                    client.Processing += new EventHandler<SvnProcessingEventArgs>(client_Processing);
                    client.Notify += new EventHandler<SvnNotifyEventArgs>(client_Notify);
                    client.Cancel += new EventHandler<SvnCancelEventArgs>(client_Cancel);

                    //SvnUpdateResult provides info about what happened during a checkout
                    SvnUpdateResult result;

                    // Checkout the code to the specified directory             
                    if (client.CheckOut(new Uri(data.uri), @data.localPath, args, out result))
                    {
                        uri = data.uri;
                        revision = result.Revision;
                        DialogResult = DialogResult.OK;
                        CloseForm();
                    }
                }
                catch (SvnOperationCanceledException) { /* Do not show Cancel Exception */ }
                catch (SvnException se)
                {
                    AddLog(String.Format("SVN checkout error: {0}", se.Message));
                    MessageBox.Show(se.Message, "SVN checkout error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    EnableDisableButtons(true);
                }
            }
        }

        private void client_Cancel(object sender, SvnCancelEventArgs e)
        {
            if (cancelSvn)
                e.Cancel = true;
        }

        private void btnOk_Click(object sender, EventArgs e)
        {
            if (txtUri.Text.Length == 0)
            {
                MessageBox.Show("The 'Repository URI' cannot be empty.");
                return;
            }

            SvnData data = new SvnData();
            data.revision = txtRevision.Text;
            data.uri = txtUri.Text;
            data.localPath = localPath;

            if (uri == null)
            {
                Thread checkOutThread = new Thread(new ParameterizedThreadStart(this.CheckOutSvn));
                checkOutThread.Start(data);
            }
            else
            {
                Thread updateThread = new Thread(new ParameterizedThreadStart(this.UpdateSvn));
                updateThread.Start(data);
            }
        }

        void client_Notify(object sender, SvnNotifyEventArgs e)
        {
            this.AddLog(String.Format("{0}: {1}", e.Action.ToString().Remove(0, "Update".Length), e.FullPath));
        }

        void client_Processing(object sender, SvnProcessingEventArgs e)
        {
            Console.WriteLine(e.CommandType);
            if(e.CommandType.ToString().Equals("CheckOut"))
            {
                this.AddLog("Checkout/Update started!");
            }
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            DialogResult = DialogResult.Cancel;
            CloseForm();
        }

        private void btnSelectFromLog_Click(object sender, EventArgs e)
        {
            SvnRevisionForm selectRevision = new SvnRevisionForm(txtUri.Text);
            DialogResult result = selectRevision.ShowDialog(this);
            if (result == DialogResult.OK)
            {
                txtRevision.Text = selectRevision.GetRevision();
            }
            else if (result == DialogResult.Retry)
            {
                MessageBox.Show("Could not show the log, please check the URI.", "Could not show log!", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }

        }
    }
}
