﻿using System;
using System.Windows.Forms;
using System.Diagnostics;

namespace DependencyAnalyzer.CCCC
{
    public class Interface
    {
        private const string LANG = " --lang=\"c++\" --report_mask=PS -";

        private Process process = new Process();
        private int currentFile = 0;
        private int previous_percentage = 0;
        private string projectPath;
        private string[] files;

        public delegate void ProgressEventHandler(string message, int percentage);
        public event ProgressEventHandler ProgressUpdated;

        public Interface(string projectPath, string[] files)
        {
            this.projectPath = projectPath;
            this.files = files;
        }

        public void CalculateDependencies()
        {
            // CCCC complete
            if (this.ProgressUpdated != null)
            {
                ProgressUpdated("Extracting Dependencies...", 0);
            }

            // Initiate CCCC
            StartProcess(this.process);
            AddFiles(this.process, this.files);

            // Run CCCC
            process.StandardInput.Close();
            process.WaitForExit();
            process.Close();

            // CCCC complete
            if (this.ProgressUpdated != null)
            {
                ProgressUpdated("Dependency extraction complete!", 100);
            }
        }

        private void StartProcess(Process process)
        {
            // Start the CCCC app
            // Note: CCCC also uses the error stream as the output stream
            process.StartInfo.FileName = String.Format("{0}\\{1}",Application.StartupPath, Constants.CCCC_EXECUTABLE);
            process.StartInfo.WorkingDirectory = projectPath;
            process.StartInfo.Arguments = Interface.LANG;
            process.StartInfo.RedirectStandardInput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.CreateNoWindow = true;
            if (this.ProgressUpdated != null)
            {
                process.ErrorDataReceived += new DataReceivedEventHandler(ErrorOutputReceived);
            }
            process.Start();
            if (this.ProgressUpdated != null)
            {
                process.BeginErrorReadLine();
            }
        }

        private void AddFiles(Process process, string[] files)
        {
            foreach (string file in files)
            {
                process.StandardInput.WriteLine(file);
            }
        }

        private void ErrorOutputReceived(object sender, DataReceivedEventArgs e)
        {
            // Return if no data is available
            if (e.Data == null)
            {
                return;
            }

            // Check if this file falls within files bounds
            if (this.currentFile < this.files.Length)
            {
                InformDependencyProgress(e.Data);
            }
            
            // Check if dependency extraction is complete
            else if (this.currentFile >= this.files.Length)
            {
                if (previous_percentage == 100)
                {
                    this.ProgressUpdated("Generating reports...", 100);
                    previous_percentage = 0;
                }

                InformReportsProgress(e.Data);
            }
        }

        private void InformDependencyProgress(string data)
        {
            if (data.Contains(this.files[this.currentFile]))
            {
                this.currentFile++;
                int current_percentage = (int)((((double)this.currentFile) / ((double)this.files.Length)) * 100.0);
                if (this.previous_percentage != current_percentage)
                {
                    this.previous_percentage = current_percentage;
                    ProgressUpdated(null, current_percentage);
                }
            }
        }

        private void InformReportsProgress(string data)
        {
            int indexOfPercentage = data.IndexOf('%');
            if (indexOfPercentage != -1)
            {
                int percentage = Int32.Parse(data.Substring(0, indexOfPercentage));
                ProgressUpdated(null, percentage);
            }
        }
    }
}
