﻿using System;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Windows.Forms;
using System.Drawing;
using System.Text;
using System.Threading;
using System.Collections.Generic;

namespace DependencyAnalyzer.Global
{
    class MainFormMouseClickDetector
    {
        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern IntPtr SetWindowsHookEx(int idHook,
            LowLevelMouseProc lpfn, IntPtr hMod, uint dwThreadId);

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool UnhookWindowsHookEx(IntPtr hhk);

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern IntPtr CallNextHookEx(IntPtr hhk, int nCode,
            IntPtr wParam, IntPtr lParam);

        [DllImport("kernel32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern IntPtr GetModuleHandle(string lpModuleName);

        [DllImport("user32.dll", CharSet = CharSet.Auto, ExactSpelling = true)]
        public static extern IntPtr GetForegroundWindow();
        [return: MarshalAs(UnmanagedType.Bool)]
        
        [DllImport("user32", CharSet = CharSet.Ansi, SetLastError = true, ExactSpelling = true)]
        public static extern bool SetForegroundWindow(IntPtr hwnd);

        private enum MouseMessages
        {
            WM_LBUTTONDOWN = 0x0201,
            WM_LBUTTONUP = 0x0202,
            WM_MOUSEMOVE = 0x0200,
            WM_MOUSEWHEEL = 0x020A,
            WM_RBUTTONDOWN = 0x0204,
            WM_RBUTTONUP = 0x0205
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct POINT
        {
            public int x;
            public int y;
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct MSLLHOOKSTRUCT
        {
            public POINT pt;
            public uint mouseData;
            public uint flags;
            public uint time;
            public IntPtr dwExtraInfo;
        }

        private const int WH_MOUSE_LL = 14;

        private LowLevelMouseProc proc;
        private IntPtr hookID = IntPtr.Zero;
        private IntPtr handle;
        private Control control;
        private Point previousPoint = new Point(0, 0);
        private int previousTickCount = 0;

        public delegate void DoubleClickEventHandler();
        public event DoubleClickEventHandler SolidSXDoubleClicked;

        private delegate IntPtr LowLevelMouseProc(int nCode, IntPtr wParam, IntPtr lParam);

        // The control is only used for the relative location, so it doesn't matter if tab1 or tab2 is passed
        public MainFormMouseClickDetector(IntPtr handle, Control control)
        {
            this.proc = HookCallback;
            this.hookID = SetHook(this.proc);
            this.handle = handle;
            this.control = control;

            callBackPtr = new CallBackPtr(EvalWindow);
        }

        public void UnHook()
        {
            UnhookWindowsHookEx(this.hookID);
        }

        private IntPtr SetHook(LowLevelMouseProc proc)
        {
            using (Process curProcess = Process.GetCurrentProcess())
            using (ProcessModule curModule = curProcess.MainModule)
            {
                return SetWindowsHookEx(WH_MOUSE_LL, proc,
                    GetModuleHandle(curModule.ModuleName), 0);
            }
        }

        [DllImport("user32.dll")]
        private static extern
            bool ShowWindowAsync(IntPtr hWnd, int nCmdShow);
        [DllImport("user32.dll")]
        private static extern
            bool IsIconic(IntPtr hWnd);
        [DllImport("user32.dll")]
        private static extern
            bool IsZoomed(IntPtr hWnd);
        [DllImport("user32.dll")]
        private static extern
            IntPtr GetWindowThreadProcessId(IntPtr hWnd, IntPtr ProcessId);
        [DllImport("user32.dll")]
        private static extern
            IntPtr AttachThreadInput(IntPtr idAttach, IntPtr idAttachTo, int fAttach);

        [DllImport("user32.dll")]
        private static extern
            int GetWindowText(int hWnd, StringBuilder title, int size);
        [DllImport("user32.dll")]
        private static extern
            int GetWindowModuleFileName(int hWnd, StringBuilder title, int size);
        [DllImport("user32.dll")]
        private static extern
            int EnumWindows(CallBackPtr ewp, int lParam);
        [DllImport("user32.dll")]
        private static extern
            bool IsWindowVisible(int hWnd);

        private List<int> hWnds;
        private bool ignore;

        private bool EvalWindow(int hWnd, int lParam)
        {
            StringBuilder title = new StringBuilder(256);
            StringBuilder module = new StringBuilder(256);

            GetWindowModuleFileName(hWnd, module, 256);
            GetWindowText(hWnd, title, 256);

            hWnds.Add(hWnd);
            
            return true;
        }

        public delegate bool CallBackPtr(int hwnd, int lParam);
        private CallBackPtr callBackPtr;

        private List<int> ignoreList = new List<int>();

        private IntPtr HookCallback(int nCode, IntPtr wParam, IntPtr lParam)
        {
            // Check for double click
            int currentTickCount = Environment.TickCount;
            bool doubleClick = (currentTickCount - previousTickCount) < 300;
            this.previousTickCount = currentTickCount;

            // First check if the mouse up isn't from another process
            if ((MouseMessages.WM_LBUTTONUP == (MouseMessages)wParam) &&
                (this.handle != GetForegroundWindow()))
            {
                // Check if it is already on the ignore list
                if (ignoreList.Contains((int)GetForegroundWindow()))
                {
                    // And ignore the event
                    this.ignore = false;
                    SetForegroundWindow(this.handle);
                    return new IntPtr(-1);
                }

                // Otherwise find the process
                Process[] processes = System.Diagnostics.Process.GetProcesses();
                foreach (Process process in processes)
                {
                    // And check if it is another window
                    if (process.MainWindowHandle == GetForegroundWindow())
                    {
                        // In that case, leave it alone :)
                        return CallNextHookEx(this.hookID, nCode, wParam, lParam);
                    }
                }

                // Otherwise check all the window hanles
                this.hWnds = new List<int>();
                EnumWindows(callBackPtr, 0);
                if ((ignoreList.Contains((int)GetForegroundWindow())) || (!this.hWnds.Contains((int)GetForegroundWindow())) && this.ignore)
                {
                    // Add to ignore list for the future
                    ignoreList.Add((int)GetForegroundWindow());

                    // And ignore the event
                    this.ignore = false;
                    SetForegroundWindow(this.handle);
                    return new IntPtr(-1);
                }
            }

            // If control is not disposed
            if (control != null)
            {
                // Calculate if the click was in the SolidSX form
                Point locationOnForm = this.control.Parent.PointToScreen(control.Location);
                MSLLHOOKSTRUCT hookStruct = (MSLLHOOKSTRUCT)Marshal.PtrToStructure(lParam, typeof(MSLLHOOKSTRUCT));
                    
                bool inSolidSxWindow = (
                    hookStruct.pt.x >= locationOnForm.X &&
                    hookStruct.pt.x <= locationOnForm.X + control.Width &&
                    hookStruct.pt.y >= locationOnForm.Y &&
                    hookStruct.pt.y <= locationOnForm.Y + control.Height);

                // The click was on this window (not another overlapping window)
                if (inSolidSxWindow && (this.handle == GetForegroundWindow()))
                {
                    // Only detect mouseclick
                    if(MouseMessages.WM_LBUTTONDOWN == (MouseMessages)wParam)
                    {
                        Point currentPoint = new Point(hookStruct.pt.x, hookStruct.pt.y);
                        doubleClick = doubleClick && currentPoint.Equals(this.previousPoint);
                        this.previousPoint = currentPoint;
                        this.ignore = true;

                        // Only on doubleclick
                        if (doubleClick)
                        {
                            // Return -1 results in that event in NOT further processed (e.g. by SolidSx)
                            SolidSXDoubleClicked();
                            return new IntPtr(-1);
                        }
                        else
                        {
                            // Store return result so the focus can be set back on the control
                            IntPtr result = CallNextHookEx(this.hookID, nCode, wParam, lParam);
                            SetForegroundWindow(this.handle);
                            return result;
                        }
                    }
                }
            }

            return CallNextHookEx(this.hookID, nCode, wParam, lParam);
        }
    }
}
