﻿using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using DependencyAnalyzer.SolidSX.SQLiteGenerator;

namespace DependencyAnalyzer.SolidSX
{
    [Serializable()]
    public class Node : ISerializable
    {
        public enum NodeType { Directory, SourceFile, Internal, External, Module }
        public long ID { get; private set; }
        public string Name { get; private set; }
        public NodeType Type { get; private set; }
        public Node Parent { get; private set; }
        public List<Node> Children { get; private set; }

        public Node(long id, string name, Node parent, NodeType type)
        {
            this.ID = id;
            this.Name = name;
            this.Parent = parent;
            this.Type = type;
            this.Children = new List<Node>();
        }

        public Node(SerializationInfo info, StreamingContext ctxt)
        {
            this.ID = (long)info.GetValue("ID", typeof(long));
            this.Name = (string)info.GetValue("Name", typeof(string));
            this.Parent = (Node)info.GetValue("Parent", typeof(Node));
            this.Type = (NodeType)info.GetValue("Type", typeof(NodeType));
            this.Children = (List<Node>)info.GetValue("Children", typeof(List<Node>));
        }

        public void GetObjectData(SerializationInfo serializationInfo, StreamingContext streamingContext)
        {
            serializationInfo.AddValue("ID", this.ID);
            serializationInfo.AddValue("Name", this.Name);
            serializationInfo.AddValue("Parent", this.Parent);
            serializationInfo.AddValue("Type", this.Type);
            serializationInfo.AddValue("Children", this.Children);
        }

        public bool Contains(string name)
        {
            return Get(name) != null;
        }

        public Node Get(string name)
        {
            Node foundNode = null;
            foreach (Node node in this.Children)
            {
                if (node.Name.Equals(name))
                {
                    foundNode = node;
                    break;
                }
            }
            return foundNode;
        }

        private Node GetNodeByID(long id)
        {
            Node foundNode = null;
            foreach (Node node in this.Children)
            {
                if (node.ID == id)
                {
                    foundNode = node;
                    break;
                }
                else if ((foundNode = node.GetNodeByID(id)) != null)
                {
                    break;
                }
            }
            return foundNode;
        }

        private Node GetNodeByName(string name)
        {
            if (this.Name == name)
            {
                return this;
            }

            Node foundNode = null;
            foreach (Node node in this.Children)
            {
                if (node.Name == name)
                {
                    foundNode = node;
                    break;
                }
            }
            return foundNode;
        }

        public List<string> GetFilePathById(long id)
        {
            if (id <= SQLiteDataset.AMOUNT_OF_RESERVED_NODES)
            {
                return null;
            }
            else
            {
                Node foundNode = GetNodeByID(id);

                if (foundNode == null)
                {
                    return null;
                }

                List<string> filePath = new List<string>();
                filePath.Add(foundNode.Name);

                Node parentNode = foundNode.Parent;
                while (parentNode != null)
                {
                    filePath.Add(parentNode.Name);
                    parentNode = parentNode.Parent;
                }
                filePath.Reverse();
                return filePath;
            }
        }

        public long GetIdByFilePath(List<string> filePath)
        {
            if (filePath == null)
            {
                return -1;
            }

            Node node = this;
            foreach (string name in filePath)
            {
                node = node.GetNodeByName(name);
                if (node == null)
                {
                    return -1;
                }
            }

            return node.ID;
        }
    }
}
