﻿using System;
using System.Runtime.CompilerServices;
using System.IO;
using System.Xml;
using System.Collections.Generic;
using DependencyAnalyzer.Dependency;
using DependencyAnalyzer.Global;

namespace DependencyAnalyzer.CCCC
{
    public class ModuleResult
    {
        public string SourceFile { get; private set; }
        public string Name { get; private set; }
        public List<Reference> References { get; private set; }
        public List<Method> Methods { get; private set; }

        public ModuleResult(string sourceFile, string name, List<Reference> references, List<Method> methods)
        {
            this.SourceFile = sourceFile;
            this.Name = name;
            this.References = references;
            this.Methods = methods;
        }
    }

    public class Parser
    {
        private int currentNode = 0;

        public delegate void ProgressEventHandler(string message, int percentage);
        public event ProgressEventHandler ProgressUpdated;

        public List<Module> Modules { get; private set; }

        [MethodImpl(MethodImplOptions.Synchronized)]
        public ModuleResult ParseFile(string filename)
        {
            string temp = Functions.LongToShortPath(@"C:\Users\Erwin\Documents\DAprojects\test4584\.cccc\FieldActionBase.xml");

            // Load the xml document
            StreamReader streamReader = new StreamReader(filename, System.Text.Encoding.UTF8);
            XmlDocument document = new XmlDocument();
            document.Load(streamReader);

            // Load the sourcefile
            // document_root -> CCCC_Project -> reference_file
            String sourcefile = document.ChildNodes[2].ChildNodes[1].InnerText;

            // Load the correct node
            // document_root -> CCCC_Project -> procedural_detail -> module
            XmlNode procedural_node = document.ChildNodes[2].ChildNodes[0];
            List<Method> methods = ReadMethods(procedural_node);

            // document_root -> CCCC_Project -> structural_detail -> module
            XmlNode structural_node = document.ChildNodes[2].ChildNodes[2].ChildNodes[0];

            // Load the name, clients and suppliers
            // document_root -> CCCC_Project -> structural_detail -> module -> name
            String name = structural_node.ChildNodes[0].InnerText;
            // document_root -> CCCC_Project -> structural_detail -> module -> client_module
            List<Reference> clients = ReadRelationShips(structural_node.ChildNodes[1]);
            // document_root -> CCCC_Project -> structural_detail -> module -> supplier_module
            //List<Reference> suppliers = ReadRelationShips(node.ChildNodes[2]);

            streamReader.Close();

            return new ModuleResult(sourcefile, name, clients, methods);
        }

        public void ParseFiles(string[] filenames)
        {
            int percentage, previous_percentage = 0, fileCounter = 0;

            // Update process manager
            if (ProgressUpdated != null)
            {
                ProgressUpdated("Processing reports...", 0);
            }

            // Create a list containing all the methods
            this.Modules = new List<Module>(filenames.Length);

            // Retrieve all references from the files
            foreach(string filename in filenames)
            {
                // Parse the file
                // Skip cccc.xml as it doesn't contain modules
                if (!filename.EndsWith("cccc.xml"))
                {
                    ModuleResult moduleResult = ParseFile(filename);

                    // Create modules (clients, suppliers) based on the references
                    foreach (Reference reference in moduleResult.References)
                    {
                        SetClientsAndSuppliers(moduleResult, reference);
                    }

                    // Find the module and add the methods
                    if (moduleResult.Methods.Count > 0)
                    {
                        Module module = FindOrCreateModule(moduleResult.Name, moduleResult.SourceFile);
                        module.Methods.AddRange(moduleResult.Methods);
                    }

                    // Update process manager
                    fileCounter++;
                    percentage = (int)((((double)fileCounter) / ((double)filenames.Length)) * 100.0);
                    if ((ProgressUpdated != null) && (previous_percentage != percentage))
                    {
                        ProgressUpdated(null, percentage);
                        previous_percentage = percentage;
                    }
                }
            }

            // Remove all modules without dependencies
            List<Module> modulesToRemove = new List<Module>();
            foreach (Module module in Modules)
            {
                if ((module.Clients.Count == 0) && (module.Suppliers.Count == 0))
                {
                    modulesToRemove.Add(module);
                }
            }
            this.Modules.RemoveAll(new Predicate<Module>(
                delegate(Module module)
                {
                    return modulesToRemove.Contains(module);
                }
            ));

            // Update process manager
            if (ProgressUpdated != null)
            {
                ProgressUpdated("Processing complete!", 100);
            }
        }

        private void SetClientsAndSuppliers(ModuleResult moduleResult, Reference reference)
        {
            // Find or create the modules
            Module client = FindOrCreateModule(reference.Name, reference.SourceFile);
            Module supplier = FindOrCreateModule(moduleResult.Name, moduleResult.SourceFile);

            // Set the client and supplier in the reference
            reference.Client = client;
            reference.Supplier = supplier;

            // Add client to supplier and vice versa
            supplier.Clients.Add(reference);
            client.Suppliers.Add(reference);
        }

        private Module FindOrCreateModule(string modulename, string sourcefile)
        {
            // Find the module if it exists
            Module module = this.Modules.Find(
                delegate(Module _module)
                {
                    return _module.Name.Equals(modulename);
                }
            );

            // Create is otherwise and add the module to the list
            if (module == null)
            {
                module = new Module(this.Modules.Count + 1, modulename, sourcefile);
                this.Modules.Add(module);
            }

            return module;
        }

        private List<Reference> ReadRelationShips(XmlNode node)
        {
            List<Reference> relationships = new List<Reference>();
            this.currentNode = 0;
            
            while (currentNode < node.ChildNodes.Count)
            {
                List<Reference> relationship = null;
                if (node.ChildNodes[currentNode].Name.Equals("name"))
                {
                    relationship = ReadRelationship(node);
                    relationships.AddRange(relationship);
                }
            }
            return relationships;
        }

        private List<Reference> ReadRelationship(XmlNode node)
        {
            string name = node.ChildNodes[currentNode++].InnerText;
            currentNode++; //bool visible = Boolean.Parse(node.ChildNodes[currentNode++].InnerText);
            currentNode++; //bool concrete = Boolean.Parse(node.ChildNodes[currentNode++].InnerText);
            List<Reference> references = new List<Reference>();

            node = node.ChildNodes[currentNode];
            // Node maybe null because extent nodes are the last child nodes for every client module
            while ((node != null) && (node.Name.Equals("extent")))
            {
                Reference reference = ReadReference(node, name);
                references.Add(reference);
                node = node.ParentNode.ChildNodes[++currentNode];
            }

            return references;
        }

        private Reference ReadReference(XmlNode node, string name)
        {
            string description = node.ChildNodes[0].InnerText;
            string source_file = node.ChildNodes[1].Attributes.GetNamedItem("file").Value;
            int source_line = Int32.Parse(node.ChildNodes[1].Attributes.GetNamedItem("line").Value);
            return new Reference(name, description, source_file, source_line);
        }

        private List<Method> ReadMethods(XmlNode node)
        {
            List<Method> methods = new List<Method>();
            foreach (XmlNode child in node.ChildNodes)
            {
                Method method = new Method(child.ChildNodes[0].InnerText);
                methods.Add(method);
            }
            return methods;
        }
    }
}
