﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.CompilerServices;
using System.Threading;
using System.Windows.Forms;
using DependencyAnalyzer.CCCC;
using DependencyAnalyzer.Dependency;
using DependencyAnalyzer.Global;
using DependencyAnalyzer.SolidSX;
using DependencyAnalyzer.SolidSX.SQLiteGenerator;

namespace DependencyAnalyzer.Forms
{
    public partial class MainForm : Form
    {
        protected delegate void SetProgressDelegate(int percentage);
        protected delegate void LogMessageDelegate(string status);
        protected delegate void LoadProjectDelegate(string filePath = null);

        private MainFormMouseClickDetector mainFormMouseClickDetector;
        private SolidSxPanel pnlSolidSXCurrent;
        private List<Module> modules;
        private ProjectInformation project = null;
        private bool solidSXloaded = false;
        private bool isChecking = false;
        private long nodeIdSelected = 0;
        private bool nodeIdReceived = false;
        private Object locker = new Object();

        public MainForm()
        {
            InitializeComponent();
            this.Text = Constants.MAIN_TITLE;
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            LogMessage("Application started.");
            if (!solidSXloaded)
            {
                SolidSxPathForm solidSxPath = new SolidSxPathForm();
                if (solidSxPath.ShowDialog(this) == DialogResult.OK)
                {
                    pnSolidSx1.LoadSolidSX(solidSxPath.GetFilePath());
                    pnSolidSx2.LoadSolidSX(solidSxPath.GetFilePath());
                    pnlSolidSXCurrent = pnSolidSx1;
                    solidSXloaded = true;
                }
                else
                {
                    MessageBox.Show("The SolidSX executable is not selected, the application will be closed now.", "SolidSX not selected!", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    Application.Exit();
                }
            }
        }

        [MethodImpl(MethodImplOptions.Synchronized)]
        private void SolidSXDoubleClicked()
        {
            if (!nodeIdReceived)
            {
                // Wait until the node id is received
                lock (locker)
                {
                    //LogMessage("Waiting...");
                    Monitor.Wait(locker, 2000, true);
                    //LogMessage("Continue...");
                }
            }
            nodeIdReceived = false;
            if (this.tabControlRevisions.SelectedTab == this.tabRevision1)
            {
                //LogMessage("SolidSX Double Clicked On Tab 1 !!!");
                if (cbAutoExpandNodes.Checked)
                {
                    this.pnlSolidSXCurrent.ExpandNodeById(this.nodeIdSelected, this.pnSolidSx2);
                }
                else
                {
                    this.pnlSolidSXCurrent.ExpandNodeById(this.nodeIdSelected, null);
                }
            }
            else
            {
                //LogMessage("SolidSX Double Clicked On Tab 2 !!!");
                if (cbAutoExpandNodes.Checked)
                {
                    this.pnlSolidSXCurrent.ExpandNodeById(this.nodeIdSelected, this.pnSolidSx1);
                }
                else
                {
                    this.pnlSolidSXCurrent.ExpandNodeById(this.nodeIdSelected, null);
                }
            }
        }

        private void LoadProject(string filePath=null)
        {
            if (this.InvokeRequired)
            {
                BeginInvoke(new LoadProjectDelegate(LoadProject), new object[] { filePath });
            }
            else
            {
                if (project == null && filePath != null)
                {
                    project = new ProjectInformation();
                    if (!project.LoadProjectFile(filePath))
                    {
                        project = null;
                    }
                    else
                    {
                        //Enable close button, disable new and load project buttons
                        closeToolStripMenuItem.Enabled = true;
                        newToolStripMenuItem.Enabled = false;
                        loadToolStripMenuItem.Enabled = false;

                        LogMessage(String.Format("Project opened: {0}", project.projectName));
                    }
                }

                if (project != null)
                {
                    this.Text = String.Format("{0} - {1}", Constants.MAIN_TITLE, project.projectName);
                    
                    if (project.local)
                    {
                        //Disable the revision buttons
                        btnAdd.Enabled = false;
                        btnRemove.Enabled = false;

                        //Add only "Local folder" to the list
                        listRevisions.Items.Clear();
                        listRevisions.Items.Add(new ListViewItem("Local"));
                    }
                    else
                    {
                        //Add the revisions to the list
                        listRevisions.Items.Clear();
                        foreach (string revision in project.revisions)
                        {
                            listRevisions.Items.Add(revision);
                        }

                        //Enable the revision add button
                        btnAdd.Enabled = true;

                        //Enable the revision remove button only when there are more than 1 revision
                        if (listRevisions.Items.Count > 1)
                        {
                            btnRemove.Enabled = true;
                        }
                    }
                }
            }
        }

        private void CloseProject()
        {
            if (project != null)
            {
                //Set the title
                this.Text = Constants.MAIN_TITLE;

                //Disable the revision buttons
                btnAdd.Enabled = false;
                btnRemove.Enabled = false;

                //Clear the list
                listRevisions.Items.Clear();

                //Set back tabs
                tabControlRevisions.SelectedTab = tabRevision1;
                tabRevision1.Text = Constants.NO_REVISION_SELECTED;
                tabRevision2.Text = Constants.NO_REVISION_SELECTED;

                //Close XML's
                pnSolidSx1.CloseDataset();
                pnSolidSx2.CloseDataset();

                //Set project back to null
                project = null;

                //Disable close project, enable new and load project buttons
                closeToolStripMenuItem.Enabled = false;
                newToolStripMenuItem.Enabled = true;
                loadToolStripMenuItem.Enabled = true;

                //Add to log
                LogMessage("Project closed!");
            }
        }

        private void SetSolidSxMenu(bool status)
        {
            screenshotToolStripMenuItem.Enabled = status;
            viewToolStripMenuItem.Enabled = status;
        }

        private void fromLocalRepositoryToolStripMenuItem_Click(object sender, EventArgs e)
        {
            project = new ProjectInformation();

            if (new CreateProjectForm(project).ShowDialog(this) == DialogResult.OK)
            {
                if (folderBrowserDialog.ShowDialog(this) == DialogResult.OK)
                {
                    project.local = true;
                    project.localFolder = folderBrowserDialog.SelectedPath;
                    project.SaveProjectFile();
                    Thread extractDependenciesThread = new Thread(new ParameterizedThreadStart(ExtractDependencies));
                    extractDependenciesThread.Start();

                    //Enable close button, disable new and load project buttons
                    closeToolStripMenuItem.Enabled = true;
                    newToolStripMenuItem.Enabled = false;
                    loadToolStripMenuItem.Enabled = false;
                }
                else
                {
                    project = null;
                }
            }
        }

        private void aboutToolStripMenuItem1_Click(object sender, EventArgs e)
        {
            new AboutBox().Show();
        }

        private void fromSVNRepositoryToolStripMenuItem_Click(object sender, EventArgs e)
        {
            project = new ProjectInformation();

            if (new CreateProjectForm(project).ShowDialog(this) == DialogResult.OK)
            {
                SvnSelectRevisionForm svnForm;
                if ((svnForm = new SvnSelectRevisionForm(String.Format("{0}\\source", project.projectPath))).ShowDialog(this) == DialogResult.OK)
                {
                    LogMessage(String.Format("Checked out/updated revision {0} from {1}", svnForm.getRevision(), svnForm.getUri()));
                    project.local = false;
                    project.repositoryUri = svnForm.getUri();
                    project.revisions.Add(svnForm.getRevision());
                    project.SaveProjectFile();
                    Thread extractDependenciesThread = new Thread(new ParameterizedThreadStart(ExtractDependencies));
                    extractDependenciesThread.Start(svnForm.getRevision());

                    //Enable close button, disable new and load project buttons
                    closeToolStripMenuItem.Enabled = true;
                    newToolStripMenuItem.Enabled = false;
                    loadToolStripMenuItem.Enabled = false;
                }
                else
                {
                    project = null;
                }
            }   
        }

        private void btnAdd_Click(object sender, EventArgs e)
        {
            if (project != null)
            {
                SvnSelectRevisionForm svnForm;
                if ((svnForm = new SvnSelectRevisionForm(String.Format("{0}\\source", project.projectPath), project.repositoryUri, project.revisions)).ShowDialog(this) == DialogResult.OK)
                {
                    project.revisions.Add(svnForm.getRevision());
                    project.SaveProjectFile();
                    Thread extractDependenciesThread = new Thread(new ParameterizedThreadStart(ExtractDependencies));
                    extractDependenciesThread.Start(svnForm.getRevision());
                }
            }
        }

        private void btnRemove_Click(object sender, EventArgs e)
        {
            if (listRevisions.SelectedItems.Count == 1)
            {
                if (new QuestionForm("Are you sure?", "Do you really want to remove the selected revision?").ShowDialog(this) == DialogResult.OK)
                {
                    ListViewItem item = listRevisions.SelectedItems[0];
                    item.Checked = false;
                    File.Delete(String.Format("{0}\\dependencies\\{1}.xml", project.projectPath, item.Text));
                    File.Delete(String.Format("{0}\\dependencies\\{1}.bin", project.projectPath, item.Text));
                    project.revisions.Remove(item.Text);
                    project.SaveProjectFile();
                    listRevisions.SelectedItems[0].Remove();

                    if (listRevisions.Items.Count == 1)
                    {
                        btnRemove.Enabled = false;
                    }
                }
            }
            else
            {
                MessageBox.Show("Please first select a revision from the list.", "No revision selected!", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        private void ProgressUpdated(string message, int percentage)
        {
            if (message != null)
            {
                LogMessage(message);
            }

            if (percentage != -1)
            {
                SetProgressBar(percentage);
            }
        }

        private void SetProgressBar(int percentage)
        {
            if (pbProgressBar.InvokeRequired)
            {
                BeginInvoke(new SetProgressDelegate(SetProgressBar), new object[] { percentage });
            }
            else
            {
                this.pbProgressBar.Value = percentage;
            }
        }

        private void loadToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (openFileDialog.ShowDialog(this) == DialogResult.OK)
            {
                LoadProject(openFileDialog.FileName);
            }
        }

        private void fileTreeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            pnlSolidSXCurrent.ToggleFileTree();
        }

        public void LogMessage(string status)
        {
            if (txtLog.InvokeRequired)
            {
                BeginInvoke(new LogMessageDelegate(LogMessage), new object[] { status });
            }
            else
            {
                txtLog.AppendText(String.Format("{0}: {1}{2}", DateTime.Now.ToString(), status, Environment.NewLine));
            }
        }

        private void ExtractDependencies(object data=null)
        {
            string folderPath;
            string revision = "";

            if(project.local)
            {
                folderPath = project.localFolder;
            }
            else
            {
                folderPath = String.Format("{0}\\source", project.projectPath);
                revision = (string)data;
            }

            // Remove the .cccc folder if they already exists
            if(Directory.Exists(String.Format("{0}\\.cccc", project.projectPath)))
            {
                Directory.Delete(String.Format("{0}\\.cccc", project.projectPath), true);
            }
           
            string[] files = ReadDirectory.readDir(folderPath, Constants.SOURCE_FILTER);
            Interface ccccinterface = new Interface(project.projectPath, files);
            ccccinterface.ProgressUpdated += new Interface.ProgressEventHandler(ProgressUpdated);
            ccccinterface.CalculateDependencies();

            files = ReadDirectory.readDir(String.Format("{0}\\.cccc", project.projectPath), Constants.XML_FILTER);
            Parser ccccparser = new Parser();
            ccccparser.ProgressUpdated += new Parser.ProgressEventHandler(ProgressUpdated);
            ccccparser.ParseFiles(files);
            this.modules = ccccparser.Modules;

            // Generate SolidSX xml
            if (project.local)
            {
                //new XmlDataset(this.modules, String.Format("{0}\\dependencies\\Local.xml", project.projectPath)).Generate();
                ModuleSerializer.Serialize(String.Format("{0}\\dependencies\\Local.bin", project.projectPath), new SQLiteDataset(this.modules, String.Format("dependencies\\Local.sql"), "dependencies\\Local.db", project.projectPath).Generate());
            }
            else
            {
                //new XmlDataset(this.modules, String.Format("{0}\\dependencies\\{1}.xml", project.projectPath, revision)).Generate();
                ModuleSerializer.Serialize(String.Format("{0}\\dependencies\\{1}.bin", project.projectPath, revision), new SQLiteDataset(this.modules, String.Format("dependencies\\{0}.sql", revision), String.Format("dependencies\\{0}.db", revision), project.projectPath).Generate());
            }

            //Load the project
            SetProgressBar(0);
            LoadProject();
        }

        private void toolbarToolStripMenuItem_Click(object sender, EventArgs e)
        {
            pnlSolidSXCurrent.ToggleToolbar();
        }

        private void normalToolStripMenuItem1_Click(object sender, EventArgs e)
        {
            pnlSolidSXCurrent.ScreenshotToClipboard(false);
        }

        private void largeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            pnlSolidSXCurrent.ScreenshotToClipboard(true);
        }

        private void normalToolStripMenuItem_Click(object sender, EventArgs e)
        {
            pnlSolidSXCurrent.ScreenshotToBMP(false);
        }

        private void detailedToolStripMenuItem_Click(object sender, EventArgs e)
        {
            pnlSolidSXCurrent.ScreenshotToBMP(true);
        }

        private void listRevisions_ItemCheck(object sender, ItemCheckEventArgs e)
        {
            if (!isChecking)
            {
                isChecking = true;
                if (e.CurrentValue == CheckState.Unchecked)
                {
                    if (listRevisions.CheckedItems.Count >= 2)
                    {
                        e.NewValue = CheckState.Unchecked;
                        MessageBox.Show("Only two revisions can be selected at the same time!", "Cannot select!", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                    else
                    {
                        LogMessage(String.Format("Revision {0} selected.", listRevisions.Items[e.Index].Text));
                    }
                }
                isChecking = false;
            }
        }

        private void listRevisions_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (e.Item.Checked)
            {
                if (tabRevision1.Text.Equals(Constants.NO_REVISION_SELECTED))
                {
                    tabRevision1.Text = String.Format("Revision: {0}", e.Item.Text);
                    tabControlRevisions.SelectedTab = tabRevision1;
                    pnSolidSx1.OpenDataset(project.projectPath, e.Item.Text);
                    SetSolidSxMenu(true);
                }
                else
                {
                    tabRevision2.Text = String.Format("Revision: {0}", e.Item.Text);
                    tabControlRevisions.SelectedTab = tabRevision2;
                    pnSolidSx2.OpenDataset(project.projectPath, e.Item.Text);
                    SetSolidSxMenu(true);
                }

                if ((tabRevision1.Text != Constants.NO_REVISION_SELECTED) && (tabRevision2.Text != Constants.NO_REVISION_SELECTED))
                {
                    cbAutoExpandNodes.Enabled = true;
                }
            }
            else
            {
                if (tabRevision1.Text.Equals(String.Format("Revision: {0}", e.Item.Text)))
                {
                    tabRevision1.Text = Constants.NO_REVISION_SELECTED;
                    pnSolidSx1.CloseDataset();
                    if (tabRevision2.Text != Constants.NO_REVISION_SELECTED)
                    {
                        tabControlRevisions.SelectedTab = tabRevision2;
                    }
                    else
                    {
                        SetSolidSxMenu(false);
                    }
                }
                else
                {
                    tabRevision2.Text = Constants.NO_REVISION_SELECTED;
                    pnSolidSx2.CloseDataset();
                    tabControlRevisions.SelectedTab = tabRevision1;
                    if (tabRevision1.Text == Constants.NO_REVISION_SELECTED)
                    {
                        SetSolidSxMenu(false);
                    }
                }
                cbAutoExpandNodes.Enabled = false;
                cbAutoExpandNodes.Checked = false;
            }
        }

        private void tabControlRevisions_SelectedIndexChanged(object sender, EventArgs e)
        {
            pnlSolidSXCurrent = tabControlRevisions.SelectedTab == tabRevision1 ? pnSolidSx1 : pnSolidSx2;
            SetSolidSxMenu(tabControlRevisions.SelectedTab.Text != Constants.NO_REVISION_SELECTED);
        }

        private void closeToolStripMenuItem_Click(object sender, EventArgs e)
        {
            CloseProject();
        }

        /*
        private void NodeClickedSolidSx(SolidSxPanel sender, string nodeID)
        {
            if (cbAutoExpandNodes.Checked && (this.pnlSolidSXCurrent == sender))
            {
                if (sender == pnSolidSx1)
                {
                    pnSolidSx2.ExpandNodeByID(nodeID);
                }
                else
                {
                    pnSolidSx1.ExpandNodeByID(nodeID);
                }
            }
        }

        private void NodeSelectedSolidSx(SolidSxPanel sender, string nodeIDs)
        {
            if (cbAutoExpandNodes.Checked && (this.pnlSolidSXCurrent == sender))
            {
                if (sender == pnSolidSx1)
                {
                    pnSolidSx2.SelectNodeIDs(nodeIDs);
                }
                else
                {
                    pnSolidSx1.SelectNodeIDs(nodeIDs);
                }
            }
        }
         */

        [MethodImpl(MethodImplOptions.Synchronized)]
        private void NodeSelectionChanged(long nodeId, SolidSxPanel sender)
        {
            //if (cbAutoExpandNodes.Checked && (this.pnlSolidSXCurrent == sender))
            //{
            //    SolidSxPanel receiver;
            //    if (sender == pnSolidSx1)
            //    {
            //        receiver = pnSolidSx2;
            //    }
            //    else
            //    {
            //        receiver = pnSolidSx1;
            //    }

            //    System.Diagnostics.Debug.WriteLine("Get mouse click type");
            //    //if (mouseClick.GetMouseClick() == MouseClickType.DOUBLE_CLICK)
            //    //{
            //    //    receiver.ExpandNodeById(nodeId, sender);
            //    //}
            //}
            this.nodeIdSelected = nodeId;
            this.nodeIdReceived = true;
            //LogMessage(this.nodeIdSelected.ToString());

            lock (this.locker)
            {
                Monitor.Pulse(this.locker);
            }
        }

        private void cbAutoExpandNodes_CheckStateChanged(object sender, EventArgs e)
        {
            if (cbAutoExpandNodes.Checked)
            {
                if (new QuestionForm("Are you sure?", "Before \"Auto expand nodes\" can be activated,\n both circles will be collapsed. Are you sure?").ShowDialog(this) == DialogResult.OK)
                {
                    //Enable mouse hook
                    this.mainFormMouseClickDetector = new MainFormMouseClickDetector(this.Handle, this.tabRevision1);
                    this.mainFormMouseClickDetector.SolidSXDoubleClicked += SolidSXDoubleClicked;

                    pnSolidSx1.CollapseToMain();
                    pnSolidSx2.CollapseToMain();
                    LogMessage("Auto expand nodes activated.");
                }
                else
                {
                    cbAutoExpandNodes.Checked = false;
                }
            }
            else
            {
                // Disable mouse hook
                this.mainFormMouseClickDetector.UnHook();
            }
        }
    }
}
