﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using DependencyAnalyzer.Dependency;
using DependencyAnalyzer.Global;
using DependencyAnalyzer.SolidSX.XmlGenerator;
using System.Diagnostics;
using System.Windows.Forms;

namespace DependencyAnalyzer.SolidSX.SQLiteGenerator
{
    public class SQLiteDataset
    {
        private const int START_LEVEL = 0;
        public const int AMOUNT_OF_RESERVED_NODES = 4;

        private const long PROJECT_NODE = 1;
        private const long REVISION_NODE = 2;
        private const long EXTERNAL_NODE = 3;
        private const long INTERNAL_NODE = 4;

        private long currentNodeId = AMOUNT_OF_RESERVED_NODES + 1;
        private List<Module> modules;
        private string sqlPath;
        private string dbPath;
        private string projectPath;

        private StringBuilder nodeBuilder = new StringBuilder();
        private StringBuilder hierarchyBuilder = new StringBuilder();
        private StringBuilder edgeBuilder = new StringBuilder();

        private Node root;

        public SQLiteDataset(List<Module> modules, string sqlPath, string dbPath, string projectPath)
        {
            this.modules = modules;
            this.sqlPath = sqlPath;
            this.dbPath = dbPath;
            this.projectPath = projectPath;
        }

        [MethodImpl(MethodImplOptions.Synchronized)]
        public Node Generate()
        {
            AddDataSet();
            return this.root;
        }

        private void AddDataSet()
        {
            this.root = new Node(AMOUNT_OF_RESERVED_NODES, "Internal modules", null, Node.NodeType.Internal);
            
            // Build the query
            AddRootNodes();
            AddSourceNodes();
            AddSourceHierarchy(root);
            AddDirectories(root);
            AddModuleEdges();

            // Combine the different queries into one transaction
            StringBuilder query = new StringBuilder();
            query.Append("BEGIN TRANSACTION;");
            CreateStructure(query);
            query.Append(this.nodeBuilder);
            query.Append(this.hierarchyBuilder);
            query.Append(this.edgeBuilder);
            query.Append("COMMIT;");

            // Remove the old files
            File.Delete(this.projectPath + "\\" + this.sqlPath);
            File.Delete(this.projectPath + "\\" + this.dbPath);

            // Write the file to disk
            File.WriteAllText(this.projectPath + "\\" + this.sqlPath, query.ToString());

            // Save database
            Process process = new Process();
            process.StartInfo.FileName = "cmd.exe";
            process.StartInfo.Arguments = String.Format("/c {0} {1} < {2}", Functions.LongToShortPath(String.Format("{0}\\{1}", Application.StartupPath, Constants.SQLITE3_EXECUTABLE)), this.dbPath, this.sqlPath);
            process.StartInfo.WorkingDirectory = this.projectPath;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.CreateNoWindow = true;
            process.Start();
            process.WaitForExit();
            process.Close();

            // Remove the sql file
            File.Delete(this.projectPath + "\\" + this.sqlPath);
        }

        private void CreateStructure(StringBuilder stringBuilder)
        {
            // Create database structure
            stringBuilder.AppendLine("CREATE TABLE edgeattr (eid INTEGER KEY, key STRING, value STRING, type STRING);");
            stringBuilder.AppendLine("CREATE TABLE edges (eid INTEGER PRIMARY KEY, fromnid INTEGER, tonid INTEGER);");
            stringBuilder.AppendLine("CREATE TABLE hierarchy (hid INTEGER, nid INTEGER, parentnid INTEGER, childorder INTEGER);");
            stringBuilder.AppendLine("CREATE TABLE information(key TEXT, value);");
            stringBuilder.AppendLine("CREATE TABLE nodeattr (nid INTEGER KEY, key STRING, value STRING, type STRING);");

            stringBuilder.AppendLine("CREATE INDEX edgeattr_eid ON edgeattr (eid);");
            stringBuilder.AppendLine("CREATE INDEX hierarchy_parentnid ON hierarchy (parentnid);");
            stringBuilder.AppendLine("CREATE INDEX hierarchy_parentnid_hid ON hierarchy (parentnid,hid);");
            stringBuilder.AppendLine("CREATE INDEX nodeattr_nid ON nodeattr (nid);");
        }

        private void AddRootNodes()
        {
            SQLiteNode.Add(this.nodeBuilder, SQLiteDataset.PROJECT_NODE, "KDEOffice", "project");
            SQLiteNode.Add(this.nodeBuilder, SQLiteDataset.REVISION_NODE, "Dataset", "dataset");
            SQLiteNode.Add(this.nodeBuilder, SQLiteDataset.EXTERNAL_NODE, "External modules", "external modules");
            SQLiteNode.Add(this.nodeBuilder, SQLiteDataset.INTERNAL_NODE, "Internal modules", "internal modules");
        }

        private void AddSourceNodes()
        {
            // Add each module
            foreach (Module module in this.modules)
            {
                string[] folders = module.SourceFile.Split(new char[] { '\\' });
                string filename = folders[folders.Length - 1];
                SQLiteNode.Add(this.nodeBuilder, module.ID + AMOUNT_OF_RESERVED_NODES, module.Name, "module");
                this.currentNodeId++;
            }

            // Add each module's method
            foreach (Module module in this.modules)
            {
                foreach (Method method in module.Methods)
                {
                    SQLiteNode.Add(this.nodeBuilder, this.currentNodeId, method.ShortName, "method", method.FullName);
                    method.Id = this.currentNodeId;
                    this.currentNodeId++;
                }
            }
        }

        private void AddSourceHierarchy(Node root)
        {
            // The default nodes
            SQLiteEdge.Add(this.hierarchyBuilder, 0, 1);
            SQLiteEdge.Add(this.hierarchyBuilder, 1, 2);
            SQLiteEdge.Add(this.hierarchyBuilder, 2, 3);
            SQLiteEdge.Add(this.hierarchyBuilder, 2, 4);

            // Link the modules hierarchy
            foreach (Module module in modules)
            {
                if (module.SourceFile.Length == 0)
                {
                    // External module
                    SQLiteEdge.Add(this.hierarchyBuilder, SQLiteDataset.EXTERNAL_NODE, module.ID + AMOUNT_OF_RESERVED_NODES);
                }
                else
                {
                    // Internal module
                    // Get the directory node from the existing directory tree for the given node
                    Node fileNode = AddToHierarchy(root, Functions.ShortToLongPath(module.SourceFile));
                    
                    // Link the module node to the filenode if it isn't linked previously
                    if (fileNode.Children.Count == 0)
                    {
                        SQLiteEdge.Add(this.hierarchyBuilder, fileNode.Parent.ID, fileNode.ID);
                    }

                    // Link the filenode to the directory node
                    SQLiteEdge.Add(this.hierarchyBuilder, fileNode.ID, module.ID + AMOUNT_OF_RESERVED_NODES);

                    // Add the module to the filenode (-1 because this id is not used)
                    fileNode.Children.Add(new Node(module.ID, module.Name, fileNode, Node.NodeType.Module));
                
                    // Add the module's methods to the module
                    foreach (Method method in module.Methods)
                    {
                        SQLiteEdge.Add(this.hierarchyBuilder, module.ID + AMOUNT_OF_RESERVED_NODES, method.Id);
                    }
                }
            }
        }

        private Node AddToHierarchy(Node root, string filePath)
        {
            // Split the folders for the hierarchy level
            string[] folders = filePath.Split(new char[] { '\\' });

            // Store the current node;
            Node node = root;

            // folders.Length - 1 because of not adding the file node
            for (int hierarchyLevel = START_LEVEL;hierarchyLevel < folders.Length - 1;hierarchyLevel++)
            {
                String filefolder = folders[hierarchyLevel];
                if (!node.Contains(filefolder))
                {
                    // Create the new level if it doesn't exists
                    node.Children.Add(new Node(this.currentNodeId, filefolder, node, Node.NodeType.Directory));
                    this.currentNodeId++;
                }

                // Go one level down
                node = node.Get(filefolder);
            }

            // Create the file node if it doesn't exists
            string filename = folders[folders.Length - 1];
            if (!node.Contains(filename))
            {
                SQLiteNode.Add(this.nodeBuilder, this.currentNodeId, filename, "sourcefile", filePath);
                // Add the file node to the tree
                node.Children.Add(new Node(this.currentNodeId, filename, node, Node.NodeType.SourceFile));
                this.currentNodeId++;
            }

            // Returns the node that represents the file and contains the module
            return node.Get(filename);
        }

        private void AddDirectories(Node root)
        {
            AddDirectoriesRecursive(root);
        }

        private void AddDirectoriesRecursive(Node root)
        {
            foreach (Node node in root.Children)
            {
                // Skip the lowest nodes, these are the sourcefiles
                if (!node.Type.Equals(Node.NodeType.SourceFile))
                {
                    if (node.Children.Count != 0)
                    {
                        // Add the subdirectories recursive
                        AddDirectoriesRecursive(node);
                    }

                    // Only add node if not root
                    if (node.ID != SQLiteDataset.INTERNAL_NODE)
                    {
                        SQLiteNode.Add(this.nodeBuilder, node.ID, node.Name, "directory");
                    }

                    // Add the hierarchy link
                    SQLiteEdge.Add(this.hierarchyBuilder, node.Parent.ID, node.ID);
                }
            }
        }

        public void AddModuleEdges()
        {
            int counter = 1;
            foreach (Module module in modules)
            {
                foreach (Reference reference in module.Clients)
                {
                    SQLiteEdge.Add(this.edgeBuilder, counter, reference.Client.ID + AMOUNT_OF_RESERVED_NODES, reference.Supplier.ID + AMOUNT_OF_RESERVED_NODES, reference.Description);
                    counter++;
                }
            }
        }
    }
}
