﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.CompilerServices;
using System.Text;
using System.Xml;
using DependencyAnalyzer.Dependency;
using DependencyAnalyzer.Global;

namespace DependencyAnalyzer.SolidSX.SQLiteGenerator
{
    public class SQLiteDatabase
    {
        private const int START_LEVEL = 0;
        public const int AMOUNT_OF_RESERVED_NODES = 4;

        private const long PROJECT_NODE = 1;
        private const long REVISION_NODE = 2;
        private const long EXTERNAL_NODE = 3;
        private const long INTERNAL_NODE = 4;

        private StringWriter stringWriter;
        private long currentNodeId = AMOUNT_OF_RESERVED_NODES + 1;
        private List<Module> modules;
        private string filePath;

        public SQLiteDatabase(List<Module> modules, string filePath)
        {
            this.modules = modules;
            this.filePath = filePath;
        }

        /*
        [MethodImpl(MethodImplOptions.Synchronized)]
        public void Generate()
        {
            //Generate SQLite database
            SQLiteConnection.CreateFile("c:\\mydatabasefile.db3");

            this.stringWriter = new StringWriterWithEncoding(Encoding.UTF8);
            using (XmlTextWriter xmlTextWriter = new XmlTextWriter(this.stringWriter))
            {
                xmlTextWriter.Formatting = Formatting.Indented;
                xmlTextWriter.WriteStartDocument();
                xmlTextWriter.WriteStartElement("dataset");
                AddDataSet(xmlTextWriter);
                xmlTextWriter.WriteEndElement();
                xmlTextWriter.WriteEndDocument();

                using (StreamWriter streamWriter = new StreamWriter(this.filePath))
                {
                    streamWriter.Write(stringWriter.GetStringBuilder().ToString());
                }
            }
        }

        private void AddDataSet(XmlTextWriter xmlTextWriter)
        {
            Node root = new Node(AMOUNT_OF_RESERVED_NODES, "Internal modules", null, Node.NodeType.Internal);
            
            xmlTextWriter.WriteStartElement("nodes");
            AddRootNodes(xmlTextWriter);
            AddSourceNodes(xmlTextWriter);
            xmlTextWriter.WriteEndElement();

            xmlTextWriter.WriteStartElement("hierarchy");
            xmlTextWriter.WriteAttributeString("name", "default");
            AddSourceHierarchy(xmlTextWriter, root);
            AddDirectories(root);
            xmlTextWriter.WriteEndElement();

            xmlTextWriter.WriteStartElement("edgeset");
            xmlTextWriter.WriteAttributeString("name", "default");
            AddModuleEdges(xmlTextWriter);
            xmlTextWriter.WriteEndElement();            
        }

        private void AddRootNodes(XmlTextWriter xmlTextWriter)
        {
            XmlNode.Add(xmlTextWriter, XmlDataset.PROJECT_NODE, "KDEOffice", "project");
            XmlNode.Add(xmlTextWriter, XmlDataset.REVISION_NODE, "Dataset", "dataset");
            XmlNode.Add(xmlTextWriter, XmlDataset.EXTERNAL_NODE, "External modules", "external modules");
            XmlNode.Add(xmlTextWriter, XmlDataset.INTERNAL_NODE, "Internal modules", "internal modules");
        }

        private void AddSourceNodes(XmlTextWriter xmlTextWriter)
        {
            // Add each module
            foreach (Module module in this.modules)
            {
                string[] folders = module.SourceFile.Split(new char[] { '\\' });
                string filename = folders[folders.Length - 1];
                XmlNode.Add(xmlTextWriter, module.ID + AMOUNT_OF_RESERVED_NODES, module.Name, "module");
                this.currentNodeId++;
            }

            // Add each module's method
            foreach (Module module in this.modules)
            {
                foreach (Method method in module.Methods)
                {
                    XmlNode.Add(xmlTextWriter, this.currentNodeId, method.ShortName, "method", method.FullName);
                    method.Id = this.currentNodeId;
                    this.currentNodeId++;
                }
            }
        }

        private void AddSourceHierarchy(XmlTextWriter xmlTextWriter, Node root)
        {
            // The default nodes
            XmlEdge.Add(xmlTextWriter, 1, 2);
            XmlEdge.Add(xmlTextWriter, 2, 3);
            XmlEdge.Add(xmlTextWriter, 2, 4);

            // Link the modules hierarchy
            foreach (Module module in modules)
            {
                if (module.SourceFile.Length == 0)
                {
                    // External module
                    XmlEdge.Add(xmlTextWriter, XmlDataset.EXTERNAL_NODE, module.ID + AMOUNT_OF_RESERVED_NODES);
                }
                else
                {
                    // Internal module
                    // Get the directory node from the existing directory tree for the given node
                    Node fileNode = AddToHierarchy(root, Functions.ShortToLongPath(module.SourceFile));
                    
                    // Link the module node to the filenode if it isn't linked previously
                    if (fileNode.Children.Count == 0)
                    {
                        XmlEdge.Add(xmlTextWriter, fileNode.Parent.ID, fileNode.ID);
                    }

                    // Link the filenode to the directory node
                    XmlEdge.Add(xmlTextWriter, fileNode.ID, module.ID + AMOUNT_OF_RESERVED_NODES);

                    // Add the module to the filenode (-1 because this id is not used)
                    fileNode.Children.Add(new Node(-1, module.Name, fileNode, Node.NodeType.Module));
                
                    // Add the module's methods to the module
                    foreach (Method method in module.Methods)
                    {
                        XmlEdge.Add(xmlTextWriter, module.ID + AMOUNT_OF_RESERVED_NODES, method.Id);
                    }
                }
            }
        }

        private Node AddToHierarchy(Node root, string filePath)
        {
            // Split the folders for the hierarchy level
            string[] folders = filePath.Split(new char[] { '\\' });

            // Store the current node;
            Node node = root;

            // folders.Length - 1 because of not adding the file node
            for (int hierarchyLevel = START_LEVEL;hierarchyLevel < folders.Length - 1;hierarchyLevel++)
            {
                String filefolder = folders[hierarchyLevel];
                if (!node.Contains(filefolder))
                {
                    // Create the new level if it doesn't exists
                    node.Children.Add(new Node(this.currentNodeId, filefolder, node, Node.NodeType.Directory));
                    this.currentNodeId++;
                }

                // Go one level down
                node = node.Get(filefolder);
            }

            // Create the file node if it doesn't exists
            string filename = folders[folders.Length - 1];
            if (!node.Contains(filename))
            {
                // Add the file node to the XML
                using (StringWriter nodesXmlStringWriter = new StringWriter())
                {
                    using (XmlTextWriter nodesXmlTextWriter = new XmlTextWriter(nodesXmlStringWriter))
                    {
                        nodesXmlTextWriter.Formatting = Formatting.Indented;
                        XmlNode.Add(nodesXmlTextWriter, this.currentNodeId, filename, "sourcefile", filePath);
                        this.stringWriter.GetStringBuilder().Insert(61, nodesXmlStringWriter);
                    }
                }

                // Add the file node to the tree
                node.Children.Add(new Node(this.currentNodeId, filename, node, Node.NodeType.SourceFile));
                this.currentNodeId++;
            }

            // Returns the node that represents the file and contains the module
            return node.Get(filename);
        }

        private void AddDirectories(Node root)
        {
            using (StringWriter
                nodesXmlStringWriter = new StringWriter(),
                linksXmlStringWriter = new StringWriter())
            {
                using(XmlTextWriter
                    nodesXmlTextWriter = new XmlTextWriter(nodesXmlStringWriter),
                    linksXmlTextWriter = new XmlTextWriter(linksXmlStringWriter))
                {
                    nodesXmlTextWriter.Formatting = Formatting.Indented;
                    linksXmlTextWriter.Formatting = Formatting.Indented;
                    AddDirectoriesRecursive(root, nodesXmlTextWriter, linksXmlTextWriter);
                    this.stringWriter.GetStringBuilder().Insert(61, nodesXmlStringWriter);
                    this.stringWriter.GetStringBuilder().Append(linksXmlStringWriter);
                }
            }
        }

        private void AddDirectoriesRecursive(Node root, XmlTextWriter nodesXmlTextWriter, XmlTextWriter linksXmlTextWriter)
        {
            foreach (Node node in root.Children)
            {
                // Skip the lowest nodes, these are the sourcefiles
                if (!node.Type.Equals(Node.NodeType.SourceFile))
                {
                    if (node.Children.Count != 0)
                    {
                        // Add the subdirectories recursive
                        AddDirectoriesRecursive(node, nodesXmlTextWriter, linksXmlTextWriter);
                    }

                    // Only add node if not root
                    if (node.ID != XmlDataset.INTERNAL_NODE)
                    {
                        XmlNode.Add(nodesXmlTextWriter, node.ID, node.Name, "directory");
                    }

                    // Add the hierarchy link
                    XmlEdge.Add(linksXmlTextWriter, node.Parent.ID, node.ID);
                }
            }
        }

        public void AddModuleEdges(XmlTextWriter xmlTextWriter)
        {
            int counter = 1;
            foreach (Module module in modules)
            {
                foreach (Reference reference in module.Clients)
                {
                    XmlEdge.Add(xmlTextWriter, counter, reference.Client.ID + AMOUNT_OF_RESERVED_NODES, reference.Supplier.ID + AMOUNT_OF_RESERVED_NODES, reference.Description);
                    counter++;
                }
            }
        }
         */
    }
}
