﻿using System;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Windows.Forms;
using System.Text;
using System.Threading;
using System.Drawing;
using DependencyAnalyzer.SolidSX;
using System.Collections.Generic;
using DependencyAnalyzer.SolidSX.SQLiteGenerator;

namespace DependencyAnalyzer.SolidSX
{
    public struct COPYDATASTRUCT
    {
        public IntPtr dwData;
        public int cbData;
        [MarshalAs(UnmanagedType.LPStr)]
        public string lpData;
    }

    public class SolidSxPanel : Panel
    {
        //Sets a window to be a child window of another window
        [DllImport("user32.dll", SetLastError = true)]
        private static extern uint SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

        //Sets window attributes 
        [DllImport("user32.dll")]
        private static extern int SetWindowLong(IntPtr hWnd, int nIndex, int dwNewLong);

        //Gets window attributes 
        [DllImport("user32.dll")]
        private static extern int GetWindowLong(IntPtr hWnd, int nIndex);

        //Send show commands
        [DllImport("user32.dll")]
        private static extern bool ShowWindow(IntPtr hWnd, int nCmdShow);

        //Find window
        [DllImport("user32.dll")]
        private static extern int FindWindow(string lpClassName, string lpWindowName);

        //Set foreground window
        [DllImport("user32.dll")]
        private static extern int SetForegroundWindow(int hWnd);

        //Send Keyboard event
        [DllImport("user32.dll")]
        private static extern void keybd_event(byte bVk, byte bScan, int dwFlags, int dwExtraInfo);

        //Menu functions
        [DllImport("user32.dll")]
        private static extern IntPtr GetMenu(IntPtr hWnd);

        [DllImport("user32.dll")]
        private static extern int GetMenuItemCount(IntPtr hMenu);

        [DllImport("user32.dll")]
        private static extern bool RemoveMenu(IntPtr hMenu, uint uPosition, uint Flags);

        [DllImport("user32.dll")]
        private static extern bool DrawMenuBar(IntPtr hWnd);

        private const int GWL_STYLE = -16;
        private const int WS_BORDER = 0x00800000; //window with border 
        private const int WS_DLGFRAME = 0x00400000; //window with double border but no title 
        private const int WS_CAPTION = WS_BORDER | WS_DLGFRAME; //window with a title bar 
        private const int SW_MAXIMIZE = 3;
        private const uint MF_BYPOSITION = 0x400;
        private const uint MF_REMOVE = 0x1000;
        private const int WM_COPYDATA = 0x004a;
        
        //SendMessage function
        [DllImport("user32.dll")]
        public extern static int SendMessage(IntPtr hWnd, int Msg, int wParam, ref COPYDATASTRUCT lParam);

        //Functions needed for resize
        [DllImport("user32.dll")]
        static extern IntPtr GetWindow(IntPtr hWnd, GetWindow_Cmd uCmd);

        [DllImport("user32.dll")]
        internal static extern void MoveWindow(IntPtr hwnd, int X, int Y, int nWidth, int nHeight, bool bRepaint);

        private enum GetWindow_Cmd : uint
        {
            GW_HWNDFIRST = 0,
            GW_HWNDLAST = 1,
            GW_HWNDNEXT = 2,
            GW_HWNDPREV = 3,
            GW_OWNER = 4,
            GW_CHILD = 5,
            GW_ENABLEDPOPUP = 6
        }

        public delegate void NodeSelectionChangedDelegate(long nodeId, SolidSxPanel sender);
        public event NodeSelectionChangedDelegate NodeSelectionChanged;

        IntPtr solidSXhandle = IntPtr.Zero;
        private bool toolbar;
        private bool DatasetOpened;
        private string filePath;

        private Node root;

        public SolidSxPanel()
        {
            DatasetOpened = false;
            Visible = false;
        }

        public void OpenDataset(string filePath, string fileName)
        {
            Visible = true;
            this.root = (Node)ModuleSerializer.DeSerialize(String.Format("{0}\\dependencies\\{1}.bin", filePath, fileName));
            this.filePath = String.Format("{0}\\dependencies\\{1}.db", filePath, fileName).Replace("\\", "\\\\");
            SendMessage(String.Format("ensureOpen('{0}')", this.filePath));
            SendMessage("openView('heb','HEB')");
            SendMessage("setNodeColorMap('Rainbow')");
            toolbar = true;
            ToggleFileTree();
            ToggleToolbar();
            DatasetOpened = true;
        }

        public void CloseDataset()
        {
            root = null;
            DatasetOpened = false;
            Visible = false;
        }

        public void ToggleFileTree()
        {
            SendMessage("showPane('tree',None)");
        }

        public void ToggleToolbar()
        {
            SendMessage("showPane('toolbar',None)");
            toolbar = !toolbar;
        }

        public void ScreenshotToClipboard(bool large)
        {
            if (DatasetOpened)
            {
                SendMessage(String.Format("makeScreenshot(None, {0})", large.ToString()));
                MessageBox.Show("A screenshot of the graph is placed on the clipboard.", "Screenshot on clipboard", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
        }

        public void ScreenshotToBMP(bool large)
        {
            if (DatasetOpened)
            {
                SendMessage(String.Format("makeScreenshot('{0}\\\\{1}.bmp', {2})", Environment.GetFolderPath(Environment.SpecialFolder.Desktop).Replace("\\", "\\\\"), DateTime.Now.ToString().Replace(':', '.'), large.ToString()));
                MessageBox.Show("A BMP file with the screenshot of the graph is placed on the desktop.", "BMP file on desktop", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
        }

        private long GetIdByFilePath(List<string> filePath)
        {
            return root.GetIdByFilePath(filePath);
        }

        private List<string> GetFilePathById(long id)
        {
            return root.GetFilePathById(id);
        }

        public void ExpandNodeById(long nodeId, SolidSxPanel receiver)
        {
            // Open in current panel
            SendMessage(String.Format("expandByNid({0})", nodeId));

            // Open same node in other panel
            if (receiver != null)
            {
                if (nodeId <= SQLiteDataset.AMOUNT_OF_RESERVED_NODES)
                {
                    receiver.SendMessage(String.Format("expandByNid({0})", nodeId.ToString()));
                }
                else
                {
                    long id = receiver.GetIdByFilePath(GetFilePathById(nodeId));
                    if (id != -1)
                    {
                        receiver.SendMessage(String.Format("expandByNid({0})", id));
                    }
                    else
                    {
                        Debug.WriteLine("-1");
                    }
                }
            }
        }

        public void CollapseToMain()
        {
            if (DatasetOpened)
            {
                SendMessage("collapseByNid(4, recursive=True)");
                SendMessage("collapseByNid(3, recursive=True)");
                SendMessage("clearNodeSelection()");
                SendMessage("clearEdgeSelection()");
            }
        }

        protected override void OnResize(EventArgs e)
        {
            ResizeSolidSX();
            base.OnResize(e);
        }

        protected override void OnCreateControl()
        {
            ResizeSolidSX();
            base.OnCreateControl();
        }

        private void ResizeSolidSX()
        {
            IntPtr childwindow = GetWindow(this.Handle, GetWindow_Cmd.GW_CHILD);
            #if DEBUG
                //Visible statusbar
                MoveWindow(childwindow, -8, -35, this.Size.Width + 16, this.Size.Height + 45, true);
            #else
                //Invisible statusbar
                MoveWindow(childwindow, -8, -35, this.Size.Width + 16, this.Size.Height + 66, true);
            #endif

            if (!toolbar && DatasetOpened)
            {
                SendMessage("showPane('toolbar',False)");
            }
        }

        protected override void OnHandleDestroyed(EventArgs e)
        {
            if (this.solidSXhandle != IntPtr.Zero)
            {
                SendMessage("exit()");
            }
            base.OnHandleDestroyed(e);
        }

        public void LoadSolidSX(string filePath)
        {
            string solidSxPath = filePath.Substring(0, filePath.LastIndexOf('\\'));
            string solidSxFile = filePath.Substring(solidSxPath.Length + 1, filePath.Length - solidSxPath.Length - 1);

            ProcessStartInfo startInfo = new ProcessStartInfo(solidSxFile);
            startInfo.WorkingDirectory = solidSxPath;
            startInfo.Arguments = String.Format("--sendto {0} --listento 0", this.Handle);
            Process p = Process.Start(startInfo);

            //Wait for input
            p.WaitForInputIdle();

            //Get focus
            SetForegroundWindow((int)p.MainWindowHandle);

            //Send enter to close the about window
            keybd_event(0x0D, 0, 0, 0);

            //Put main window into the panel
            PutWindowInPanel(this);

            //Resize
            ResizeSolidSX();
        }
         

        private void PutWindowInPanel(Panel panel)
        {
            while (true)
            {
                foreach (Process proc in Process.GetProcesses())
                {
                    if (proc.MainWindowTitle.StartsWith(Constants.SOLIDSX_WINDOW_TITLE))
                    {
                        IntPtr handle = proc.MainWindowHandle;

                        //Put main program into the panel
                        SetParent(handle, panel.Handle);
                        
                        //Remove the titelbar
                        int style = GetWindowLong(handle, GWL_STYLE);
                        SetWindowLong(handle, GWL_STYLE, (style & ~WS_DLGFRAME));

                        //Remove menubar
                        RemoveMenubar(handle);
                        return;
                    }
                }
                Thread.Sleep(10);
            }
        }

        private void RemoveMenubar(IntPtr handle)
        {
            //Get menu
            IntPtr menuHandler = GetMenu(handle);

            //Get items
            int count = GetMenuItemCount(menuHandler);

            //Remove menu items
            for (int i = 0; i < count; i++)
            {
                RemoveMenu(menuHandler, 0, (MF_BYPOSITION | MF_REMOVE));
            }

            //Redraw menu
            DrawMenuBar(handle);
        }

        protected override void WndProc(ref Message m)
        {
            if (m.Msg == WM_COPYDATA)
            {
                COPYDATASTRUCT data = new COPYDATASTRUCT();
                data = (COPYDATASTRUCT)m.GetLParam(typeof(COPYDATASTRUCT));
                string dataStr = data.lpData.Substring(0, data.cbData);
                ProcessMessage(dataStr);
            }
            base.WndProc(ref m);
        }

        private void ProcessMessage(string msg)
        {
            System.Diagnostics.Debug.WriteLine(String.Format("Received from SolidSX ({0}): {1}", this.Handle.ToString(), msg));

            string cmd = msg.Substring(0, msg.IndexOf('('));
            string var = msg.Substring(msg.IndexOf('(')+1, (msg.IndexOf(')') - msg.IndexOf('('))-1);

            switch (cmd)
            {
                case "OnSolidSXStarted": 
                    this.solidSXhandle = (IntPtr)Convert.ToInt32(var);
                    break;
                
                case "OnNodeSelectionChanged":
                    string nodeId = var.Substring(var.IndexOf('[') + 1, (var.IndexOf(']') - var.IndexOf('[')) - 1);
                    if(nodeId.Length > 0)
                    {
                        this.NodeSelectionChanged.Invoke(long.Parse(nodeId), this);
                    }
                    break;
                    
                case "OnEdgeSelectionChanged":
                    string nodeIDs = var.Substring(var.IndexOf('[') + 1, (var.IndexOf(']') - var.IndexOf('[')) - 1);
                    break;
            }
        }

        private void SendMessage(string msg)
        {
            byte[] array = Encoding.Default.GetBytes(msg);
            COPYDATASTRUCT cds = new COPYDATASTRUCT();
            cds.dwData = (IntPtr)100;
            cds.lpData = msg;
            cds.cbData = array.Length;
            SendMessage((IntPtr)this.solidSXhandle, WM_COPYDATA, 0, ref cds);  
        }
    }
}
