/**
 * created on 19.10.2007
 */
package recoder.testsuite.fixedbugs;

import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import recoder.CrossReferenceServiceConfiguration;
import recoder.ModelException;
import recoder.ParserException;
import recoder.ProgramFactory;
import recoder.abstraction.ClassType;
import recoder.abstraction.ErasedConstructor;
import recoder.abstraction.Method;
import recoder.abstraction.Type;
import recoder.bytecode.ASMBytecodeParser;
import recoder.bytecode.AnnotationUseInfo;
import recoder.bytecode.ByteCodeParser;
import recoder.bytecode.ClassFile;
import recoder.bytecode.MethodInfo;
import recoder.bytecode.ReflectionImport;
import recoder.convenience.CustomTreeWalker;
import recoder.convenience.TreeWalker;
import recoder.io.PropertyNames;
import recoder.java.CompilationUnit;
import recoder.java.Expression;
import recoder.java.JavaProgramFactory;
import recoder.java.StatementBlock;
import recoder.java.declaration.ClassDeclaration;
import recoder.java.declaration.ConstructorDeclaration;
import recoder.java.declaration.FieldDeclaration;
import recoder.java.declaration.LocalVariableDeclaration;
import recoder.java.declaration.MethodDeclaration;
import recoder.java.declaration.modifier.Final;
import recoder.java.expression.Assignment;
import recoder.java.expression.operator.New;
import recoder.java.reference.ConstructorReference;
import recoder.java.reference.MethodReference;
import recoder.java.reference.TypeReference;
import recoder.java.statement.LoopStatement;
import recoder.kit.MethodKit;
import recoder.kit.MiscKit;
import recoder.kit.TypeKit;
import recoder.service.NameInfo;
import recoder.service.SemanticsChecker;
import recoder.service.SourceInfo;
import recoder.testsuite.RecoderTestCase;
/**
 * @author Tobias Gutzmann
 *
 */
public class FixedBugs extends RecoderTestCase {
	////////////////////////////////////////////////////////////
	// test cases
	////////////////////////////////////////////////////////////
	
	public void testConstructorStartPosition() throws ParserException {
		ProgramFactory f = sc.getProgramFactory();
		CompilationUnit cu = f.parseCompilationUnit(
				"public class Test\n{\nTest s;\npublic Test(Test s)" + 
				"\n{\nthis.s = s;\n}\n}");
		sc.getChangeHistory().attached(cu);
		sc.getChangeHistory().updateModel();
		cu.validateAll();
		assertTrue(((ConstructorDeclaration)sc.getNameInfo().getClassType("Test").getConstructors().get(0)).getStartPosition().getLine() == 4);
	}
	
	/**
	 * Test for absence of a bug in PrettyPrinter that, after transformation, may 
	 * mess up single line comments.
	 * Bug reported and testcase submitted by M.Ullbrich 
	 * @throws ParserException
	 */
    public void testSingleLineCommentBug() throws ParserException {
    	ProgramFactory f = sc.getProgramFactory();
        CompilationUnit cu = f.parseCompilationUnit("class A {\n\n\n" +
                        "//some comment\r\nA a; } class B {}");
        sc.getChangeHistory().attached(cu);
        sc.getChangeHistory().updateModel();
        cu.validateAll();
        FieldDeclaration fd = (FieldDeclaration) cu.getDeclarations().get(0).getMembers().get(0);
        TypeReference oldType = fd.getTypeReference();
        TypeReference newType = TypeKit.createTypeReference(sc.getProgramFactory(), "B");
        fd.replaceChild(oldType, newType);
        sc.getChangeHistory().replaced(oldType, newType);
        sc.getChangeHistory().updateModel();
        cu.validateAll();
        String s = cu.toSource().replaceAll(" ", "");
        assertTrue(s.equals("classA{\n\n\n//somecomment\nBa;\n}classB{\n}\n"));
    }
    
    /**
     * Test for implemented generic type argument resolving in field references
     * @throws ParserException
     */
    public void testFieldTypes() throws ParserException {
        sc.getProjectSettings().ensureSystemClassesAreInPath();
        ProgramFactory f = sc.getProgramFactory();
        
        CompilationUnit cu = f.parseCompilationUnit("class B { } class G<E> { E field;   void m() { B b; b = new G<B>().field; } }");
        sc.getChangeHistory().attached(cu);
        sc.getChangeHistory().updateModel();
        cu.validateAll();

        ClassDeclaration classB = (ClassDeclaration) cu.getDeclarations().get(0);
        ClassDeclaration classG = (ClassDeclaration) cu.getDeclarations().get(1);
        MethodDeclaration methodM = (MethodDeclaration) classG.getMethods().get(0);
        Assignment assignment = (Assignment) methodM.getBody().getChildAt(1);
        Expression rhs = (Expression) assignment.getChildAt(1);
        Type rhsType = sc.getSourceInfo().getType(rhs);
        
        assertEquals(rhsType, classB);
    }
    
    public void testBasicReflectionImport() {
    	// make sure non-public fields can be read... 
    	assertTrue(ReflectionImport.getClassFile("java.lang.String") != null);
    	assertTrue(ReflectionImport.getClassFile("asdasdasd") == null);
    }
    
    public void testIncorrectFileEnd() {
    	ProgramFactory f = sc.getProgramFactory();
    	String cuText = "class B { }//";
    	for (int i = 0; i < 4081; i++) {
    		cuText += " ";
    	}
    	for (int i = 4081; i < 4087; i++) {
    		// that's around the critical part, where the 
    		// size of the CU matches the JavaCCParser buffer
    		try {
        		f.parseCompilationUnit(cuText);
        		cuText += " ";
    		} catch (ParserException pe) {
    			fail();
    		}
    	}
    }
    
    public void testBug1894463() throws ParserException {
    	ProgramFactory f = sc.getProgramFactory();
    	String cuText = "public class MockClass1 {\n"+
    	"public class InnerClass {"+
    	"public class InnerInnerClass { }" +
    	"}"+
    	"}"+
    	"class MockClass2 {"+
    	"void test(MockClass1 mockClass1) {"+
    	"mockClass1.new InnerClass();" +
    	"MockClass1.InnerClass ic = new InnerClass();" +
    	"ic.new InnerInnerClass();" +
    	"new InnerClass();"+
    	"}"+
    	"public class InnerClass {"+
    	"}"+
    	"}";
        CompilationUnit cu = f.parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
        sc.getChangeHistory().updateModel();
        cu.validateAll();
        ClassDeclaration mock2 = (ClassDeclaration)sc.getNameInfo().getType("MockClass2");
        StatementBlock sb = ((MethodDeclaration)mock2.getMembers().get(0)).getBody();
        assertTrue(
        		((ClassDeclaration)sc.getCrossReferenceSourceInfo().getType(
        				((New)sb.getStatementAt(0)))).getFullName().equals("MockClass1.InnerClass"));
        assertTrue(
        		((ClassDeclaration)sc.getCrossReferenceSourceInfo().getType(
        				((New)sb.getStatementAt(2)))).getFullName().equals("MockClass1.InnerClass.InnerInnerClass"));
        assertTrue(
        		((ClassDeclaration)sc.getCrossReferenceSourceInfo().getType(
        				((New)sb.getStatementAt(3)))).getFullName().equals("MockClass2.InnerClass"));
    }
    
    public void testBug1895973() throws ParserException {
        sc.getProjectSettings().ensureSystemClassesAreInPath();
    	ProgramFactory f = sc.getProgramFactory();
    	String cuText = 
    	"import java.util.ArrayList;\n"+
    	"public class MockClass extends ArrayList<String> {\n"+
    	"public String test() {\n"+
    	"// Recoder doesn't detect that 'get(0)' returns a String !\n"+
    	"return get(0).toLowerCase();\n"+
    	"}\n}\n";
    	CompilationUnit cu = f.parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testBug1911073() throws ParserException{
    	// PrettyPrinter shouldn't be messed up when statement-blocks are empty  
        String cuText =
        	"class A {\n\tint l;\n\tvoid m() {\n\t\tswitch(l) {\n\t\t}\n\t}\n}\n";
        CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
        // check for proper indentation at end of file:
        assertTrue(cu.toSource().endsWith("        }\n    }\n}\n"));
    }
    
    public void testBug1936528() throws ParserException {
    	// find most specific method when applicable methods all have varargs  
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
        	"class A {\n" +
        	"void method(String s, Object... params){ }\n" +
        	"void method(String s, Throwable t, Object... params){ \n"+
        	"method(\"toto\", new Exception());}}\n";
        CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
        sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    
    public void testBug1936842() throws ParserException {
    	// raw types and autoboxing should work together 
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    	"import java.util.*;\n"+
        "class Mock\n"+
    	"{\n"+
    	"void method()\n"+
    	"{\n"+
    	"List l = new ArrayList();\n"+
    	"l.add(0);\n"+

    	"List<Object> l2 = new ArrayList<Object>();\n"+
    	"l2.add(0);\n"+
    	"}\n"+
    	"}\n";

        CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
        sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testCommentAttachment() throws ParserException {
    	String cuText = 
    		"class A {\n" +
    		"\tvoid m(/*c1*/ int p /*c2*/) {\n" +
    		"\t\t//Comment\n" +
    		"\t\tint x = /*c3*/3;\n" +
    		"\t}\n" +
    		"}\n";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    	MethodDeclaration md = (MethodDeclaration)cu.getTypeDeclarationAt(0).getMembers().get(0); 
        assertTrue(
        		md.getBody().getStatementAt(0).getComments().size() == 1);
        assertTrue(md.getParameterDeclarationAt(0).getComments().size() == 2);
        assertTrue(
        		((LocalVariableDeclaration)md.getBody().getStatementAt(0))
        		.getVariableSpecifications().get(0).getInitializer().getComments().size() == 1
        		);
    }
    
    public void testBug1948045() throws ParserException {
    	// parser should accept final inner classes
    	String cuText = 
    		"public class FinalInnerClass {" +
    		"public void method(){" +
    		"final class InnerClass{" +
    		"}}}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    	assertTrue(
    	((MethodDeclaration)cu.getTypeDeclarationAt(0).getMethods().get(0))
    	.getTypes().get(0).getDeclarationSpecifiers().get(0).getClass() == Final.class);
    }
    
    public void testBug1965975() throws ParserException {
    	// Endless loop when querying Methods of a parameterized Type (bytecode)
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	
    	String cuText =
    		"import java.util.List; class A { void m() { List<String> l; } }";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    	sc.getSourceInfo().getMethods((ClassType)
    	sc.getSourceInfo().getType((((LocalVariableDeclaration)
    	((MethodDeclaration)cu.getTypeDeclarationAt(0).getMethods().get(0))
    	.getBody().getStatementAt(0))).getTypeReference()));
    }
    
    public void testBug1974988() throws ParserException {
    	String declText =
    		"@Annot String decl = \"Test\";";
    	FieldDeclaration fd = new JavaProgramFactory().parseFieldDeclaration(declText);
    	fd.deepClone();
    }
    
    public void testBug1988746() throws ParserException {
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    		"class A { void setDouble(Double p) {" + 
    		"Double toto = new Double(1) ;  " +
    		"Double titi = new Double(2) ;	" +
    		"this.setDouble(toto * titi);}}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }

//    covered in java5-testsuite / EnumTest now.
//    public void testBug1988780() throws ParserException {
//    }
    
    public void testBug1996819() throws ParserException {
    	// as of java 5, return types of array types' clone()-methods
    	// have the return type
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    		"class A { void m(String[] s) {" + 
    		"	s.clone(); ((Object)s).clone();}}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
        MethodReference expr = (MethodReference)
				((MethodDeclaration)cu.getTypeDeclarationAt(0).getMethods()
						.get(0)).getBody().getStatementAt(0);
        assertTrue(sc.getSourceInfo().getType(expr).getName().equals("String[]"));
        Method cloneMethod = 
        		sc.getSourceInfo().getMethod(expr);
        Type clonedType = sc.getSourceInfo().getType(cloneMethod);

        assertTrue(clonedType.getFullName().equals("java.lang.String[]"));
        // use the opportunity to check another bug:
        // array types' clone() methods overwrite java.lang.Object.clone() 
        // and do not throw (checked) exceptions ("older" java versions and java 5)
        assertTrue(cloneMethod.getExceptions().size() == 0);
        Method objCloneMethod = sc.getSourceInfo().getMethod((MethodReference)
				((MethodDeclaration)cu.getTypeDeclarationAt(0).getMethods().get(0)).getBody().getStatementAt(1));
        assertTrue(objCloneMethod != cloneMethod);
        assertTrue(MethodKit.getAllRedefinedMethods(cloneMethod).contains(objCloneMethod));
        // TODO 0.90 check the below check...
//        assertTrue(MethodKit.getRedefiningMethods(sc.getCrossReferenceSourceInfo(), objCloneMethod).contains(cloneMethod));
    }
    
    public void testBug2020825() throws ParserException {
    	// inner types don't "inherit" type parameters of outer classes
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	// this should be valid code:
    	String cuText =
    		"import java.util.*; class A<E> { class B extends ArrayList<E> {} B m() { new A<String>().m().get(0).toLowerCase(); return null; }}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testBug2013315() throws ParserException {
    	// generic fields in bytecode
    	// this is the source code of Test.class:
		//import java.util.*;
    	//public class Test
		//{
		//public Map<String, Number> myMap = new HashMap<String, Number>() ;
		//}

    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2013315");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    		"class X {{Test instance = new Test();"+
    		"instance.myMap.get(\"\").shortValue(); }}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testBug2044230() throws ParserException {
    	// this should parse...
    	String cuText =
    		"class A { public static final double MAX_VALUE = 0x1.fffffffffffffP+1023;}";
    	sc.getProgramFactory().parseCompilationUnit(cuText);
    }
    
    public void testBug2045181() throws ParserException {
    	// on demand imports don't find imported types
    	String cuText =
    		"package B; class A { static class InA {}}";
    	String cu2Text =
    		"package B;"+
    		"import B.A.*;"+
    		"class B extends InA {}";
       	runIt(cuText, cu2Text);
    }
    
    public void testBug2045207() throws ParserException {
    	// type parameters not supported for constructor declarations
    	// this should compile:
    	String cuText =
    		"class A {public <T> A(T p1, Class<T> p2) { }}";
       	runIt(cuText);
    }
    
    public void testBug2045354 () throws ParserException {
    	// this should be valid code, but cu.validateAll() fails currently:
    	String cuText =
    		"import java.security.*;" +
    		"class A implements PrivilegedAction<String> {" +
    		"public String run() {" +
    		"	return null;" +
    		"}" +
    		"{ A a = new A(); String s; s = AccessController.doPrivileged(a); }" +
    		"}";
       	runIt(cuText);
    }
    
    public void testBug2046005() throws ParserException {
    	// the code below should be valid
    	// this is actually not bug 2046005, but highly related.
    	// note the <?> in Constructor<?> constr
    	// which makes the code valid. In the initial bugreport at sf.net,
    	// <?> was not present, making the code invalid (but still compilable with javac).
    	// we use this method now to fix the related bug...
    	String cuText =
    		"class A<T> {\n" +
    		"	<T> T foo(Class<T> p) { return null;}" +
    		"}\n"+
    		"class B {\n"+
    		"{\n"+
    		"final Class<String> v = String.class;\n"+
    		"A<?> a = null;\n"+
    		"a.foo(v).toLowerCase();\n"+
    		"}\n"+
    		"}\n";
    	runIt(cuText);
    }
    
    public void testBug2046005_2() throws ParserException {
    	// like above, but made invalid as described
    	// TODO check if this *really* is not allowed.
    	String cuText =
    		"class A<T> {\n" +
    		"	<T> T foo(Class<T> p) { return null;}" +
    		"}\n"+
    		"class B {\n"+
    		"{\n"+
    		"final Class<String> v = String.class;\n"+
    		"A a = null;\n"+
    		"a.foo(v).toLowerCase();\n"+
    		"}\n"+
    		"}\n";
    	runIt(new SilentErrorHandler(1), cuText);
    }
    
    public void testBug2046005_3() throws ParserException {
    	// like above, but with some changes
    	String cuText =
    		"class A<T> {\n" +
    		"	<Q> Q foo(String s, Class<Q> c) { return null; }" +
    		"	<T> T foo(Class<T> p) { return null;}" +
    		"}\n"+
    		"class B {\n"+
    		"{\n"+
    		"final Class<String> v = String.class;\n"+
    		"A x = null;\n" +
    		"A<?> a = null;\n"+
    		"a.foo(v).toLowerCase();\n"+
    		"}\n"+
    		"}\n";
    	runIt(cuText);
    }
    
    public void testBug2046167() throws ParserException {
    	// TODO should be rewritten to not use com.sun...
    	String cuText = 
    		"package com.sun.org.apache.xerces.internal.impl.xs.identity;"+
    		"import com.sun.org.apache.xerces.internal.impl.xpath.XPathException;"+
    		"import com.sun.org.apache.xerces.internal.util.SymbolTable;"+
    		"import com.sun.org.apache.xerces.internal.xni.NamespaceContext;"+
    		"public class Field {"+
    		"public static class XPath"+
    		"    extends com.sun.org.apache.xerces.internal.impl.xpath.XPath {"+
    		"    public XPath() throws XPathException {"+
    		"        super(null, null, null);"+
    		"        com.sun.org.apache.xerces.internal.impl.xpath.XPath.Axis axis = null;"+
    		"        if (axis.type == XPath.Axis.ATTRIBUTE) {"+
    		"               throw new XPathException(null);"+
    		"        }"+
    		"    }"+
    		"}"+
    		"}";
    	runIt(cuText);
    }

    public void testBug2046337() throws ParserException {
    	String cuText = "class A { void foo() {new String[3].getClass();}}";
    	CompilationUnit cus = runIt(cuText).get(0);
    	getClass();
    	ClassType t = (ClassType)cus.getFactory().getServiceConfiguration().getSourceInfo().getType(
    	((MethodDeclaration)cus.getTypeDeclarationAt(0).getMembers().get(0))
    	.getBody().getStatementAt(0));
    	assertTrue(t.getFullSignature().equals("java.lang.Class<? extends java.lang.String[]>"));
    }
    public void testExplicitGenInvokeParserTest() throws ParserException {
    	String cuText = "class A{{java.util.Collections.<String, String[]>emptyMap();}}";
    	runIt(cuText);
    }
    
    public void testCaptureOfIssue() throws ParserException {
    	// unreported, as in development version only. Issue 
    	// when passing "capture-of types" as arguments. 
    	String cuText = "class A {" +
    			"void foo(Number n) { }" +
    			"void bar() {" +
    			"  java.util.Iterator<? extends Number> it = null;" +
    			"  foo(it.next());" +
    			"}" +
    			"}";
    	runIt(cuText);
    }
    
    
    public void testEnumSupertypeBug() throws ParserException {
    	String cuText = "enum A implements java.util.Comparator<Number> {" +
    			"; { Number[] n = null; java.util.Arrays.sort(n, this); }}";
    	runIt(cuText);
    }
    
    public void testCaptureBugParamMatches() throws ParserException {
    	// unfiled bug, DefaultProgramModelInfo.paramMatches() had a problem.
    	String cuText = 
    		"import java.util.Comparator;"+
    		"class A {"+
    		"private static <T> void binarySearch0(T[] a, T key, Comparator<? super T> c) {"+
    		"	    c.compare(key, key);"+
    		"}"+
    		"}";
    	runIt(cuText);
    }
    
    public void testClassLiteralTypes() throws ParserException {
    	// correct types for class literals? related to bug 2046337
    	String cuText =
    		"class A {" +
    		"void m() {" +
    		"void.class.cast(null);" +
    		"int.class.cast(null);" +
    		"Integer.class.cast(null);}}";
    	CompilationUnit cu = runIt(cuText).get(0);
    	StatementBlock sb = ((MethodDeclaration)cu.getTypeDeclarationAt(0).getMembers().get(0)).getBody();
    	SourceInfo si = cu.getFactory().getServiceConfiguration().getSourceInfo();
    	assertTrue(((ClassType)si.getType(sb.getStatementAt(0))).getFullSignature().equals("java.lang.Void"));
    	assertTrue(((ClassType)si.getType(sb.getStatementAt(1))).getFullSignature().equals("java.lang.Integer"));
    	assertTrue(((ClassType)si.getType(sb.getStatementAt(2))).getFullSignature().equals("java.lang.Integer"));
    }
    
    public void testValidateBug() throws ParserException {
    	// unfiled validate()-bug. Code below is valid but wasn't in
    	// intermediate development version
    	String cuText =
    		"class A { {byte b; b = 5; " +
    		"Class<?> classes[]; classes = new Class[20];" +
    		"}}";
    	runIt(cuText);
    }
    
    public void testTypeInferenceBug() throws ParserException {
    	// unfiled. Type inference issue.
    	String cuText =
    		"import java.util.Collection;"+
    		"class A { " +
    		"static class B<E> {"+
    		"E bar[];"+
    		"}" +
    		"static class C<E> extends B<E> {" +
    		"public void foo(Collection<? extends E> c) {"+
    		"      E[] a = null;"+
    		"      a = c.toArray(bar);" +
    		"}"+
    		"	}"+
    		"}";
    	runIt(cuText);
    }
    
    public void testAnonymousEnumInstanceParent() throws ParserException {
    	// unfiled. Recoder didn't properly return 
    	// the supertypes of an anonymous enum class body.
    	String cuText = "public enum A {"+
    		  "CONST { public int foo() { return 0; }};"+
    		  "public int foo() { return -1; }"+
    		  "}"+

    		  "class B {"+
    	      "void foo(A a) { foo(A.CONST); }"+
    		  "}";
    	runIt(cuText);
    }
    
    public void testValidateBug2() throws ParserException {
    	// another unfiled validate() bug
    	String cuText = "public class A<E> extends java.util.AbstractQueue<E> {"+
    	"private final Object lock;"+
    	"private class B {"+
    	"public void foo() { final Object i = A.this.lock; }"+
    	"}}";
    	runIt(cuText);
    }
    
    public void testBug2044375() throws ParserException {
    	// should report an error, not throw an exception. 
    	String cuText =
    		"import java.util.HashMap;" +
    		"class A {" +
    		"  HashMap<String, Unknown<String, String>> field;" +
    		"  public Unknown<String, String> getChangeString(String s) {" +
    		"    return field.get(s);" +
    		"  }" +
    		"}";
    	try {
    		runIt(new SilentErrorHandler(2), cuText);
    	} catch (ModelException me) {
    		// okej, we expect this!
    	}
    }
    
	public void testEnumValueOfReturnType() throws ParserException {
    	// unfiled
    	String cuText = "public class A { void foo() { " +
    			"Class c = null; Enum.valueOf(c, \"WAITING\"); }}";
    	CompilationUnit cu = runIt(cuText).get(0);
    	SourceInfo si = cu.getFactory().getServiceConfiguration().getSourceInfo();
    	Type t = si.getType(((MethodDeclaration)cu.getTypeDeclarationAt(0).getMembers().get(0))
    			.getBody().getStatementAt(1));
    	assertEquals(t.getFullName(), "java.lang.Enum"); // leftmost bound
    }
    
    public void testArraySupertypes() {
       	sc.getProjectSettings().ensureSystemClassesAreInPath();
       	ClassType ct = sc.getNameInfo().getClassType("java.lang.String[]");
       	Set<String> supStr = new HashSet<String>();
       	for (ClassType sup : ct.getSupertypes())
       		supStr.add(sup.getFullSignature());
       	assertTrue(supStr.size() == 4);
       	assertTrue(supStr.contains("java.lang.Object[]"));
       	assertTrue(supStr.contains("java.io.Serializable[]"));
       	assertTrue(supStr.contains("java.lang.Comparable<java.lang.String>[]"));
       	assertTrue(supStr.contains("java.lang.CharSequence[]"));
    }

    
    public void testRHSUnboxing() throws ParserException {
    	// unreported bug: instead of boxing LHS, unbox RHS of an assignment.
    	String cuText = 
    		"class A { {" +
    		"float num = 0;" +
    		"int i = 0;" +
    		"int[] args = null;"+
    	"	i++;"+
    	"	num = Integer.valueOf(args[i]);"+
    	"}"+
    	"}";
    	CompilationUnit cu = runIt(cuText).get(0);
    	cu.validateAll();
    }
    
    public void testBug2071287() throws ParserException {
    	// scoping after transformation
    	String cuText =
    		"class A {\n" +
    		"void m() {\n" +
    		"  {\n" +
    		"    for (;;) {\n" +
    		"      int n;\n" +
    		"    }\n" +
    		"  }\n" +
    		"  int n;\n" +
    		"}\n" +
    		"}\n";
    	List<CompilationUnit> cul = runIt(cuText);
    	for (CompilationUnit cu : cul)
    		cu.validateAll();
    	
    	TreeWalker tw = new TreeWalker(cul.get(0));
    	while (tw.next(LoopStatement.class)) {
    		LoopStatement ls = ((LoopStatement)tw.getProgramElement());
    		LoopStatement repl = ls.deepClone();
    		ls.getASTParent().replaceChild(ls, repl);
    		sc.getChangeHistory().replaced(ls, repl);
    	}
    	sc.getChangeHistory().updateModel();
    }
    
    public void testBytecodeInnerClass() {
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	ClassType itr = sc.getNameInfo().getClassType("java.util.AbstractList.Itr");
    	assertTrue(itr.isInner());
    }
    
    public void testBug2088980() throws ParserException {
    	// actually, a duplicate of bug 2000780. But we don't have an explicit
    	// testcase for that one.
    	String cuText = 
    			"import java.util.*;" +
    			"class A {" +
    			"	public <T> T methodCallingGenericMethod(T bean) {"+
    			"		Collection<T> aList = new ArrayList<T>();"+
    			"		aList.add(Collections.singletonList(bean).get(0));"+
    			"	}" +
    			"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu : cus)
    		cu.validateAll();
    }
    
    public void testBug2132665() throws ParserException {
    	// anonymous class with following call to a newly declared method
    	String cuText =
    		"import java.awt.Dialog;"+
    		"import java.awt.Frame;"+

    		"public class Main {"+
    		"public static void main(String[] args) {"+
    		"Frame frame = new Frame();"+

    		"new Dialog(frame){"+
    		"public void init(){"+
    		"//initme\n"+
    		"}"+
    		"}.init();"+
    		"}"+
    		"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu : cus)
    		cu.validateAll();
    	// the following is some experimental testing. Reported via email
    	// to me, but I don't think it's a bug (getName() == null should
    	// be desired behavior!?)
    	CustomTreeWalker ctw = new CustomTreeWalker(cus.get(0));
    	ctw.next(MethodDeclaration.class);
    	ctw.next(MethodDeclaration.class);
    	MethodDeclaration method = (MethodDeclaration)ctw.getProgramElement();
        assertNotNull(method.getContainingClassType().getFullName());
        // TODO think about this
//        System.out.println(method.getContainingClassType().getName()); 
    }
    
    public void testBug2134267() throws ParserException {
    	// problem resolving package names from bytecode classes when naming conventions are not follow (e.g., COM.xyz)
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2134267/*.jar");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText = 
    		"package runner;"+
    		"public class Main {"+
    		"public static void main(String[] args) {"+
    		"COM.AClass value = null;"+
    		"}"+
    		"}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
   		cu.validateAll();
    }
    
    public void testBug2155226() throws ParserException {
    	// this should report a ModelException, not throw an UnsupportedOperationException
    	String cuText =
       		"class KnownBasicClass"+
       		"{}"+

       		"class TestFailed1 extends UnknownGenericClass<KnownBasicClass>{"+
       		"}";
        runIt(new SilentErrorHandler(1), cuText);
    }
    
    public void testBug2230018() throws ParserException {
    	// should again report a ModelException, not throw an Exception...
    	String cuText =
    		"import com.pkg.UnknownClass;"+
    		"class KnownBasicClass"+
    		"{"+
    		"    int nb = UnknownClass.CONSTANT;"+
    		"}"	;
    	runIt(new SilentErrorHandler(4), cuText); 
    }

    public void testBug2230018_2() throws ParserException {
    	// should again report a ModelException, not throw an Exception...
    	// DO NOT MERGE WITH testBug2230018(), OTHERWISE THIS BUG WON'T BE DETECTED DUE TO CACHING ISSUES! 
    	String cuText = 
    		"package com.example.generics;"+
    		"import com.Blubb;"+
    		"public class ExtendsUnknownGeneric extends Toto<DynamicType>"+
    		"{"+ 
    		"  public void method()"+
    		"  {"+
    		"    String s = Blubb.CONSTANTE ;"+
    		"  }"+
    		"}";
    	runIt(new SilentErrorHandler(6), cuText); 
    }

    public void testOverwritingWithPrimitives() throws ParserException {
    	// Never been a bug, and never should be, so we add this here: 
    	// don't use autoboxing for determining overwriting methods ;-)
    	String cuText =
    		"abstract class A<T> { abstract void foo(T a); }\n"+
    		"class B extends A<Integer> {\n"+ 
    		"	void foo(int a) {}\n"+
    		"}\n"+
    		"class C extends B {" +
    		"	void foo(long a) {" +
    		"	}\n" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    	assertEquals(MethodKit.getRedefinedMethods(
    			cus.get(0).getTypeDeclarationAt(1).getMethods().get(0)).size(), 0);
    	assertEquals(sc.getSourceInfo().getAllMethods(cus.get(0).getTypeDeclarationAt(1)).size(), 13);
    	assertEquals(MethodKit.getRedefinedMethods(
    			cus.get(0).getTypeDeclarationAt(2).getMethods().get(0)).size(), 0);
    	assertEquals(sc.getSourceInfo().getAllMethods(cus.get(0).getTypeDeclarationAt(2)).size(), 14);
    }

    public void testCrossReferencer() throws ParserException {
    	// Never been a bug, and never should be. I just suspected it once, 
    	// and now that it's written, it stays for the sake of testing :-P
    	String cuText =
    		"class A {\n"+
    		"	void foo(long a) {" +
    		"		java.util.LinkedList ll = new java.util.LinkedList();\n" +
    		"	}" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    	assertEquals(sc.getCrossReferenceSourceInfo().getReferences(
    			sc.getNameInfo().getType("java.util.LinkedList")).size(), 2);
    }

    
    public void testBug2343547() throws Exception {
    	new ByteCodeParser().parseClassFile(new FileInputStream("test/fixedBugs/Bug2343547/ICacheManager.class"));
    }
    
    public void testBug2510517() throws Exception {
    	// not necessary any longer as of 0.94. 
    }
    
    public void testBug2523272() throws Exception {
    	
    	// the .class file is corrupted so that it references an unknown file!
    	// further, recoder is not in classpath, so it would even be sufficient to not modify it...
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2523272");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	sc.getProjectSettings().setErrorHandler(new SilentErrorHandler(0));
    	String cuText =
    		"class X { void foo(recoder.convenience.ASTIteratorListener o) {} }";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
        
        ClassType ct = sc.getNameInfo().getClassType("recoder.convenience.ASTIteratorListener");
        List<Type> sig = ct.getMethods().get(0).getSignature();
        assertNotNull(sig.get(0));
        assertNotNull(sig.get(1));
        assertEquals(sig.get(0), sc.getNameInfo().getUnknownClassType());
        assertEquals(sig.get(1), sc.getNameInfo().getUnknownClassType());
    }
    
    public void testBug2609084() throws Exception {
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2609084");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	
    	String cuText =
    		"class Y { String s; void foo(X<String> x) {s = x.o.toUpperCase(); } }";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    	assertNotSame(sc.getNameInfo().getClassType("X").getFields().get(0).getType(), sc.getNameInfo().getUnknownClassType());
    }
    
    public void testBug2808379() throws Exception {
    	String cuText =
    		"class A {\n"+
    		"	static Object foo() { return null; }\n" +
    		"}\n" +
    		"class B extends A {\n"+
    		"	static String foo() { return null; }\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	assertEquals(1,
    			MethodKit.getRedefiningMethods(sc.getCrossReferenceSourceInfo(), cus.get(0).getTypeDeclarationAt(0).getMethods().get(0)).size()
    			);
    	assertEquals(1,
    			MethodKit.getRedefinedMethods(cus.get(0).getTypeDeclarationAt(1).getMethods().get(0)).size()
    			);
    }
    
    public void testBug2813956() throws Exception {
    	String cuText =
    		"class Y<T> {\n" +
    		"	public void foo(Class<T> c) { }\n" +
    		"}\n" +
    		"class X extends Y {\n" +
    		"	public void foo(Class c) { } \n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    	assertEquals(1, MethodKit.getAllRedefinedMethods(
    			cus.get(0).getTypeDeclarationAt(1).getMethods().get(0)).size());
    }
    
    public void testBug2824786() throws Exception {
    	String cuText = 
    		"import static a.X2.I;" + 
    		"class X {" + 
    		"	I e = null;" +
    		"}";
    	String cuText2 = 
    		"package a;"+
    		"public class X2 {"+
    		"	public interface I { void foo(); }"+
    		"}";
    	List<CompilationUnit> cus = runIt(cuText, cuText2);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testBug2828305() throws Exception {
    	String cuText = 
    		"class C {"+
    		"@Deprecated <A extends Number> A foo(A a) { return null; }"+
    		"}";

    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testBugImportButNoTypeDeclaration() throws Exception {
    	// error in DefaultSourceInfo: No type declaration, but imports,
    	// causing an IndexOutOfBoundsException.
    	// further, x can't be resolved. Bug has been not reported 
    	// resolved
    	String cuText =
    		"package a;\n public class A { public @interface AA { String s(); }; static final String x = \"\";}";
    	String cuText2 = 
    		"@A.AA(s = x) package a;\n import static a.A.x;";
    	
    	List<CompilationUnit> cus = runIt(cuText, cuText2);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testBug2828368() throws Exception {
    	String cuText =
    		"class X {"+
    		"interface I { }"+
    		"class Y extends java.util.ArrayList<Number> implements I { }"+
    		"void g(I o) { }"+
    		"void g(java.util.Collection<String> p) { }"+
    		"void h() {"+
    		"	g(new Y());"+
    		"}}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void test2828368_2() throws Exception {
    	// something that dug up during fixing of bug 2828368. Not reported, but related.
    	// the reference to the constructor of Y cannot be resolved in an
    	// intermediate Recoder version.
    	String cuText =
    		"public class X { " +
    		"	void foo() { new Y(new java.util.ArrayList()); }" +
    		"}";
    	String cuText2 =
    		"public class Y extends X {" +
    		"	Y(java.util.List<X> l) { }" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText, cuText2);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void test2828368_3() throws Exception {
    	// the return type of java.util.Collections.enumeration(...) should be 
    	// %raw%Enumeration!
       	String cuText =
    		"public class S { " +
    		"	S(java.util.Enumeration<? extends Number> p) { }" +
    		"	void foo() {" +
    		"		new S(java.util.Collections.enumeration(new java.util.ArrayList()));" +
    		"	}" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testHidingTypeParameterDeclarationBug() throws Exception {
    	// there is a bug when it comes to type parameter declarations
    	// in methods hiding a surrounding one
       	String cuText =
       		"import java.util.*;\n" +
       		"class X {\n"+
       		"{\n"+
       		"		X comp = null;\n"+
       		"		List<String> nameList = null;\n"+
       		"		comp.compile(Y.from(nameList.toArray(new String[0])));\n"+ 
       		"	}\n"+
       		"	void compile(List<String> f) { }\n"+
       		"}\n"+
       		"class Y<A> {\n"+
       		"	static <A> List<A> from(A[] array) { return null; }\n"+
       		"}\n";

       	List<CompilationUnit> cus = runIt(cuText);
       	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testFindInheritedTypeAndFieldThroughPackageImport() throws Exception {
    	String cuText = 
    		"import static x.A.*;" +
    		"class X {" +
    		"	Member m = null;" +
    		"	int x = intField;" +
    		"}";
    	String cuText2 = 
    		"package x;" +
    		"public class A extends Base {}";
    	String cuText3 = 
    		"package x;" +
    		"public class Base {" +
    		"	public static class Member { }" +
    		"	public static final int intField = 123;" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText, cuText2, cuText3);
       	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testInheritedTypeByOuterClass() throws Exception {
    	// the actual cause for this bug was that InterfaceDeclaration
    	// contained a bug: it reported that an interface is "never protected".
    	// That's just not the case, the code below is valid!
    	String cuText = 
    		"package a; public class A { protected interface I {}; }";
    	String cuText2 = 
    		"public class B extends a.A {" +
    		"	I i = null;" +
    		"	class Inner {" +
    		"		I i = null;" +
    		"	}" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText, cuText2);
       	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }

    public void testResolveTypeParameterReference() throws Exception {
    	String cuText = 
    		"abstract class X<T extends Number> {" +
    		"private static abstract class Member<T extends Number> extends X<T> {"+
    		"	protected F<T> filter() {"+
    		"		return new F<T>() {"+
    		"			protected boolean matches(T d) {"+
    		"				return match(d);"+
    		"			}"+
    		"		};"+
    		"	}"+
    		"	protected abstract boolean match(T d);" +
    		"}"+
    		"}"+ 
    		// if the name of type parameter was something other than T,
    		// then there wasn't a bug in the first place
    		"abstract class F<T> {"+
    		"	protected abstract boolean matches(T t);"+
    		"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
    
    public void testGetRawMemberType() throws Exception {
    	// strange bug that occurred in an intermediate version of 
    	// recoder only.
    	String cuText = 
    		"class X {" +
    		"	private class Inner extends a.Other {" +
    		"		private class X extends Member { }" +
    		"	}" +
    		"}";
    	String cuText2 = 
    		"package a;" +
    		"public abstract class Other<M> {" +
    		"	protected abstract static class Member { }" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText, cuText2);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
    }
   
    public void testPrintEmptyMethodComments() throws Exception {
    	String cuText = 
    		"class X {\n" +
    		"	X() { /* empty */ }\n" +
    		"	X(int i) {\n // empty \n}\n" +
    		"	foo() {\n" +
    		"		// empty as well\n" +
    		"	}\n" +
    		"	bar() {\n" +
    		"		/* empty 4*/\n" +
    		"	}\n" +
    		"}\n" +
    		"interface A { /* interfaceComment*/ }";
    	List<CompilationUnit> cus = runIt(cuText);
    	for (CompilationUnit cu: cus)
    		cu.validate();
    	new SemanticsChecker(sc).checkAllCompilationUnits();
        String s = cus.get(0).toSource().replaceAll("( |\n)", "");
        assertEquals("classX{X(){/*empty*/}X(inti){//empty}foo(){//emptyaswell}" +
        		"bar(){/*empty4*/}}interfaceA{/*interfaceComment*/}", s);
    }
    
    public void testImportBug() throws Exception {
    	// for some strange reasons, this below shall fail: X cannot be resolved.
    	// obviously, in the case of imports, inherited members are *not* considered.
    	// accessing class X is both valid in the context of class A, or when changing
    	// the import statement to import p.A.*; or to import static p.A2.*;
    	// That's at least the behavior of Eclipse and javac.
    	// TODO check in JLS why this is.
    	String cuText1 = 
    		"package p;\n" +
    		"class A {\n" +
    		"	public static class X {\n" +
    		"		public static final int q = 3;\n" +
    		"	}\n" +
    		"	public class Y { public static final int q = 5;\n }" +
    		"}\n";
    	String cuText2 = "package p;\n" +
    			"class A2 extends A { }\n"; 
    	String cuText3 = // faulty!
    		"package q;\n" +
    		"import p.A2.*;" +
    		"class I {\n" +
    		"	void foo() {\n" +
    		"		int i = X.q;"+
    		"	}\n" +
    		"}\n";
    	String cuText4 = // ok !
    		"package q;\n" +
    		"import static p.A2.*;" +
    		"class J {\n" +
    		"	void foo() {\n" +
    		"		int i = X.q;"+
    		"	}\n" +
    		"}\n";
    	String cuText5 = // faulty again!
    		"package q;\n" +
    		"import static p.A2.*;" +
    		"class K {\n" +
    		"	void foo() {\n" +
    		"		int i = Y.q;"+
    		"	}\n" +
    		"}\n";
   		runIt(new SilentErrorHandler(6), cuText1, cuText2, cuText3, cuText4, cuText5);
    }
    
    public void testAmbiguousMethodBecauseOfErasure() throws Exception {
    	// in 0.93, Recoder accepted the code below. However, class C is invalid:
    	// both foo(Object) and foo(T) have the same erasure! Recoder should
    	// discover this when resolving the method invocation.
    	String cuText1 = 
    		"abstract class A<T> {"+
    		"void foo(T t) { C<T> c = new C<T>(); c.foo(t); } "+
    		"}"+
    		"class B extends A<String> {"+
    		"void foo(String s) { }"+
    		"}" +
    		"class C<T> extends A<T> {" +
    		"void foo(Object s) { }" +
    		"void foo(T t) { }" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText1);
    	// TODO make this fail!
    }
    
    public void testBug2838441() throws Exception {
    	// the following is incorrect code(!)
    	String cuText1 = 
    		"import java.util.*;\n" +
    		"class X extends HashMap {\n" +
    		"	void foo(Entry e) { }\n" +  // error!
    		"	void bar(Map.Entry e) { }\n" + // valid!
    		"}\n";
    	runIt(new SilentErrorHandler(1), cuText1);
    }
    
    public void testGenericConstructors() throws Exception {
    	String cuText1 = 
    		"class NN {\n"+
    		"<T> NN (T t1, T t2) { }\n"+
    		"NN() {\n"+
    		"	<String>this(null, null);\n"+
    		"}\n"+
    		"}\n"+
    		"class MM extends NN {\n"+
    		"MM() {\n"+
    		"	<String>super(null, null);\n"+
    		"	new <String>NN(null, null);\n"+
    		"}\n"+
    		"}\n";
    	CompilationUnit cu = runIt(cuText1).get(0);
    	String res = cu.toSource();
    	assertEquals(cuText1.replaceAll("( |\n|\t)", ""), res.replaceAll("( |\n|\t)", ""));
    	// TODO not only check if it parses, but also if the result remains the same!
    }
    
    public void testBug2838979() throws Exception {
    	// the following caused a stack overflow in 0.93 (and probably before as well):
    	String cuText1 = 
    		"class B<T> { }\n"+
    		"class A {\n" +
    		"  public void foobar( B<B<B<A>>> a, B<B<A>> b) {}\n" +
    		"}\n";
    	runIt(cuText1);
    }
    
    public void testLocalTypesReadFromBytecode() throws Exception {
    	// TODO fix...
    	sc = new CrossReferenceServiceConfiguration();
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	ClassType coll = sc.getNameInfo().getClassType("java.util.Collections");
    	for (Method m : coll.getMethods()) {
    		if (m.getName().equals("enumeration")) {
    			MethodInfo mi = (MethodInfo)m;
    			ClassType inner = mi.getTypes().get(0);
    		}
    	}
    }

    public void testBug2843524() throws Exception {
    	// most likely a duplicate of 2838979. Anyhow, an additional test case
    	// can't hurt.
        String cuText =
            "public abstract class SomeClass {\n"+
            "    public abstract <T extends Comparable<T>> boolean check(T x);\n"+
            "    public boolean match(String s) {\n"+
            "        return check(s);\n"+
            "    }\n"+
            "}\n";
        runIt(cuText);
    }

    public void testGenericsIssue() throws Exception {
    	// didn't run in an intermediate Recoder version. 
    	String cuText1 = 
    		"package p;" +
    		"import java.util.*;" +
    		"class D {\n" +
    		"	void foo() { " +
    		"		String x[] = new String[1];" +
    		"		Arrays.asList(x).get(0).toLowerCase(); " +
    		"	}" +
    		"}\n";
    	runIt(cuText1);
    }

    public void testOverwriteWithMethodTypeParams() throws Exception {
    	// overwrite / implements bug (matching signatures in conjunction with method type parameters)
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	SourceInfo si = sc.getSourceInfo();
    	NameInfo ni = sc.getNameInfo();
    	ClassType ct = ni.getClassType("java.util.AbstractList");
    	List<Method> ms = si.getMethods(ct, "remove", Collections.<Type>singletonList(ni.getIntType()));
    	ClassType ct2 = ni.getClassType("java.util.AbstractCollection");
    	List<Method> ms2 = si.getMethods(ct2, "remove", ms.get(0).getSignature(), false, false);
    	assertEquals(0, ms2.size());
    }
    
    public void testBug2849358() throws Exception {
    	String cuText =
    		"class A {"+
    		"A(java.awt.Window s) { }"+
    		"A(java.awt.Frame n) { }"+
    		"void foo() {"+
    		"  new A(null);"+
			"}"+
    		"}";
    	CompilationUnit cu = runIt(cuText).get(0);
    	ConstructorReference cr = (ConstructorReference)((MethodDeclaration)cu.getTypeDeclarationAt(0).getMembers().get(2)).getBody().getBody().get(0);
    	assertEquals(cu.getTypeDeclarationAt(0).getMembers().get(1),
    			sc.getSourceInfo().getConstructor(cr)); 
    }
    
        
    public void testBug2860511() throws Exception {
    	String cuText1 = 
    		"class A {\n" +
    		"	void foo() {\n " +
    		"		String s = new String[] {\"aa\"} [0];\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	assertEquals(
    			"classA{voidfoo(){Strings=newString[]{\"aa\"}[0];}}",
    			cus.get(0).toSource().replaceAll("( |\n)", "")
    	);
    }
    
    public void testInnerClass() throws Exception {
    	// not a bug, but something that tests a strange implementation. See ClassFile.isInner() for explanation.
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	assertTrue(sc.getNameInfo().getClassType("java.util.HashMap.Values").isInner());
    	assertFalse(sc.getNameInfo().getClassType("java.util.HashMap.Entry").isInner());
    }
    
    public void testBug2910715() throws Exception {
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2910715/gin-1.0-20090622.jar");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    		"import com.google.gwt.inject.client.binder.GinLinkedBindingBuilder;\n"+
    		"public class TestCase { }\n";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testBug2912087() throws Exception {
    	String cuText1 = 
    		"@interface A {\n" +
    		"	public int A_CONST = 0; " + 
    		"}\n";
    	runIt(cuText1);
    }

    public void testBug2927536_WorksWithSourceCodeOnly() throws Exception {
    	// The source code for the following two test cases; no problem with source code only.
    	String cuText = "public class A<X, Y>"+
    	"{"+
    	"	protected void parentMethod1(A<?, ?> p) { }"+
    	"	protected void parentMethod2(Y input) { }"+
    	"}"+
    	"class A2<X> extends A<java.util.List<X>, X>"+
    	"{"+
    	"	public void childMethod() {}"+
    	"}";
    	String cuText2 =
    		"public class Test"+
    		"{"+
    		  "public void test(A2<String> child)"+
    		  "{"+
    		    "child.childMethod();"+
    		    "child.parentMethod1(child);"+
    		  "}"+
    		"}";
    	runIt(cuText, cuText2);
    }
    
    public void testBug2927536() throws Exception {
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2927536");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    		"public class Test"+
    		"{"+
    		  "public void test(A2<String> child)"+
    		  "{"+
    		    "child.childMethod();"+
    		  "}"+
    		"}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testBug2927536_2() throws Exception {
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug2927536_2");
    	sc.getProjectSettings().ensureSystemClassesAreInPath();
    	String cuText =
    		"public class Test"+
    		"{"+
    		  "public void test(A2<String> child)"+
    		  "{"+
    		    "child.parentMethod1(child);"+
    		  "}"+
    		"}";
    	CompilationUnit cu = sc.getProgramFactory().parseCompilationUnit(cuText);
    	sc.getChangeHistory().attached(cu);
    	sc.getChangeHistory().updateModel();
        cu.validateAll();
    }
    
    public void testBug2980457() throws Exception {
    	String cuText =
    		"import p.q.C;\n"+
    		"import static p.q.C.CONST;\n"+
    		"public class Test {\n"+
    		"	private C f = CONST;\n"+
    		"}\n";
    	runIt(new SilentErrorHandler(5), cuText);
    }
    
    
    public void testBug2983137() throws Exception {
    	// the following is valid code. Recoder 0.94c throws a NullPointerException.
    	String cuText = 
    		"class A {\n"+
    			"int f;\n"+
    		"}\n"+
    		"class B extends A implements java.io.Serializable {}\n"+
    		"class C extends A implements java.io.Serializable {}\n"+
    		"public class IT2 {\n"+
    		"	public static void main(String[] args) {\n"+
    		"		new IT2().foo(new B(), new C()).f = 3;\n"+
    		"	}\n"+
    		"	<T> T foo(T t1, T t2) {"+
    		"		return null;"+
    		"	}"+
    		"}";
    	runIt(cuText);
    }
    public void testXXX() throws Exception {
    	// TODO This is invalid source code. Write a bug report and fix it.
    	String cuText = 
    		"package p1;\n" +
    		"class A {\n" +
    		"  protected int f;\n" +
    		"}\n";
    	String cuText2 = 
    		"package p2;\n" +
    		"class B extends p1.A{\n" +
    		"  void foo() { new p1.A().f = 3; }\n" +
    		"}\n";
    	try {
    		runIt(cuText, cuText2);
    		fail("This code should not be accepted!");
    	} finally { }
    }
    
    public void testGetClass1() throws Exception {
    	// This is invalid source code. 
    	// The type of type argument argument of Class<T> is the *erasure* of the prefix, not the prefix
    	String cuText = "class A {"+
    	"	int f;"+
    	"}"+
    	"class B extends A implements I {"+
    	"	public void bar() { }"+
    	"}"+
    	"class C extends A implements I {"+
    	"	public void bar() { }"+
    	"}"+
    	"interface I {"+
    	"	void bar();"+
    	"}"+
    	"public class IT2 extends A {"+
    	"	public static void main(String[] args) {"+
    	"		new IT2().foo(new B(), new C()).getClass().newInstance().bar();"+
    	"	}"+
    	"	<T> T foo(T t1, T t2) {"+
    	"		return null;"+
    	"	}"+
    	"}";
    	try {
    		runIt(cuText);
    		fail("This code should not be accepted!");
    	} catch (ModelException e) {
    		// yes :-)
    	}
    }

    public void testGetClass2() throws Exception {
    	// On the other hand, if I is the most common supertype (more common than Object), this should pass.
    	String cuText = "class A {"+
    	"	int f;"+
    	"}"+
    	"class B implements I {"+
    	"	public void bar() { }"+
    	"}"+
    	"class C implements I {"+
    	"	public void bar() { }"+
    	"}"+
    	"interface I {"+
    	"	void bar();"+
    	"}"+
    	"public class IT2 extends A {"+
    	"	public static void main(String[] args) {"+
    	"		new IT2().foo(new B(), new C()).getClass().newInstance().bar();"+
    	"	}"+
    	"	<T> T foo(T t1, T t2) {"+
    	"		return null;"+
    	"	}"+
    	"}";
   		runIt(cuText);
    }
    
    public void testGetClass3() throws Exception {
    	// This one will print a warning on syserr.
    	// TODO figure out how to really do it!
    	String cuText = "class A {"+
    	"	int f;"+
    	"}"+
    	"class B implements I,I2 {"+
    	"	public void bar() { }"+
    	"}"+
    	"class C implements I,I2 {"+
    	"	public void bar() { }"+
    	"}"+
    	"interface I2 { }\n " +
    	"interface I {"+
    	"	void bar();"+
    	"}"+
    	"public class IT2 extends A {"+
    	"	public static void main(String[] args) {"+
    	"		new IT2().foo(new B(), new C()).getClass().newInstance().bar();"+
    	"	}"+
    	"	<T> T foo(T t1, T t2) {"+
    	"		return null;"+
    	"	}"+
    	"}";
   		runIt(cuText);
    }

    public void testBug3000357() throws Exception {
    	String cuText =
    		"public class Test\n"+
    		"{\n"+
    		  "public void test()\n"+
    		  "{\n"+
    		    "java.util.List l = null;\n" +
    		    "String[] s = null;" +
    		    "l.add(l.toArray(s));\n"+
    		  "}\n"+
    		"}\n";
    	runIt(cuText);
    }
    
    public void testBug3000357_2() throws Exception {
    	String cuText =
    		"import java.util.List;" +
    		"public class Test<E>\n"+
    		"{\n" +
    		"  Test(E e) { }\n" +
    		"  Test(E[] e) { int i = 0;/*to indicate correct resolved */ }\n"+
    		  "public void test()\n"+
    		  "{\n"+
    		    "new Test(new Object[1]);"+
    		  "}\n"+
    		"}\n";
    	CompilationUnit cu = runIt(cuText).get(0);
    	assertEquals(
    	((ConstructorDeclaration)((ErasedConstructor)
    	sc.getSourceInfo().getConstructor((ConstructorReference)
    	((MethodDeclaration)cu.getTypeDeclarationAt(0).getMembers().get(2))
    		.getBody().getBody().get(0))).getGenericMethod()).getBody().getBody().size(), 1);
    }
    
    public void testBug3073325() throws Exception {
    	String cuText = "public class X { }";
    	CompilationUnit cu = runIt(cuText).get(0);
    	Method m = cu.getTypeDeclarationAt(0).getAllMethods().get(0);
    	assertEquals("getClass", m.getName());
    	m.getProgramModelInfo().getReturnType(m);
    }
    
    public void testBug3139168() throws Exception {
    	String cuText = "class A {\n" +
    			"\tvoid foo(java.util.List<String> l) {\n" +
    			"\t\tfinal @SuppressWarnings(\"unused\") Object o = null;\n" +
    			"\t\tfor (@SuppressWarnings(\"unused\") String s : l) { }\n" +
    			"\t}" +
    			"}";
    	CompilationUnit cu = runIt(cuText).get(0);
    	assertEquals(cuText.replaceAll("( |\n|\t)", ""), cu.toSource().replaceAll("( |\n|\t)", ""));
    }
    
    public void testLineFeedAfterMLComment() throws Exception {
    	String cuText = "class A {\n" +
		"    /*Test\n" +
		"    Test\n" +
		"    Test*/\n" +
		"    int f;\n" +
		"}\n";
    	CompilationUnit cu = runIt(cuText).get(0);
    	MiscKit.unindent(cu);
    	assertEquals(cuText, cu.toSource());
    }
    
    public void testConstructorExceptions() throws Exception {
    	String cuText = "class A {\n" +
    			"    A() throws Exception {}\n" +
    			"}\n";
    	CompilationUnit cu = runIt(cuText).get(0);
    	assertEquals(cuText, cu.toSource());
    }
    
    public void testBug3195113() throws Exception {
    	// TODO don't fail if jdk is not available...
    	String cuText = "package recoder.CipherTest;\n" +
    			"import javax.crypto.Cipher;\n" +
    			"public class CryptFactorySymetric {\n" +
    			"\tprivate Cipher cipher = null;\n" +
    			"}";
    	//sc.getProjectSettings().ensureExtensionClassesAreInPath();
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, 
    			"c:/program files/java/jdk1.5.0_18/jre/lib/*.jar");
    	runIt(cuText);
    }
    
    public void testBug3202867() throws Exception {
    	// source code of class file:
    	//package p;
    	//import javax.persistence.Table;
    	//@Table(name="T_P_UTILISATEUR", uniqueConstraints={@javax.persistence.UniqueConstraint(columnNames={"UTI_CODE"})})
    	//public class A { }
    	ClassFile cf = new ASMBytecodeParser().parseClassFile(new FileInputStream("test/fixedBugs/Bug3202867/A.class"));
    	List<AnnotationUseInfo> annots = cf.getAnnotations();
    	assertEquals("[@javax.persistence.Table(name=\"T_P_UTILISATEUR\",uniqueConstraints={@javax.persistence.UniqueConstraint(columnNames={\"UTI_CODE\"})})]", annots.toString());
    }
    
    public void testBug3216466() throws Exception {
    	String cuText = "class A { { java.util.Arrays.asList(); }}\n";
    	runIt(cuText);
    	Arrays.asList();
    }
    
    public void testBug3219362() throws Exception {
    	// source of .class file: 
    	// import java.util.*;
    	// public class X {
    	//       public List<Double[]> f = new ArrayList();
    	// }
    	String cuText = "import java.util.*;\n" +
    					"public class Test {\n"+
    					"  public void method(X u) {\n"+
    					"    u.f.add(new Double[]{0.0d, 0.0d});\n"+
						"  }\n" +
						"}\n";
    	sc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, "test/fixedBugs/Bug3219362");
    	runIt(cuText);
    }
}