package recoderanalyser.analyse;

import java.util.ArrayList;
import java.util.List;

import recoder.CrossReferenceServiceConfiguration;
import recoder.abstraction.ArrayType;
import recoder.abstraction.ClassType;
import recoder.abstraction.Constructor;
import recoder.abstraction.Field;
import recoder.abstraction.Method;
import recoder.abstraction.PrimitiveType;
import recoder.abstraction.ProgramModelElement;
import recoder.abstraction.Type;
import recoder.convenience.TreeWalker;
import recoder.java.CompilationUnit;
import recoder.java.declaration.ConstructorDeclaration;
import recoder.java.declaration.MethodDeclaration;
import recoder.java.declaration.TypeDeclaration;
import recoder.service.NameInfo;
import recoder.service.SourceInfo;
import recoderanalyser.solidsxtree.SolidSXClassNode;
import recoderanalyser.solidsxtree.SolidSXConstructorNode;
import recoderanalyser.solidsxtree.SolidSXFieldNode;
import recoderanalyser.solidsxtree.SolidSXMethodNode;
import recoderanalyser.solidsxtree.SolidSXNode;
import recoderanalyser.solidsxtree.SolidSXPackageNode;
import recoderanalyser.solidsxtree.SolidSXReference;
import recoderanalyser.solidsxtree.SolidSXReference.ReferenceType;
import recoderanalyser.solidsxtree.SolidSXRootNode;

public class SolidSXTree 
{
    private SolidSXRootNode rootNode;
    private SourceInfo si;
    private NameInfo ni;
    private List<ProgramModelElement> references;
    
    public SolidSXRootNode generateTree(CrossReferenceServiceConfiguration crsc, List<CompilationUnit> units, long revision)
    {
        System.out.println("STATUS_MESSAGE: Generating tree");
        this.rootNode = new SolidSXRootNode(revision);
        this.si = crsc.getSourceInfo();
        this.ni = crsc.getNameInfo();
        this.references = new ArrayList<ProgramModelElement>();
        processCompilationUnits(units);
        processReferences();
        sortNodes(rootNode);
        return rootNode;
    }
    
    private void sortNodes(SolidSXNode node)
    {
        node.sortChildren();
        for(SolidSXNode child : node.getChildren())
        {
            sortNodes(child);
        }
    }
    
    private void processReferences()
    {        
        for(ProgramModelElement pme : references)
        {           
            if(pme.getFullName().equals("<unknownClassType>"))
            {
                continue;
            }
            String path = (pme instanceof Method) ? getPath(pme.getFullName()).concat(".").concat(getMethodSignature((Method)pme)) : pme.getFullName();
            boolean isPrimitive = isPrimitive(path);
            
            if(isPrimitive)
            {
                path = "<build-ins>."+path;
            }
            
            if(!rootNode.nodeExists(path, rootNode.getRevision()))
            {
                String classPath;
                if(pme instanceof ClassType)
                {
                    classPath = isPrimitive ? "<build-ins>."+pme.getFullName() : pme.getFullName();
                }
                else
                {
                    classPath = pme.getFullName().contains(".") ? pme.getFullName().substring(0, pme.getFullName().lastIndexOf(".")) : pme.getFullName();
                    if(isPrimitive)
                    {
                        classPath = String.format("<build-ins>.%s", classPath);
                    }
                }
                
                if(classPath.equals("<unknownMethod>"))
                {
                    continue;
                }
                
                SolidSXClassNode classNode = addClassNode(classPath);
                
                if(pme instanceof Field)
                {
                    Field f = (Field)pme;
                    classNode.addChildNode(new SolidSXFieldNode(f, rootNode.getRevision()));                    
                } 
                else if(pme instanceof Constructor)
                {
                    Constructor c = (Constructor)pme;
                    classNode.addChildNode(new SolidSXConstructorNode(getMethodSignature(c), rootNode.getRevision(), c));
                }
                else if(pme instanceof Method)
                {
                    Method m = (Method)pme;
                    classNode.addChildNode(new SolidSXMethodNode(getMethodSignature(m), rootNode.getRevision(), m));           
                }
            }
        }
    }
    
    private void processCompilationUnits(List<CompilationUnit> units)
    {   
        int unitsProcessed = 1;
        int lastPercentage = -1;
        
        for(CompilationUnit cu : units)
        { 
            //Show progress
            int calculatedPercentage = (unitsProcessed * 100) / units.size();
            if(lastPercentage != calculatedPercentage)
            {
                System.out.println(String.format("STATUS_VALUE: %d%%", calculatedPercentage));
                lastPercentage = calculatedPercentage;
            }
            unitsProcessed++;
            
            SolidSXClassNode classNode = null;
            
            //Get classes
            for(TypeDeclaration td : cu.getDeclarations())
            {
                if(td instanceof ClassType)
                {
                    if(classNode == null)
                    {
                        classNode = processClassType(td);
                    }
                    else
                    {
                        processClassType(td);
                    }
                }
            }
        }
    }
    
    private SolidSXClassNode processClassType(ClassType ct)
    {   
        //Create class node
        SolidSXClassNode classNode = addClassNode(ct.getFullName());
        
        //Get inheritance
        for(ClassType inheritClass : ct.getSupertypes())
        {
            this.addReference(classNode, inheritClass.getFullName(), inheritClass, ReferenceType.Inheritance);
        }
        
        //Get constructors
        for(Constructor c : ct.getConstructors())
        {
            ConstructorDeclaration cd = si.getConstructorDeclaration(c);
            if(cd != null)
            {
                SolidSXConstructorNode contructorNode  = (SolidSXConstructorNode)classNode.addChildNode(new SolidSXConstructorNode(getMethodSignature(c), rootNode.getRevision(), c));
                SolidSXMethodVisitor methodVisitor = new SolidSXMethodVisitor(contructorNode, si, this);            
                TreeWalker tw = new TreeWalker(cd);
                while(tw.next())
                {
                    tw.getProgramElement().accept(methodVisitor);
                }
            }
        }
        
        //Get fields
        for(Field f : ct.getFields())
        {
            SolidSXFieldNode fieldNode = (SolidSXFieldNode)classNode.addChildNode(new SolidSXFieldNode(f, rootNode.getRevision()));
            addReference(fieldNode, f.getType().getFullName(), f.getType(), ReferenceType.IsA);         
        }
        
        //Get methods
        for(Method m : ct.getMethods())
        {          
            MethodDeclaration md = si.getMethodDeclaration(m);
            if(md != null)
            {
                SolidSXMethodNode methodNode  = (SolidSXMethodNode)classNode.addChildNode(new SolidSXMethodNode(getMethodSignature(m), rootNode.getRevision(), m));
                
                if(md.getReturnType() != null)
                {
                    addReference(methodNode, md.getReturnType().getFullName(), md.getReturnType(), ReferenceType.Uses);
                }
                for(Type t : md.getSignature())
                {
                    addReference(methodNode, t.getFullName(), t, ReferenceType.Uses);
                }
                
                SolidSXMethodVisitor methodVisitor = new SolidSXMethodVisitor(methodNode, si, this);            
                TreeWalker tw = new TreeWalker(md);
                while(tw.next())
                {
                    tw.getProgramElement().accept(methodVisitor);
                }
            }
        }
        
        //Get classes
        for(ClassType classType : ct.getTypes())
        {
            processClassType(classType);
        }
        return classNode;
    }
    
    private SolidSXClassNode addClassNode(String classPath)
    {
        SolidSXClassNode node = (SolidSXClassNode)rootNode.getNodeByPath(classPath, rootNode.getRevision());
        if(node == null)
        {
            SolidSXNode parent = rootNode;
            String path = "";
            for(String nodeName : getPath(classPath).split("\\."))
            {
                path = path.isEmpty() ? nodeName : path.concat(".").concat(nodeName);
                SolidSXNode foundNode = parent.getChildByName(nodeName, rootNode.getRevision());
                if(foundNode == null)
                {
                    if(ni.getClassType(path) != null)
                    {
                        parent = parent.addChildNode(new SolidSXClassNode(nodeName, rootNode.getRevision()));
                    }
                    else
                    {
                        parent = parent.addChildNode(new SolidSXPackageNode(nodeName, rootNode.getRevision()));
                    }
                }
                else
                {
                    parent = foundNode;
                }
            }
            return (SolidSXClassNode)parent.addChildNode(new SolidSXClassNode(getName(classPath), rootNode.getRevision()));
        }
        return node;
    }
          
    public void addReference(SolidSXNode node, String ref, ProgramModelElement pme, ReferenceType type)
    {
        if(isPrimitive(ref))
        {
            ref = String.format("<build-ins>.%s", ref);
        }
        node.addReference(new SolidSXReference(ref, type));
        if(!references.contains(pme))
        {
            references.add(pme);
        }
    }
    
    private boolean isPrimitive(String ref)
    {
        Type type = ref.contains(".") ? ni.getType(ref.substring(0, ref.indexOf("."))) : ni.getType(ref);
        
        if(type instanceof PrimitiveType)
        {
            return true;
        }
        else
        {
            while(type instanceof ArrayType)
            {
                type = ((ArrayType)type).getBaseType();
                if(type instanceof PrimitiveType)
                {
                    return true;
                }
            }
        }
        return false;
    }
    
    public String getMethodSignature(Method m)
    {
        StringBuilder name = new StringBuilder(getName(m.getFullName()));
        name.append("(");
        for(Type type : m.getSignature())
        {
            if(type.getFullSignature().startsWith(String.format("%s extends", type.getName())))
            {
                name.append(type.getFullSignature().substring(type.getFullSignature().lastIndexOf(".")+1));
            }
            else
            {
                name.append(type.getName());
            }
            name.append(", ");
        }
        if(!name.toString().endsWith("("))
        {
            name.delete(name.length()-2, name.length());
        }
        name.append(")");
        
        return name.toString();
    }
    
    private String getPath(String fullPath)
    {
        return (fullPath.equals("<unknownMethod>") || !fullPath.contains(".")) ? fullPath : fullPath.substring(0, fullPath.lastIndexOf("."));
    }
    
    private String getName(String fullPath)
    {
        return fullPath.substring(fullPath.lastIndexOf(".")+1);
    }
    
    public SolidSXRootNode getRootNode()
    {
        return rootNode;
    }
}
