package recoderanalyser.output;

import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Calendar;
import java.util.List;
import recoderanalyser.solidsxtree.SolidSXNode;
import recoderanalyser.solidsxtree.SolidSXRootNode;
import recoderanalyser.solidsxtree.SolidSXNode.NodeType;
import recoderanalyser.solidsxtree.SolidSXReference;

public class TreeToSolidSX2DB 
{   
    private static long edgeID;
    private static long numberOfNodes;
    private static long nodesProcessed;
    private static int lastPercentage;
    private static Statement statement;
    
    public static void generateDB(SolidSXRootNode rootNode, String outputPath)
    {
        System.out.println("STATUS_MESSAGE: Generating SolidSX2 database");
        String filePath = outputPath.toLowerCase().endsWith(".db") ? outputPath : String.format("%s%s.db", outputPath, rootNode.getStringRevision());
        edgeID = 1;
        numberOfNodes = rootNode.updateNodeIDs();
        nodesProcessed = 1;
        lastPercentage = -1;
        
        Connection connection = null;
        
        //Check if there is a database with the same path
        File dbFile = new File(filePath);
        if(dbFile.exists())
        {
            dbFile.delete();
        }
        
        try
        {
            //Load the sqlite-JDBC driver using the current class loader
            Class.forName("org.sqlite.JDBC");

            //Create a database connection
            connection = DriverManager.getConnection(String.format("jdbc:sqlite:%s", filePath));
            connection.setAutoCommit(false);
            statement = connection.createStatement();
            
            //Create table and process nodes
            createTables();
            processSolidSXNode(rootNode, rootNode);
            
            //Commit generated transaction
            System.out.println(String.format("STATUS_MESSAGE: Writing file %s", filePath));
            connection.commit();            
        }
        catch(ClassNotFoundException cnfe)
        {
            System.err.println(String.format("ClassNotFoundException: %s", cnfe.getMessage()));
        }
        catch(SQLException e)
        {
            System.err.println(String.format("SQLException: %s", e.getMessage()));
        }
        finally
        {
            try
            {
                if(connection != null)
                {
                    connection.close();
                }
            } 
            catch(SQLException e)
            {
                System.err.println(String.format("SQLException: %s", e.getMessage()));
            }
        }
    }
    
    private static void createTables() throws SQLException
    {
        //Create tables and indexs
        statement.executeUpdate("CREATE TABLE edgeattr (eid INTEGER KEY, key STRING, value STRING, type STRING);");
        statement.executeUpdate("CREATE TABLE edges (eid INTEGER PRIMARY KEY, fromnid INTEGER, tonid INTEGER);");
        statement.executeUpdate("CREATE TABLE hierarchy (hid INTEGER, nid INTEGER, parentnid INTEGER, childorder INTEGER);");
        statement.executeUpdate("CREATE TABLE nodeattr (nid INTEGER KEY, key STRING, value STRING, type STRING);");
        statement.executeUpdate("CREATE TABLE information(key TEXT, value);");
        statement.executeUpdate("CREATE INDEX edgeattr_eid ON edgeattr (eid)");
        statement.executeUpdate("CREATE INDEX hierarchy_parentnid ON hierarchy (parentnid);");
        statement.executeUpdate("CREATE INDEX hierarchy_parentnid_hid ON hierarchy (parentnid,hid);");
        statement.executeUpdate("CREATE INDEX nodeattr_nid ON nodeattr (nid);");
        
        //Insert infromation
        statement.executeUpdate("INSERT INTO information VALUES('information','Produced by Revision Analyser (developed by Mark Ettema and Johan van der Geest).');");
        statement.executeUpdate("INSERT INTO information VALUES('database format version',2);");
        statement.executeUpdate(String.format("INSERT INTO information VALUES('creation date','%s');", Calendar.getInstance().getTime().toString()));
    }
    
    private static void processSolidSXNode(SolidSXNode node, SolidSXRootNode rootNode) throws SQLException
    {
        //Calculate process
        int calculatedPercentage = (int)((nodesProcessed * 100) / numberOfNodes);
        if(lastPercentage != calculatedPercentage)
        {
            System.out.println(String.format("STATUS_VALUE: %d%%", calculatedPercentage));
            lastPercentage = calculatedPercentage;
        }
        nodesProcessed++;
        
        //Create node
        createNode(node);
        
        //Create hierarchy
        createHierarchy(node);
        
        //Create edges
        for(SolidSXReference ref : node.getReferences())
        {      
            SolidSXEdges solidSXEdges = new SolidSXEdges();
            
            for(long revision : node.getRevisions())
            {
                SolidSXNode result = rootNode.getNodeByPath(ref.getPath(), revision);
                if(result == null)
                {
                    System.err.println("WARNING: Could not find reference: "+ref.getPath()+" (for revision "+revision+")");
                }
                else
                {
                    solidSXEdges.addEdge(result.getID(), revision);
                }
            }
            
            for(SolidSXEdge edge : solidSXEdges.getEdgeList())
            {
                createEdge(node.getID(), edge.getToNodeID(), ref.getType().toString(), edge.getRevisionList());
            }
        }
        
        //Process children
        for(SolidSXNode child : node.getChildren())
        {
            processSolidSXNode(child, rootNode);
        }
    }
    
    private static void createNode(SolidSXNode node) throws SQLException
    {
        statement.executeUpdate(String.format("INSERT INTO nodeattr VALUES(%d,'name','%s',NULL);", node.getID(), node.getName()));
        statement.executeUpdate(String.format("INSERT INTO nodeattr VALUES(%d,'type','%s',NULL);", node.getID(), node.getType().toString()));
        statement.executeUpdate(String.format("INSERT INTO nodeattr VALUES(%d,'_icon','%s',NULL);", node.getID(), getIconType(node.getType())));
        statement.executeUpdate(String.format("INSERT INTO nodeattr VALUES(%d,'numberOfRevisions','%d',NULL);", node.getID(), node.getRevisions().size()));
        for(Long revision : node.getRevisions())
        {
            statement.executeUpdate(String.format("INSERT INTO nodeattr VALUES(%d,'revision%d','%d',NULL);", node.getID(), revision, revision));
        }
    }
    
    private static void createHierarchy(SolidSXNode node) throws SQLException
    {
        if(node.getParent() != null)
        {
            statement.executeUpdate(String.format("INSERT INTO hierarchy VALUES(1,%d,%d,0);", node.getID(), node.getParent().getID()));
        }
        else
        {
            statement.executeUpdate(String.format("INSERT INTO hierarchy VALUES(1,%d,NULL,0);", node.getID()));
        }
    }
    
    private static void createEdge(long fromID, long toID, String type, List<Long> revisions) throws SQLException
    {
        statement.executeUpdate(String.format("INSERT INTO edges VALUES(%d,%d,%d);", edgeID, fromID, toID));        
        statement.executeUpdate(String.format("INSERT INTO edgeattr VALUES(%d,'type','%s',NULL);", edgeID, type));
        for(long revision : revisions)
        {
            statement.executeUpdate(String.format("INSERT INTO edgeattr VALUES(%d,'revision%d','%d',NULL);", edgeID, revision, revision));
        }
        statement.executeUpdate(String.format("INSERT INTO edgeattr VALUES(%d,'count',%d,NULL);", edgeID, 1));
        edgeID++;
    }
    
    private static String getIconType(NodeType type)
    {
        switch(type)
        {
            case Root: return "database";
            case Package: return "namespace";
            case Class: return "class";
            case Constructor:
            case Method: return "method_public";
            case Field: return "field";
            default: return "";
        }  
    }
}
