package recoderanalyser.output;

import java.io.File;
import java.util.List;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import recoderanalyser.solidsxtree.SolidSXNode;

import recoderanalyser.solidsxtree.SolidSXNode.NodeType;
import recoderanalyser.solidsxtree.SolidSXReference;
import recoderanalyser.solidsxtree.SolidSXRootNode;

public class TreeToSolidSX2XML 
{       
    private static Document xmlDoc;
    private static Element nodes;
    private static Element hierarchy;
    private static Element edges;
    private static long numberOfNodes;
    private static long nodesProcessed;
    private static int lastPercentage;
        
    public static void generateXML(SolidSXRootNode rootNode, String outputPath)
    {
        System.out.println("STATUS_MESSAGE: Generating SolidSX2 XML");
        numberOfNodes = rootNode.updateNodeIDs();
        nodesProcessed = 1;
        lastPercentage = -1;
        
        createBasicXmlDoc();
        processSolidSXNode(rootNode, rootNode);
        
        String filePath = outputPath.toLowerCase().endsWith(".xml") ? outputPath : String.format("%s%s.xml", outputPath, rootNode.getStringRevision());
        if(!outputPath.endsWith(".xml"))
        {
            new File(outputPath).mkdirs();
        }
        writeToFile(filePath);
    }
    
    private static void createBasicXmlDoc()
    {
        try 
        {
            xmlDoc = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
            Element solidSXDataset = xmlDoc.createElement("SolidSXDataset");
            xmlDoc.appendChild(solidSXDataset);
            
            //Nodes
            nodes = xmlDoc.createElement("nodes");
            solidSXDataset.appendChild(nodes);
            
            //Hierarchy
            hierarchy = xmlDoc.createElement("hierarchy");
            hierarchy.setAttribute("name", "default");
            solidSXDataset.appendChild(hierarchy);
            
            //Edges
            edges = xmlDoc.createElement("edges");
            solidSXDataset.appendChild(edges);
        }
        catch(ParserConfigurationException pce) 
        {
            System.err.println("ParseConfigurationException: "+pce.getMessage());
        }
    }
    
    private static void processSolidSXNode(SolidSXNode node, SolidSXRootNode rootNode)
    {
        //Calculate process
        int calculatedPercentage = (int)((nodesProcessed * 100) / numberOfNodes);
        if(lastPercentage != calculatedPercentage)
        {
            System.out.println(String.format("STATUS_VALUE: %d%%", calculatedPercentage));
            lastPercentage = calculatedPercentage;
        }
        nodesProcessed++;
        
        createXmlNode(node.getID(), node.getName(), node.getType(), node.getRevisions());
        if(node.getParent() != null)
        {
            createHierarchy(node.getParent().getID(), node.getID());
        }
        
        for(SolidSXReference ref : node.getReferences())
        {      
            SolidSXEdges solidSXEdges = new SolidSXEdges();
            
            for(long revision : node.getRevisions())
            {
                SolidSXNode result = rootNode.getNodeByPath(ref.getPath(), revision);
                if(result == null)
                {
                    System.err.println("WARNING: Could not find reference: "+ref.getPath()+" (for revision "+revision+")");
                }
                else
                {
                    solidSXEdges.addEdge(result.getID(), revision);
                }
            }
            
            for(SolidSXEdge edge : solidSXEdges.getEdgeList())
            {
                createEdge(node.getID(), edge.getToNodeID(), ref.getType().toString(), edge.getRevisionList());
            }
        }
        
        for(SolidSXNode child : node.getChildren())
        {
            processSolidSXNode(child, rootNode);
        }
    }
    
    private static void writeToFile(String filePath)
    {
        System.out.println(String.format("STATUS_MESSAGE: Writing file %s", filePath));
        try 
        {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();            
            transformer.setOutputProperty(OutputKeys.ENCODING, "utf-8");
            transformer.setOutputProperty(OutputKeys.INDENT, "yes");
            transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");

            StreamResult result =  new StreamResult(new File(filePath));
            transformer.transform(new DOMSource(xmlDoc), result);
        }
        catch(TransformerException tfe) 
        {
            System.err.println(String.format("TransformerException: %s", tfe.getMessage()));
        }
    }

    private static void createXmlNode(long nodeID, String name, NodeType type, List<Long> revisions)
    {
        Element node = xmlDoc.createElement("node");
        node.setAttribute("nid", Long.toString(nodeID));
        Element nameAttribute = xmlDoc.createElement("attribute");
        nameAttribute.setAttribute("key", "name");
        nameAttribute.setAttribute("value", name);
        node.appendChild(nameAttribute);
        Element typeAttribute = xmlDoc.createElement("attribute");
        typeAttribute.setAttribute("key", "type");
        typeAttribute.setAttribute("value", type.toString());
        node.appendChild(typeAttribute);   
        Element iconAttribute = xmlDoc.createElement("attribute");
        iconAttribute.setAttribute("key", "_icon");
        iconAttribute.setAttribute("value", getIconType(type));
        node.appendChild(iconAttribute);
        Element numberOfRevisionAttribute = xmlDoc.createElement("attribute");
        numberOfRevisionAttribute.setAttribute("key", "numberOfRevisions");
        numberOfRevisionAttribute.setAttribute("value", Integer.toString(revisions.size()));
        node.appendChild(numberOfRevisionAttribute);
        for(Long revision : revisions)
        {
            Element revisionAttribute = xmlDoc.createElement("attribute");
            revisionAttribute.setAttribute("key", String.format("revision%d", revision));
            revisionAttribute.setAttribute("value", Long.toString(revision));
            
            node.appendChild(revisionAttribute);
        }
        nodes.appendChild(node);
    }
    
    private static String getIconType(NodeType type)
    {
        switch(type)
        {
            case Root: return "database";
            case Package: return "namespace";
            case Class: return "class";
            case Constructor:
            case Method: return "method_public";
            case Field: return "field";
            default: return "";
        }  
    }
    
    private static void createHierarchy(long fromID, long toID)
    {
        Element edge = xmlDoc.createElement("edge");
        edge.setAttribute("fromnid", Long.toString(fromID));
        edge.setAttribute("tonid", Long.toString(toID));
        hierarchy.appendChild(edge);
    }
    
    private static void createEdge(long fromID, long toID, String type, List<Long> revisions)
    {
        Element edge = xmlDoc.createElement("edge");
        edge.setAttribute("fromnid", Long.toString(fromID));
        edge.setAttribute("tonid", Long.toString(toID));
        Element typeAttribute = xmlDoc.createElement("attribute");
        typeAttribute.setAttribute("key", "type");
        typeAttribute.setAttribute("value", type);
        edge.appendChild(typeAttribute);
        for(long revision : revisions)
        {
            Element revisionAttribute = xmlDoc.createElement("attribute");
            revisionAttribute.setAttribute("key", String.format("revision%d", revision));
            revisionAttribute.setAttribute("value", Long.toString(revision));
            edge.appendChild(revisionAttribute);
        }
        Element countAttribute = xmlDoc.createElement("attribute");
        countAttribute.setAttribute("key", "count");
        countAttribute.setAttribute("value", "1");
        edge.appendChild(countAttribute);
        edges.appendChild(edge);
    }
}
