package recoderanalyser.solidsxtree;

import java.util.List;

public class SolidSXRootNode extends SolidSXNode
{
    public final static long MERGED = -1;
    private long nodeID;
    private long revision;

    public SolidSXRootNode(long revision) 
    {
        super("Root", revision, NodeType.Root);
        this.addChildNode(new SolidSXPackageNode("<build-ins>", revision));
        this.revision = revision;
    }
    
    public SolidSXNode getNodeByPath(String path, long revision)
    {
        SolidSXNode node = this;
        for(String nodeName : path.split("\\."))
        {
            node = node.getChildByName(nodeName, revision);
            if(node == null)
            {
                return null;
            }
        }
        return node;
    }
    
    public boolean nodeExists(String path, long revision)
    {
        return getNodeByPath(path, revision) != null;
    }
    
    public long updateNodeIDs()
    {
        this.nodeID = 1;
        setNodeID(this);
        return nodeID-1;
    }
    
    public long getRevision()
    {
        return revision;
    }
    
    public String getStringRevision()
    {
        if(this.revision == MERGED)
        {
            return "merged";
        }
        else 
        {
            return Long.toString(this.revision);
        }
    }
    
    private void setNodeID(SolidSXNode node)
    {
        node.setID(nodeID++);
        for(SolidSXNode child : node.getChildren())
        {
            setNodeID(child);
        }
    }
    
    public void merge(SolidSXRootNode mergeWith) throws Exception
    {
        if(mergeWith.getRevision() == MERGED)
        {
            throw new Exception("Merge error: could not merge with an already merged file");
        }
        this.revision = MERGED;
        mergeNode(mergeWith, this, mergeWith.getRevision());
    }
    
    private void mergeNode(SolidSXNode node, SolidSXNode mergedNode, long revision)
    {
        mergedNode.addRevision(revision);
        for(SolidSXNode child : node.getChildren())
        {
            //Check if the child is also a child of mergedNode
            int index = mergedNode.getChildren().indexOf(child);
            if(index == -1)
            {
                //Node is not a child, so add
                mergedNode.addChildNode(child);
            }
            else
            {
                //Node is a child, so process the childNode
                mergeNode(child, mergedNode.getChildren().get(index), revision);
            }
        }
    }
    
    public void removeRevision(long revision)
    {
        removeRevisionFromNode(this, revision);
    }
    
    private boolean removeRevisionFromNode(SolidSXNode node, long revision)
    {
        //Remove revision from the revision list
        node.getRevisions().remove(revision);
        
        //Check if there are no more revision, then the node can be removed
        if(node.getRevisions().isEmpty())
        {
            return true;
        }
        else
        {
            //Process children
            List<SolidSXNode> children = node.getChildren();
            for(int i=0; i<children.size(); i++)
            {
                if(removeRevisionFromNode(children.get(i), revision))
                {
                    node.getChildren().remove(i);
                    i--;
                }
            }
        }
        return false;
    }
}
