// This file is part of the RECODER library and protected by the LGPL.

package recoder.bytecode;

import recoder.ModelException;
import recoder.abstraction.ProgramModelElement;
import recoder.service.ByteCodeInfo;
import recoder.service.ProgramModelInfo;

public abstract class ByteCodeElement implements ProgramModelElement {

    protected int accessFlags;

    private String name;

    protected ByteCodeInfo service;

    public ByteCodeElement() {
    	super();
    }

    public ByteCodeElement(int accessFlags) {
        this.accessFlags = accessFlags;
    }

    public ByteCodeElement(int accessFlags, String name) {
        setName(name);
        this.accessFlags = accessFlags;
    }

    /**
     * Returns the name of the byte code element. Strings are interned so that
     * they can be compared by identity.
     */
    public final String getName() {
        return name;
    }

    final void setName(String name) {
        this.name = name.intern();
    }

    public abstract String getTypeName();

    public final int getAccessFlags() {
        return accessFlags;
    }

    public void setAccessFlags(int accessFlags) {
        this.accessFlags = accessFlags;
    }

    public boolean isAbstract() {
        return (accessFlags & ABSTRACT) != 0;
    }

    public boolean isFinal() {
        return (accessFlags & FINAL) != 0;
    }

    public boolean isStatic() {
        return (accessFlags & STATIC) != 0;
    }

    public boolean isPrivate() {
        return (accessFlags & PRIVATE) != 0;
    }

    public boolean isProtected() {
        return (accessFlags & PROTECTED) != 0;
    }

    public boolean isPublic() {
        return (accessFlags & PUBLIC) != 0;
    }

    public boolean isStrictFp() {
        return (accessFlags & STRICT) != 0;
    }

    public boolean isNative() {
        return (accessFlags & NATIVE) != 0;
    }

    public boolean isSynchronized() {
        return (accessFlags & SYNCHRONIZED) != 0;
    }

    public ByteCodeInfo getProgramModelInfo() {
        return service;
    }

    public void setProgramModelInfo(ProgramModelInfo service) {
    	if (!(service instanceof ByteCodeInfo))
    		throw new IllegalArgumentException("service for MethodDeclaration must be of type SourceInfo.");
        this.service = (ByteCodeInfo)service;
    }

    public void validate() throws ModelException {
    	// not checked yet
    }

}