/*
 * Created on 11.03.2005

 *
 * This file is part of the RECODER library and protected by the LGPL.
 */
package recoder.testsuite.transformation;
import java.util.List;

import recoder.abstraction.ClassType;
import recoder.abstraction.Field;
import recoder.abstraction.Method;
import recoder.java.CompilationUnit;
import recoder.java.Identifier;
import recoder.java.StatementBlock;
import recoder.java.declaration.ClassDeclaration;
import recoder.java.declaration.ConstructorDeclaration;
import recoder.java.declaration.MethodDeclaration;
import recoder.java.declaration.modifier.Private;
import recoder.java.declaration.modifier.Public;
import recoder.java.declaration.modifier.VisibilityModifier;
import recoder.java.reference.MethodReference;
import recoder.java.reference.TypeReference;
import recoder.kit.MethodKit;
import recoder.kit.NoProblem;
import recoder.kit.transformation.java5to4.EnhancedFor2For;
import recoder.kit.transformation.java5to4.FloatingPoints;
import recoder.kit.transformation.java5to4.MakeConditionalCompatible;
import recoder.kit.transformation.java5to4.RemoveAnnotations;
import recoder.kit.transformation.java5to4.RemoveCoVariantReturnTypes;
import recoder.kit.transformation.java5to4.RemoveStaticImports;
import recoder.kit.transformation.java5to4.ReplaceEnumsNew;
import recoder.kit.transformation.java5to4.ResolveBoxing;
import recoder.kit.transformation.java5to4.ResolveGenericsNew;
import recoder.kit.transformation.java5to4.ResolveVarArgs;
import recoder.kit.transformation.java5to4.methodRepl.ApplyRetrotranslatorLibs;
import recoder.kit.transformation.java5to4.methodRepl.ReplaceEmptyCollections;
import recoder.service.ChangeHistory;
import recoder.service.NameInfo;
import recoder.testsuite.RecoderTestCase;
import application.Obfuscate;


/**
 * @author Tobias Gutzmann
 *
 * Tests some transformations.
 */
public class TransformationTests extends RecoderTestCase {
    private boolean silent = true;
    
       
    public void testObfuscater() {
        setPath("test/transformation/obfuscate");
        runIt();
        Obfuscate of = new Obfuscate(sc);
        if (of.analyze() instanceof NoProblem)
            of.transform();
        // TODO write back and compare!
    }
    
    public void testReadOnly() {
    	setPath("test/transformation/readOnly");
    	runIt();
    	
    	List<TypeReference> trl = sc.getCrossReferenceSourceInfo().getReferences(sc.getNameInfo().getType("Test"));
    	for (int i = 0; i < trl.size(); i++) {
    		TypeReference tr = trl.get(i);
    		if (!silent)
    			System.out.println(tr.toSource());
    	}
    }
    
    private void defaultConstructorReferences(VisibilityModifier vm) {
    	setPath("test/transformation/defaultCons");
    	runIt();
    	// add a constructor now
    	ChangeHistory ch = sc.getChangeHistory();
    	NameInfo ni = sc.getNameInfo();
    	ClassDeclaration cd = (ClassDeclaration)ni.getType("DefaultCons");
    	ConstructorDeclaration cons = new ConstructorDeclaration(vm, new Identifier("DefaultCons"),
    			null, null, new StatementBlock());
    	cd.getMembers().add(cons);
    	cd.makeParentRoleValid();
    	ch.attached(cons);
    	ch.updateModel();
    	for (CompilationUnit cu : sc.getSourceFileRepository().getCompilationUnits()) {
    		cu.validateAll();
    	}
    	// now, the default constructor no longer exists.
    	// The added constructor should be referenced, however, if it is still visible, 
    	// and an error should occur if not.
    	// TODO Recoder does not check visibility yet! So this works fine for both cases (for now)!
    	
    	int newref = sc.getCrossReferenceSourceInfo().getReferences(cons).size();
		
    	MethodDeclaration md = (MethodDeclaration)cd.getMethods().get(0); // main()
     }
    
    public void testDefaultConstructorReferences1() {
    	defaultConstructorReferences(new Public());
    }
    public void testDefaultConstructorReferences2() {
    	defaultConstructorReferences(new Private());
    }
    
    public void testReplaceEmptyCollections() {
    	setPath("test/transformation/emptyCollections");
    	runIt();
    	ClassType coll = sc.getNameInfo().getClassType("java.util.Collections");
    	checkFRefCnt(coll, "EMPTY_LIST", 0);
    	checkFRefCnt(coll, "EMPTY_MAP", 0);
    	checkFRefCnt(coll, "EMPTY_SET", 0);
    	checkFRefCnt(coll, "emptyList", 2);
    	checkFRefCnt(coll, "emptyMap", 2);
    	checkFRefCnt(coll, "emptySet", 2);
    	new ReplaceEmptyCollections(sc).execute();
    	sc.getChangeHistory().updateModel();
    	checkFRefCnt(coll, "EMPTY_LIST", 2);
    	checkFRefCnt(coll, "EMPTY_MAP", 2);
    	checkFRefCnt(coll, "EMPTY_SET", 2);
    	checkFRefCnt(coll, "emptyList", 0);
    	checkFRefCnt(coll, "emptyMap", 0);
    	checkFRefCnt(coll, "emptySet", 0);    	
    }
    // helper method for testReplaceEmptyCollections
    private void checkFRefCnt(ClassType coll, String memberName, int cnt) {
    	boolean found = false;
    	for (Field f : coll.getFields()) {
    		if (memberName.equals(f.getName())) {
    			assertTrue(cnt == sc.getCrossReferenceSourceInfo().getReferences(f).size());
    			found = true;
    			break;
    		}
    	}
    	if (!found) {
        	for (Method m : coll.getMethods()) {
        		if (memberName.equals(m.getName())) {
        			assertTrue(cnt == sc.getCrossReferenceSourceInfo().getReferences(m).size());
        			found = true;
        			break;
        		}
        	}    		
    	}
    	assertTrue(found);
    }
    
    // Testing for different issues in java5to4:
    public void testXXX() throws Exception {
    	String cuText = 
    		"import java.util.*;" +
    		"class X {\n" +
    		"public int hashCode() {\n" +
    		"List<Collection> l = null;" +
    		"String s = null;" +
    	    "return (l != null ? l : s).hashCode();"+
            "}}";
    	List<CompilationUnit> cus = runIt(cuText);
    	new MakeConditionalCompatible(sc, cus).execute();
    	// TODO move to TestTransformations
    }
    
    public void testZZZ() throws Exception {
    	String cuText1 = 
    		"abstract class A<T> {"+
    		"abstract void foo(T t);"+
    		"}"+
    		"class B extends A<String> {"+
    		"void foo(String s) {}"+
    		"}" +
    		"class C<T> extends A<T> {" +
    		"void foo(String s) { }" +
    		"void foo(T t) { }" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	// TODO move to TestTransformations
    }
    
    public void testYASDF() throws Exception {
    	String cuText1 = 
    		"class A<Q> {\n"+
    		"	<A> A foo(Class<A> c) { return null; }\n" +
    		"}\n"+
    		"class B {\n" +
    		"	void bar(A a) {\n" +
    		"		String s = a.foo(String.class);\n" +
    		"	}\n"+
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	// TODO move to TestTransformations
    }
    
    public void testQQQ() throws Exception {
    	String cuText1 = 
    		"import java.io.*;\n" +
    		"class A {\n"+
    		"<T extends Exception> int foo(Class<T> exc) throws T { return 0; }\n"+
    		"void bar() throws IOException {\n"+
    		"foo(IOException.class);\n"+
    		"int i = foo(IOException.class);\n"+
			"}\n"+
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	// TODO move to TestTransformations
    }
    public void testWrongStaticImport() throws Exception {
    	String cuText1 = 
    		"package a; \n public class X { protected int i; }\n";
    	String cuText2 = 
    		"package b;\n" +
    		"import static a.X.*;\n" +
    		"import static a.X.i;\n" +
    		"class Y extends a.X {\n" +
    		"	void foo() { int j = i; }\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1, cuText2);
    	new RemoveStaticImports(sc, cus).execute();
    	assertEquals(
    			"packageb;classYextendsa.X{voidfoo(){intj=i;}}", 
    			cus.get(1).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testGenericsAndCovariant() throws Exception {
    	String cuText1 =
    		"class RAPII extends APII<String> implements RAPI { }\n"+
    		"class APII<T> extends PII<T> { }\n" +
    		"class PII<T> { T foo() { } }\n" +
    		"interface RAPI extends RPI {}\n" +
    		"interface RPI { String foo(); }";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	assertEquals(
    			"classRAPIIextendsAPIIimplementsRAPI{}" +
    			"classAPIIextendsPII{}" +
    			"classPII{Objectfoo(){}}" +
    			"interfaceRAPIextendsRPI{}" +
    			"interfaceRPI{Objectfoo();}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    


    public void testChangeSigsWhenResolvingGenerics() throws Exception {
    	String cuText1 = 
    		"interface I<T> { <A extends Number> A foo(Class<A> s, T t); }\n" +
    		"interface J<T> extends I<T> { }\n"+
    		"abstract class A<T> implements I<T> {"+
    		"}"+
    		"class B extends A<String> {"+
    		"public <A extends Number> A foo(Class<A> s, String t) {}"+
    		"}\n" + 
    		"class C extends A<String> implements J<String> {"+
    		"public <A extends Number> A foo(Class<A> s, String t) {}"+
    		"}";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	assertEquals(
    			"interfaceI{Numberfoo(Classs,Objectt);}" +
    			"interfaceJextendsI{}" +
    			"abstractclassAimplementsI{}" +
    			"classBextendsA{publicNumberfoo(Classs,Objectt){}}" +
    			"classCextendsAimplementsJ{publicNumberfoo(Classs,Objectt){}}"
    			,cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testApplicableMethodsIssue () throws Exception {
    	String cuText1 = 
    		"import java.util.*;" +
    		"class A {\n" +
    		"	void foo() {" +
    		"		List<String> l = new ArrayList<String>();\n" +
    		"		l.toArray(new String[0]);" +
    		"	}" +
    		"}";
    	// TODO test somehow that DefaultProgramModelInfo.internalGetApplicableMethods returns
    	// only 1 method (not 2 or more). That would be a bug (!)
    	List<CompilationUnit> cus = runIt(cuText1);
    	MethodDeclaration md = (MethodDeclaration)cus.get(0).getTypeDeclarationAt(0).getMembers().get(0);
    	MethodReference mr = (MethodReference)md.getBody().getBody().get(1);
    	List<Method> m = sc.getSourceInfo().getMethods(mr);
    }

    public void testArrayIssue() throws Exception {
    	String cuText1 =
    		"abstract class A<T> {" +
    		"	abstract void foo(T a);\n" +
    		"}\n" +
    		"class B<T> extends A<T[]> {\n" +
    		"	void foo(T a[]) { int i = a.length; }\n" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText1);
    	// actual test!
    	MethodDeclaration md = (MethodDeclaration)cus.get(0).getTypeDeclarationAt(1).getMembers().get(0);
    	List<Method> redef = MethodKit.getAllRedefinedMethods(md);
    	assertEquals(1, redef.size());
    	new ResolveGenericsNew(sc, cus).execute();
    	assertEquals(
    			"abstractclassA{abstractvoidfoo(Objecta);}" +
    			"classBextendsA{voidfoo(Objecta){inti=((Object[])a).length;}}"
    			,cus.get(0).toSource().replaceAll("( |\n)", ""));
    }

    
    public void testJustAnotherCovariantIssue() throws Exception {
    	String cuText1 = 
    		"class A {\n" +
    		"	Number foo() { return null;}\n" +
    		"}\n" +
    		"class B extends A {\n" +
    		"	Integer foo() { return null;}\n" +
    		"	void bar(B b) { Number n = b.foo(); }\n" +
    		"	void foobar(B b) { Integer i = b.foo(); }\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	assertEquals(
    			"classA{Numberfoo(){returnnull;}}" +
    			"classBextendsA{Numberfoo(){returnnull;}voidbar(Bb){Numbern=b.foo();}" +
    			"voidfoobar(Bb){Integeri=((Integer)b.foo());}}"
    			,cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    
    
    public void testGenericsSignatureChanges() throws Exception {
    	String cuText1 = 
    		"interface A<T, U> {\n" +
    		"	void foo(T t, U u);\n" +
    		"}\n" +
    		"interface B<T> extends A<T, String>{\n" +
    		"	void foo(T t, String s);\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	assertEquals(
    			"interfaceA{voidfoo(Objectt,Objectu);}" +
    			"interfaceBextendsA{voidfoo(Objectt,Objects);}"
    			,cus.get(0).toSource().replaceAll("( |\n)", ""));    }
    
    public void testResolveGenericsWithVarArgsPresent() throws Exception {
    	String cuText1 = 
    		"class A<T extends Number> {\n" +
    		"	void foo(T ... t) {\n " +
    		"		int i = t.length;\n" +
    		"		t[0].intValue();\n" +
    		"		bar(t);\n " +
    		"		foobar(t[0]);\n" +
    		"	};\n" +
    		"	void bar(Number[] o) { }\n" +
    		"	void foobar(Number n) { }\n" +
    		"}\n" +
    		"class Q {\n" +
    		"	void foo(A<Integer> a) {\n" +
    		"		a.foo(new Integer(3));\n" +
    		"	}" +
    		"}";
    		
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	sc.getChangeHistory().updateModel();
    	assertEquals(
    			"classA{voidfoo(Number...t){" +
    			"inti=t.length;t[0].intValue();bar(t);foobar(t[0]);}" +
    			"voidbar(Number[]o){}voidfoobar(Numbern){}}" +
    			"classQ{voidfoo(Aa){a.foo(newInteger(3));}}" +
    			"" +
    			"",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testResolveGenericsWithVarArgsPresent_2() throws Exception {
    	String cuText1 = 
    		"class A<T> {\n" +
    		"	void foo(T ... t) {\n " +
    		"	}\n" +
    		"}\n" +
    		"class Q extends A<String> {\n" +
    		"	void foo(String ... t) { }\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new ResolveGenericsNew(sc, cus).execute();
    	sc.getChangeHistory().updateModel();
    	assertEquals("classA{voidfoo(Object...t){}}classQextendsA{voidfoo(Object...t){}}",
    		cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    
    public void testCovariantWithMemberTypes() throws Exception {
    	String cuText1 = 
    		"interface T { }\n" +
    		"interface CT { CT foo(); }\n" +
    		"class UT implements T { }\n" +
    		"class N {\n" +
    		"	static class UCT extends UT implements CT { UCT foo(); }\n" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	assertEquals(
    			"interfaceT{}interfaceCT{CTfoo();}classUTimplementsT{}"+
    			"classN{staticclassUCTextendsUTimplementsCT{CTfoo();}}", 
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }

    public void testCovariantArraysAndClone() throws Exception {
    	String cuText1 = 
    		"import recoder.list.generic.*;\n" +
    		"class A {" +
    		"	ASTArrayList a1 = null;\n" +
    		"	ASTArrayList a2 = a1.clone();\n" +
    		"	String[] t = new String[1];\n" +
    		"	String[] s = t.clone();\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	assertEquals(
    			"importrecoder.list.generic.*;classA{ASTArrayLista1=null;"+
    			"ASTArrayLista2=((ASTArrayList)a1.clone());String[]t=newString[1];"+
    			"String[]s=((String[])t.clone());}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testCovariantRelatedMethods() throws Exception {
    	String cuText1 = 
    		"interface I { java.io.Serializable f(); }\n" +
    		"interface J { Number f(); }\n" +
    		"interface K { Number f(); }\n" +
    		"class C implements I, J { public Number f() { return null; }}\n" +
    		"class D implements J, K { public Number f() { return null; }}\n" +
    		"class E { private J d = new K() { public Number f() { return null; }};}\n";
    	List<CompilationUnit> cus = runIt(cuText1);
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	assertEquals(
    			"interfaceI{java.io.Serializablef();}"+
    			"interfaceJ{java.io.Serializablef();}"+
    			"interfaceK{java.io.Serializablef();}"+
    			"classCimplementsI,J{publicjava.io.Serializablef(){returnnull;}}"+
    			"classDimplementsJ,K{publicjava.io.Serializablef(){returnnull;}}" +
    			"classE{privateJd=newK(){publicjava.io.Serializablef(){returnnull;}};}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    	
    }
    
    public void testConditionalWithBoxing() throws Exception {
    	String cuText =
    		"class A { void foo(String s) {\n" +
    		"	Boolean b = null;\n" +
    		"	boolean c = s == null ? false : b;\n" +
    		"}}";
    	List<CompilationUnit> cus = runIt(cuText);
    	new MakeConditionalCompatible(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals("classA{voidfoo(Strings){Booleanb=null;booleanc=s==null?false:(boolean)b.booleanValue();}}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testBoxingIssue() throws Exception {
    	String cuText =
    		"class A<T> {\n" +
    		"	void set(T i) { }\n" +
    		"	T get() { return null; }\n" +
    		"	void foo() {\n " +
    		"		A<Integer> a = new A<Integer>();\n" +
    		"		a.set(a.get() + 1);\n" +
    		"	}\n" +
    		"}";
    	List<CompilationUnit> cus = runIt(cuText);
    	new ResolveGenericsNew(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals("classA{voidset(Objecti){}Objectget(){returnnull;}" +
    			"voidfoo(){Aa=newA();a.set(Integer.valueOf((((Integer)a.get()).intValue())+1));}}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testBoxingIssue2() throws Exception {
    	String cuText =
    		"class A {\n" +
    		"	{\n" +
    		"		Boolean v = true;\n" +
    		"		int i = (v) ? 1 : 0;\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new ResolveBoxing(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
    			"classA{{Booleanv=Boolean.valueOf(true);inti=(v.booleanValue())?1:0;}}"
    			, cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    public void testBoxingIssue3() throws Exception {
    	String cuText =
    		"class A {\n" +
    		"	void foo(Number n) {\n" +
    		"		double d = (n instanceof Float) ? (Float)n : (Double)n;\n" + 
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new MakeConditionalCompatible(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
    			"classA{voidfoo(Numbern){doubled=(ninstanceofFloat)?(double)((Float)n)." +
    			"floatValue():(double)((Double)n).doubleValue();}}"
    			, cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    public void testMakeSureVarArgsBeforeBoxing() throws Exception {
    	String cuText =
    		"class A {\n" +
    		"	void bar(Object ... o) {\n" +
    		"		bar(new Object(), true, 3);\n" +
    		"	}\n" +
    		"	void foo(boolean ... b) {\n" +
    		"		foo(true, false);" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new ResolveVarArgs(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
    		"classA{" +
    		"voidbar(Object[]o){bar(newObject[]{newObject(),Boolean.valueOf(true),Integer.valueOf(3)});}" +
    		"voidfoo(boolean[]b){foo(newboolean[]{true,false});}}",
    		cus.get(0).toSource().replaceAll("( |\n)", "")
    	);
    }

    public void testNewArray() throws Exception {
    	String cuText =
    		"class A {\n" +
    		"	boolean [][] bs = new boolean[][] {{ true }, {false}};" +
    		"	Object[] os = new Object[] { 3, 5};\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
        		"classA{" +
        		"boolean[][]bs=newboolean[][]{{true},{false}};" +
        		"Object[]os=newObject[]{Integer.valueOf(3),Integer.valueOf(5)};}",
        		cus.get(0).toSource().replaceAll("( |\n)", "")
        	);
    }
    
    public void testDoubleUnboxing() throws Exception {
    	String cuText =
    		"class X {\n"+
    		"Integer max;\n"+
    		"Integer foo(X lhs, X rhs) {\n" +
    		"	boolean x = lhs.max == rhs.max;\n"+
    		"	return (lhs.max==null||rhs.max==null)?\n"+
    		"			(lhs.max + rhs.max):\n"+
    		"				(Integer)(lhs.max + rhs.max);\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new MakeConditionalCompatible(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
    			"classX{Integermax;Integerfoo(Xlhs,Xrhs){" +
    			"booleanx=lhs.max==rhs.max;" +
    			"return(lhs.max==null||rhs.max==null)?(Integer)" +
    			"(Integer.valueOf(lhs.max.intValue()+rhs.max.intValue())):" +
    			"(Integer)(Integer.valueOf(lhs.max.intValue()+rhs.max.intValue()));}}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    public void testCovariantIssue() throws Exception {
    	String cuText = 
    		"class C {\n" +
    		"	interface F {\n" +
    		"		Object m();\n" +
    		"	}\n" +
    		"}\n" +
    		"class B {\n" +
    		"	void p() {\n" +
    		"		new C.F() { B m() { return null; }};\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	assertEquals(
    			"classC{interfaceF{Objectm();}}" +
    			"classB{voidp(){newC.F(){Objectm(){returnnull;}};}}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testUnboxingIssue() throws Exception {
    	String cuText =
    		"class C {\n" +
    		"	void foo(Integer i, Object[] arr) {\n" +
    		"		Object o = arr[i];\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
    			"classC{voidfoo(Integeri,Object[]arr){" +
    			"Objecto=arr[i.intValue()];}}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testUnboxAndBoxBackIssue() throws Exception {
    	String cuText =
    		"class A {\n" +
    		"	Integer i;\n" +
    		"	void foo() {\n" +
    		"		i++;\n" +
    		"		Integer j = --i;\n" +
    		"		j = i++;\n" +
    		"		while (i-- > 0) { }\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new ResolveBoxing(sc, cus).execute();
    	assertEquals(
    			"classA{Integeri;voidfoo(){" +
    			"i=Integer.valueOf(i.intValue()+1);" +
    			"Integerj=(i=Integer.valueOf(i.intValue()-1));" +
    			"j=((i=Integer.valueOf(i.intValue()+1))-1);" +
    			"while(((i=Integer.valueOf(i.intValue()-1))+1)>0){}" +
    			"}}"
    			,cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
    public void testIncDecOperatorWithBoxing() throws Exception {
    	String cuText =
    		"class A {\n" +
    		"	Integer i;\n" +
    		"	void foo() {\n" +
    		"		i++;" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	assertEquals("java.lang.Integer",sc.getSourceInfo().getType(
    			((MethodDeclaration)cus.get(0).getTypeDeclarationAt(0).getMembers().get(1))
    			.getBody().getBody().get(0)).getFullName());
    }
    
    public void testIntegerValueOfIssue() throws Exception {
    	sc.getProjectSettings().ensureExtensionClassesAreInPath();
    	String cuText =
    		"class A{\n" +
    		"	void foo() {\n" +
    		"		Integer i = 5;\n" +
    		"	}\n" +
    		"}\n";
    	List<CompilationUnit> cus = runIt(cuText);
    	new MakeConditionalCompatible(sc, cus).execute();
    	new EnhancedFor2For(sc, cus).execute();
    	new ResolveGenericsNew(sc, cus).execute();
    	new RemoveCoVariantReturnTypes(sc, cus).execute();
    	new RemoveAnnotations(sc, cus).execute();
    	new RemoveStaticImports(sc, cus).execute();
    	new ResolveVarArgs(sc, cus).execute();
    	new ResolveBoxing(sc, cus).execute();
    	new ReplaceEnumsNew(sc).execute();
    	new FloatingPoints(sc, cus).execute();
    	new ApplyRetrotranslatorLibs(sc, "lib/").execute();
    	assertEquals("classA{voidfoo(){Integeri=net.sf.retrotranslator.runtime.java.lang._Integer.valueOf(5);}}",
    			cus.get(0).toSource().replaceAll("( |\n)", ""));
    }
    
}