package recoderanalyser;

import java.io.File;
import java.io.PrintWriter;
import org.apache.commons.cli.*;
import recoderanalyser.analyse.Analyser;
import recoderanalyser.input.BinToTree;
import recoderanalyser.output.TreeToBin;
import recoderanalyser.output.TreeToSolidSX2DB;
import recoderanalyser.output.TreeToSolidSX2XML;
import recoderanalyser.solidsxtree.SolidSXRootNode;

public class RecoderAnalyser 
{     
    public static void main(String[] args) 
    {   
        Options options = new Options();
        Option outputOption = new Option("output", true, "Output path for the resulting bin file");
        outputOption.setArgName("outputPath");
        options.addOption(outputOption);
        
        Option analyseOption = new Option("analyse", true, "Analyse the JAVA source code from the given folder");
        analyseOption.setArgName("sourcePath");
        options.addOption(analyseOption);
                
        Option revisionOption = new Option("revision", true, "Revision number of the input source");
        revisionOption.setArgName("revision number");
        options.addOption(revisionOption);
        
        Option libFilesOption = new Option("libfiles", true, "Add jar/.class files, use ; to split multiple files");
        libFilesOption.setArgName("lib files");
        options.addOption(libFilesOption);
        
        Option libFoldersOption = new Option("libfolders", true, "Add all .jar/.class files from a folder (including subfolders), use ; to split multiple folders");
        libFoldersOption.setArgName("lib folder");
        options.addOption(libFoldersOption);
        
        Option ignoreOption = new Option("ignore", false, "Ignore analyse errors, show warnings");
        options.addOption(ignoreOption);
        
        Option inputOption = new Option("input", true, "Input bin file (only needed when analyse parameter is not used)");
        inputOption.setArgName("bin file");
        options.addOption(inputOption);
        
        Option mergeOption = new Option("merge", true, "Merge the result with the .bin file");
        mergeOption.setArgName("bin file");
        options.addOption(mergeOption);
        
        Option removeOption = new Option("remove", true, "Remove a revision from the input bin file (this option is ignored when using merge)");
        removeOption.setArgName("revision number");
        options.addOption(removeOption);
        
        Option solidSX2DBOption = new Option("solidsx2db", true, "Generate a SolidSX2 database file from the result");
        solidSX2DBOption.setArgName("DB path");
        options.addOption(solidSX2DBOption);
        
        Option solidSX2XMLOption = new Option("solidsx2xml", true, "Generate a SolidSX2 XML file from the result");
        solidSX2XMLOption.setArgName("XML path");
        options.addOption(solidSX2XMLOption);
       
        try
        {
            SolidSXRootNode tree = null;
            CommandLine arguments = new GnuParser().parse(options, args);
            if(!arguments.hasOption("analyse") && !arguments.hasOption("merge") && !arguments.hasOption("remove") && !arguments.hasOption("solidsx2db") && !arguments.hasOption("solidsx2xml"))
            {
                throw new ParseException("No task argument found.");
            }
            
            if(arguments.hasOption("analyse"))
            {
                if(arguments.hasOption("revision"))
                {
                    try
                    {
                        Long revision = Long.parseLong(arguments.getOptionValue("revision"));
                        tree = Analyser.analyseCode(arguments.getOptionValue("analyse"), revision, arguments.getOptionValue("libfiles"), arguments.getOptionValue("libfolders"), arguments.hasOption("ignore"));
                    }
                    catch(NumberFormatException nfe)
                    {
                        throw new ParseException("Invalid revision argument.");
                    }
                }
                else
                {
                    throw new ParseException("The revision argument is required when using analyse");
                }
            }
            else
            {
                if(!arguments.hasOption("input"))
                {
                    throw new ParseException("No input or analyse argument found");
                }
                
                if(!new File(arguments.getOptionValue("input")).exists())
                {
                    throw new ParseException("Input error: The bin file does not exists.");
                }
                
                tree = BinToTree.getSolidSXNode(arguments.getOptionValue("input"));
            }
            
            if(tree == null)
            {
                throw new ParseException("No input argument or analysed tree found");
            }
            
            if(arguments.hasOption("merge"))
            {
                if(!new File(arguments.getOptionValue("merge")).exists())
                {
                    throw new ParseException("Merge error: The bin file does not exists.");
                }
                
                SolidSXRootNode mergedTree = BinToTree.getSolidSXNode(arguments.getOptionValue("merge"));
                
                try
                {
                    mergedTree.merge(tree);          
                    tree = mergedTree;
                }
                catch(Exception e)
                {
                    throw new ParseException(e.getMessage());
                }
            }
            else if(arguments.hasOption("remove") && !arguments.hasOption("analyse"))
            {
                try
                {
                    Long revision = Long.parseLong(arguments.getOptionValue("remove"));
                    tree.removeRevision(revision);
                }
                catch(NumberFormatException nfe)
                {
                    throw new ParseException("Invalid revision argument.");
                }
            }
            
            if(arguments.hasOption("analyse") || arguments.hasOption("merge") || arguments.hasOption("remove"))
            {
                if(arguments.hasOption("output"))
                {
                    TreeToBin.generateBin(tree, arguments.getOptionValue("output"));
                }
                else if(arguments.hasOption("remove"))
                {
                    TreeToBin.generateBin(tree, arguments.getOptionValue("input"));
                }
                else
                {
                    throw new ParseException("No output argument found.");
                }
            }
            
            if(arguments.hasOption("solidsx2xml"))
            {
                TreeToSolidSX2XML.generateXML(tree, arguments.getOptionValue("solidsx2xml"));
            }
            
            if(arguments.hasOption("solidsx2db"))
            {
                TreeToSolidSX2DB.generateDB(tree, arguments.getOptionValue("solidsx2db"));
            }
            
            //Exit message
            System.out.println("STATUS_EXIT");
        }
        catch(ParseException ue)
        {
            System.err.println(String.format("%s\n", ue.getMessage()));
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(new PrintWriter(System.err, true), 90, "RecoderProject <options>", null, options, 2, 3, null);   
        }
    }
}
