package recoderanalyser.analyse;

import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.List;
import recoder.CrossReferenceServiceConfiguration;
import recoder.ParserException;
import recoder.io.PropertyNames;
import recoder.io.SourceFileRepository;
import recoder.java.CompilationUnit;
import recoder.util.FileCollector;
import recoderanalyser.analyse.Status.StatusType;
import recoderanalyser.solidsxtree.SolidSXRootNode;

public class Analyser 
{
    public static SolidSXRootNode analyseCode(String sourcePath, long revision, String libFiles, String libFolders, boolean ignore)
    {
        System.out.println("STATUS_MESSAGE: RecoderAnalyser started");

        //Project settings
        CrossReferenceServiceConfiguration crsc = new CrossReferenceServiceConfiguration();
        ErrorHandler errorHandler = new ErrorHandler();
        errorHandler.setIgnoreErrors(ignore);
        crsc.getProjectSettings().setProperty(PropertyNames.INPUT_PATH, generateInputPath(sourcePath, libFiles, libFolders));
        crsc.getProjectSettings().ensureSystemClassesAreInPath();
        crsc.getProjectSettings().setErrorHandler(errorHandler);

        //Get SourceFileRepository
        SourceFileRepository sfr = crsc.getSourceFileRepository();

        //Add listeners to show progress
        sfr.addProgressListener(new Status(StatusType.PARSING));
        crsc.getSourceInfo().addProgressListener(new Status(StatusType.REFERENCES));

        //Get compilationUnit list
        List<CompilationUnit> units = null;
        try
        {
            units = sfr.getAllCompilationUnitsFromPath();
        }
        catch(ParserException pe) 
        {
            System.out.println(String.format("Parse error: %s", pe.getMessage()));
            System.exit(1);
        }

        //Update model
        crsc.getChangeHistory().updateModel();

        //Create tree
        SolidSXTree tree = new SolidSXTree();
        return tree.generateTree(crsc, units, revision);
    }
    
    private final static FilenameFilter JAR_FILENAME_FILTER = new FilenameFilter() 
    {
        @Override
        public boolean accept(File dir, String name) 
        {
            return (name.toLowerCase().endsWith(".jar") || name.toLowerCase().endsWith(".class"));
        }
    };
    
    private static String generateInputPath(String root, String libFiles, String libFolders) 
    {   
        StringBuilder inputPath = new StringBuilder(root);
        
        if(libFolders != null)
        {
            for(String folderPath : libFolders.split(";")) 
            {
                List<String> files = readJarFiles(folderPath);
                for(String file : files) 
                {
                    inputPath.append(";").append(file);
                }
            }
        }
        
        if(libFiles != null) 
        {
            inputPath.append(";").append(libFiles);
        }
        return inputPath.toString();
    }
    
    private static List<String> readJarFiles(String dir) 
    {
	FileCollector col = new FileCollector(dir);
	List<String> list = new ArrayList<String>();
	while (col.next(JAR_FILENAME_FILTER)) 
        {
	    list.add(col.getFile().getPath());
	}
	return list;
    }
}
