package recoderanalyser.analyse;

import java.util.List;

import recoder.abstraction.Constructor;
import recoder.abstraction.Field;
import recoder.abstraction.Method;
import recoder.abstraction.Type;
import recoder.bytecode.ClassFile;
import recoder.java.Expression;
import recoder.java.SourceVisitor;
import recoder.java.declaration.TypeDeclaration;
import recoder.java.expression.operator.New;
import recoder.java.reference.FieldReference;
import recoder.java.reference.MethodReference;
import recoder.java.reference.TypeReference;
import recoder.service.SourceInfo;
import recoderanalyser.solidsxtree.SolidSXMethodNode;
import recoderanalyser.solidsxtree.SolidSXReference.ReferenceType;

public class SolidSXMethodVisitor extends SourceVisitor
{
    private SolidSXMethodNode methodNode;
    private SourceInfo si;
    private SolidSXTree solidSXTree;
    
    SolidSXMethodVisitor(SolidSXMethodNode methodNode, SourceInfo si, SolidSXTree solidSXTree)
    {
        this.methodNode = methodNode;
        this.si = si;
        this.solidSXTree = solidSXTree;
    }
    
    private String getFullMethodPath(Method m)
    {
        StringBuilder fullPath = new StringBuilder(m.getFullName().substring(0, m.getFullName().lastIndexOf(".")+1));
        fullPath.append(solidSXTree.getMethodSignature(m));
        return fullPath.toString();
    }

    @Override
    public void visitTypeReference(TypeReference x)
    {
        if(!x.getName().equals("void"))
        {
            Type t = si.getType(x);
            solidSXTree.addReference(methodNode, t.getFullName(), t, ReferenceType.Uses);
        }
    }

    @Override
    public void visitFieldReference(FieldReference x) 
    {
        Field f = si.getField(x);
        if(f != null)
        {
            solidSXTree.addReference(methodNode, f.getFullName(), f, ReferenceType.Access);
        }
    }

    @Override
    public void visitMethodReference(MethodReference x) 
    {
        Method m = si.getMethod(x);
        solidSXTree.addReference(methodNode, getFullMethodPath(m), m, ReferenceType.Call);
    }

    @Override
    public void visitNew(New x) 
    {  
        Type type = si.getType(x);        
        List<? extends Constructor> constructorList = null;
        if(type instanceof ClassFile)
        {
            constructorList = si.getConstructors((ClassFile)type);
        }
        else if(type instanceof TypeDeclaration)
        {
            constructorList = si.getConstructors((TypeDeclaration)type);
        }
        
        //Check if there is is a contructors 
        if(constructorList != null)
        {
            Constructor foundConstructor = null;
            
            //If there is one constructor, we need this one
            if(constructorList.size() == 1)
            {
                foundConstructor = constructorList.get(0);
            }
            //If there are more constructors, we have to select the right one
            else if(constructorList.size() > 1)
            {
                //Create the constructor path
                StringBuilder constructorPath = new StringBuilder(type.getFullName());
                constructorPath.append(".").append(type.getName()).append("(");

                //Check if there are arguments
                if(x.getArguments() != null)
                {
                    //Add the argument types
                    for(Expression e : x.getArguments())
                    {
                        String typeName = (si.getType(e) == null) ? "<unknown type>" : si.getType(e).getName();
                        constructorPath.append(typeName);
                        constructorPath.append(", ");
                    }
                    
                    //Remove the last ", "
                    constructorPath.delete(constructorPath.length()-2, constructorPath.length());
                }
                constructorPath.append(")");
                
                //Compare the constructor path to find the right constructor
                for(Constructor c : constructorList)
                {
                    if(getFullMethodPath(c).equals(constructorPath.toString()))
                    {
                        foundConstructor = c;
                        break;
                    }
                }
            }
            
            //Create reference if there was a constructor found
            if(foundConstructor != null)
            {
                solidSXTree.addReference(methodNode, getFullMethodPath(foundConstructor), foundConstructor, ReferenceType.Call);
            }
        }
    }
}
