package recoderanalyser.solidsxtree;

import java.io.Serializable;
import java.util.Collections;
import java.util.ArrayList;
import java.util.List;

public class SolidSXNode implements Comparable<SolidSXNode>, Serializable
{
    public enum NodeType {Root, Package, Class, Constructor, Method, Field}
    
    private long ID;
    private String name;
    protected NodeType type;
    private SolidSXNode parent;
    private List<SolidSXNode> children;
    private List<SolidSXReference> references;
    private List<Long> revisions;
    
    public SolidSXNode(String name, long revision, NodeType type)
    {
        this.ID = -1;
        this.name = name;
        this.parent = null;
        this.type = type;
        
        children = new ArrayList<SolidSXNode>();
        references = new ArrayList<SolidSXReference>();
        revisions = new ArrayList<Long>();
        revisions.add(revision);
    }
    
    public long getID()
    {
        return ID;
    }
    
    public void setID(long ID)
    {
        this.ID = ID;
    }
    
    public String getName()
    {
        return name;
    }
    
    public NodeType getType()
    {
        return type;
    }
    
    public SolidSXNode getParent()
    {
        return parent;
    }
    
    public void setParent(SolidSXNode node)
    {
        this.parent = node;
    }
    
    public List<SolidSXNode> getChildren()
    {
        return children;
    }
    
    public List<SolidSXReference> getReferences()
    {
        return references;
    }
    
    public List<Long> getRevisions()
    {
        return revisions;
    }
    
    public boolean hasRevision(long revision)
    {
        return revisions.contains(revision);
    }
    
    public SolidSXNode getChildByName(String name)
    {
        for(SolidSXNode child : children)
        {
            if(child.name.equals(name))
            {
                return child;
            }
        }
        return null;
    }
    
    public SolidSXNode addChildNode(SolidSXNode childNode)
    {
        childNode.setParent(this);
        if(children.add(childNode))
        {
            return childNode;
        }
        return null;
    }
    
    public boolean addReference(SolidSXReference ref)
    {
        if(!references.contains(ref))
        {
            return references.add(ref);
        }
        return false;
    }
    
    public boolean addRevision(long revision)
    {
        if(!revisions.contains(revision))
        {
            return revisions.add(revision);
        }
        return false;
    }
    
    public SolidSXNode getChildByName(String name, long revision)
    {
        for(SolidSXNode node : this.children)
        {
            if(node.name.equals(name) && node.hasRevision(revision))
            {
                return node;
            }
        }
        return null;
    }
    
    public void sortChildren()
    {
        Collections.sort(children);
    }
    
    @Override
    public int compareTo(SolidSXNode compareObject)
    {
        int compareType = this.type.compareTo(compareObject.getType());
        if(compareType == 0)
        {
            return this.name.compareToIgnoreCase(compareObject.getName());
        }
        else
        {
            return compareType;
        }
    }

    @Override
    public boolean equals(Object obj) 
    {
        if(!(obj instanceof SolidSXNode))
        {
            return false;
        }
        else
        {
            SolidSXNode inputObj = (SolidSXNode)obj;
            return (this.name.equals(inputObj.name) && this.type.equals(inputObj.type) && this.references.equals(inputObj.references));
        }
    }

    @Override
    public int hashCode() 
    {
        int hash = 7;
        hash = 67 * hash + (this.name != null ? this.name.hashCode() : 0);
        hash = 67 * hash + (this.type != null ? this.type.hashCode() : 0);
        hash = 67 * hash + (this.references != null ? this.references.hashCode() : 0);
        return hash;
    }
}
