﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ICSharpCode.NRefactory.CSharp;
using CSharpMetrics.Scopes;

namespace CSharpMetrics.Visitors
{
    class MethodVisitor : Visitor
    {
        protected Method Method { get; set; }

        protected int Depth { get; set; }

        protected int MaxDepth { get; set; }

        public MethodVisitor(Method method)
        {
            Method = method;
        }

        protected override void VisitedChildren(AstNode node)
        {
            Method.LOC = LOC;
            Method.CLOC = CLOC;
            Method.MaxDepth = MaxDepth;
        }

        public override void VisitParameterDeclaration(ParameterDeclaration parameterDeclaration)
        {
            Method.AddParam(parameterDeclaration.Type.ToString());

            base.VisitParameterDeclaration(parameterDeclaration);
        }

        public override void VisitInvocationExpression(InvocationExpression invocationExpression)
        {
            Method.Invocations++;

            base.VisitInvocationExpression(invocationExpression);
        }

        public override void VisitObjectCreateExpression(ObjectCreateExpression objectCreateExpression)
        {
            Method.Invocations++;
            
            base.VisitObjectCreateExpression(objectCreateExpression);
        }

        public override void VisitIfElseStatement(IfElseStatement ifElseStatement)
        {
            IncreaseComplexity();

            CaptureDepth(() => ifElseStatement.TrueStatement.AcceptVisitor(this));

            if (ifElseStatement.FalseStatement is IfElseStatement)
            {
                ifElseStatement.FalseStatement.AcceptVisitor(this);
            }
            else
            {
                CaptureDepth(() => ifElseStatement.FalseStatement.AcceptVisitor(this));
            }
        }

        public override void VisitConditionalExpression(ConditionalExpression conditionalExpression)
        {
            IncreaseComplexity();

            base.VisitConditionalExpression(conditionalExpression);
        }

        public override void VisitForeachStatement(ForeachStatement foreachStatement)
        {
            IncreaseComplexity();

            CaptureDepth(() => base.VisitForeachStatement(foreachStatement));
        }

        public override void VisitForStatement(ForStatement forStatement)
        {
            IncreaseComplexity();

            CaptureDepth(() => base.VisitForStatement(forStatement));
        }

        public override void VisitWhileStatement(WhileStatement whileStatement)
        {
            IncreaseComplexity();

            CaptureDepth(() => base.VisitWhileStatement(whileStatement));
        }

        public override void VisitDoWhileStatement(DoWhileStatement doWhileStatement)
        {
            IncreaseComplexity();

            CaptureDepth(() => base.VisitDoWhileStatement(doWhileStatement));
        }

        public override void VisitSwitchSection(SwitchSection switchSection)
        {
            IncreaseComplexity();

            CaptureDepth(() => base.VisitSwitchSection(switchSection));
        }

        public override void VisitReturnStatement(ReturnStatement returnStatement)
        {
            // Don't count return statements in first level
            if (returnStatement.Parent.Parent.Role != Roles.TypeMemberRole)
            {
                IncreaseComplexity();
            }

            base.VisitReturnStatement(returnStatement);
        }

        public override void VisitThrowStatement(ThrowStatement throwStatement)
        {
            IncreaseComplexity();

            base.VisitThrowStatement(throwStatement);
        }

        public override void VisitGotoStatement(GotoStatement gotoStatement)
        {
            IncreaseComplexity();

            base.VisitGotoStatement(gotoStatement);
        }

        public override void VisitGotoCaseStatement(GotoCaseStatement gotoCaseStatement)
        {
            IncreaseComplexity();

            base.VisitGotoCaseStatement(gotoCaseStatement);
        }

        public override void VisitGotoDefaultStatement(GotoDefaultStatement gotoDefaultStatement)
        {
            IncreaseComplexity();

            base.VisitGotoDefaultStatement(gotoDefaultStatement);
        }

        public override void VisitCheckedStatement(CheckedStatement checkedStatement)
        {
            CaptureDepth(() => base.VisitCheckedStatement(checkedStatement));
        }

        public override void VisitUncheckedStatement(UncheckedStatement uncheckedStatement)
        {
            CaptureDepth(() => base.VisitUncheckedStatement(uncheckedStatement));
        }

        public override void VisitUsingStatement(UsingStatement usingStatement)
        {
            CaptureDepth(() => base.VisitUsingStatement(usingStatement));
        }

        public override void VisitTryCatchStatement(TryCatchStatement tryCatchStatement)
        {
            CaptureDepth(() => tryCatchStatement.TryBlock.AcceptVisitor(this));

            base.VisitTryCatchStatement(tryCatchStatement);
        }

        public override void VisitCatchClause(CatchClause catchClause)
        {
            CaptureDepth(() => base.VisitCatchClause(catchClause));
        }

        protected void IncreaseComplexity()
        {
            Method.Complexity++;
        }

        protected void CaptureDepth(Action action)
        {
            IncreaseDepth();
            action();
            DecreaseDepth();
        }

        protected void IncreaseDepth()
        {
            Depth++;
        }

        protected void DecreaseDepth()
        {
            MaxDepth = Math.Max(MaxDepth, Depth);

            Depth--;
        }
    }
}
