//
// SetterDoesNotUseValueParameterTests.cs
//
// Author:
//       Simon Lindgren <simon.n.lindgren@gmail.com>
//
// Copyright (c) 2012 Simon Lindgren
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
using System.Collections.Generic;
using System.Linq;
using ICSharpCode.NRefactory.TypeSystem;
using ICSharpCode.NRefactory.CSharp.Resolver;
using System.Threading;
using ICSharpCode.NRefactory.Semantics;

namespace ICSharpCode.NRefactory.CSharp.Refactoring
{
	[IssueDescription("The value parameter is not used in a context where is should be",
	       Description = "Warns about property or indexer setters and event adders or removers that do not use the value parameter.",
	       Category = IssueCategories.CodeQualityIssues,
	       Severity = Severity.Warning)]
	public class ValueParameterUnusedIssue : ICodeIssueProvider
	{
		public IEnumerable<CodeIssue> GetIssues(BaseRefactoringContext context)
		{
			return new GatherVisitor(context, this).GetIssues();
		}
		
		class GatherVisitor : GatherVisitorBase
		{
			public GatherVisitor(BaseRefactoringContext context, ValueParameterUnusedIssue inspector) : base (context)
			{
			}

			public override void VisitIndexerDeclaration(IndexerDeclaration indexerDeclaration)
			{
				FindIssuesInNode(indexerDeclaration.Setter, indexerDeclaration.Setter.Body);
			}

			public override void VisitPropertyDeclaration(PropertyDeclaration propertyDeclaration)
			{
				FindIssuesInNode(propertyDeclaration.Setter, propertyDeclaration.Setter.Body);
			}

			public override void VisitCustomEventDeclaration(CustomEventDeclaration eventDeclaration)
			{
				var addAccessor = eventDeclaration.AddAccessor;
				var removeAccessor = eventDeclaration.RemoveAccessor;
				// don't warn on empty custom events
				if (addAccessor.Body.Statements.Count == 0 && removeAccessor.Body.Statements.Count == 0)
					return;
				FindIssuesInNode(addAccessor, addAccessor.Body, "add accessor");
				FindIssuesInNode(removeAccessor, removeAccessor.Body, "remove accessor");
			}

			void FindIssuesInNode(AstNode anchor, BlockStatement body, string accessorName = "setter")
			{
				if (!IsEligible(body))
					return;

				var localResolveResult = ctx.GetResolverStateBefore(body)
					.LookupSimpleNameOrTypeName("value", new List<IType>(), NameLookupMode.Expression) as LocalResolveResult; 
				if (localResolveResult == null)
					return;

				bool referenceFound = false;
				foreach (var result in ctx.FindReferences (body, localResolveResult.Variable)) {
					var node = result.Node;
					if (node.StartLocation >= body.StartLocation && node.EndLocation <= body.EndLocation) {
						referenceFound = true;
						break;
					}
				}

				if(!referenceFound)
					AddIssue(anchor, ctx.TranslateString("The " + accessorName + " does not use the 'value' parameter"));
			}

			static bool IsEligible(BlockStatement body)
			{
				if (body == null || body.IsNull)
					return false;
				if (body.Statements.FirstOrNullObject() is ThrowStatement)
					return false;
				return true;
			}
		}
	}
}
