﻿using System;
using SQLite;
using System.Text.RegularExpressions;
using CSharpDeps.Scopes;
using Microsoft.TeamFoundation.VersionControl.Client;

namespace SolidTA.SCMImporter.Commands.Deps
{
    abstract class Processor
    {
        public SQLiteConnection Database { get; private set; }

        public SCMServer Server { get; private set; }

        public ChangeSet Changeset { get; private set; }

        protected Models.Deps.Solution Solution;

        protected Models.Deps.Project Project;

        protected Models.Deps.File File;

        public Processor Init(Processor processor)
        {
            return processor.Init(Database, Server, Changeset, Solution, Project, File);
        }

        public Processor Init(SQLiteConnection database, SCMServer Server, ChangeSet changeset, Models.Deps.Solution solution = null, Models.Deps.Project project = null, Models.Deps.File file = null)
        {
            Database = database;
            this.Server = Server;
            Changeset = changeset;
            Solution = solution;
            Project = project;
            File = file;

            return this;
        }

        protected string RepositoryPathFromLocalPath(string path)
        {
            return Regex.Replace(path.Replace("\\", "/"), ".*/extra/TFS/", "$/", RegexOptions.Compiled);
        }

        protected void Try(Action action)
        {
            try
            {
                action();
            }
            catch (Exception e)
            {
                Console.WriteLine(e);
            }
        }

        public virtual void PreProcess()
        {

        }

        public virtual void PostProcess()
        {

        }

        public virtual void Process(Solution solution)
        {
            foreach (var project in solution.Projects)
            {
                Process(solution, project);
            }
        }

        public virtual void Process(Solution solution, Project project)
        {
            foreach (var file in project.Files)
            {
                Process(solution, project, file);
            }
        }

        public virtual void Process(Solution solution, Project project, File file)
        {
            foreach (var ns in file.Namespaces.Values)
            {
                Process(file, ns);
            }
        }

        public virtual void Process(File file, Namespace ns)
        {
            foreach (var klass in ns.Classes.Values)
            {
                Process(file, ns, klass);
            }
        }

        public virtual void Process(File file, Namespace ns, Class klass)
        {
            foreach (var method in klass.Methods)
            {
                Process(file, ns, klass, method);
            }
        }

        public virtual void Process(File file, Namespace ns, Class klass, Method method)
        {
            
        }
    }
}
