﻿using System;
using System.Collections.Generic;
using static SolidTA.SCMImporter.SCMServer;

namespace SolidTA.SCMImporter.Commands
{
    class ImportFileTree : TfsCommand
    {
        public ImportFileTree()
        {
            IsCommand("files", "Imports the file tree into the database.");

            AllowsAnyAdditionalArguments("[timestamp, ...] (UNIX timestamps of snapshots to include in the result)");
        }

        protected override void Execute(string[] snapshots, string[] input)
        {
            var files = FilesInSnapshots(snapshots);
            var time = DateTime.UtcNow;

            ExitIfCancelled();

            // Add to the list of files the current list of files from the repository.
            Item[] items = Server.GetItems(Path, VersionSpec.Latest, RecursionType.Full, DeletedState.NonDeleted, Item.ItemType.File, false);

            AddItemsTo(files, items, time);

            InitProgress(files.Count);

            // Foreign key support has to be disabled during insert or update statements, deletion triggers are always run
            Database.DisableForeignKeys();

            Database.BeginTransaction();

            // Now that we have a complete map of when a file was last updated, we can update the
            // database. A file is considered valid if its time is equal to the current because that
            // means that the item was last updated in the GetItems call for the latest version.
            foreach (KeyValuePair<string, DateTime> pair in files)
            {
                ExitIfCancelled();

                Database.InsertOrUpdate(new Models.File
                {
                    Path = pair.Key,
                    Type = pair.Value.CompareTo(time) >= 0 ? Models.File.Types.Valid : Models.File.Types.Obsolete,
                    Time = pair.Value,
                });

                UpdateProgress();
            }

            Database.Commit();

            Database.EnableForeignKeys();

            // Every file which does not have any of the times we handled this import have to be
            // deleted from the database as they are no longer included in this set of snapshots.

            // NOTE: This has been disabled because further analyzation results may be coupled
            // to these files, we do not want to delete those results.
#if FALSE
            IEnumerable<DateTime> timestamps = files.Values.Distinct();

            Database.Table<Models.File>().Where(f => !timestamps.Contains(f.Time)).Delete();
#endif
        }

        protected Dictionary<string, DateTime> FilesInSnapshots(params string[] snapshots)
        {
            int timestamp;
            var files = new Dictionary<string, DateTime>();

            // For every timstamp (snapshot) we will get all files and store them
            // in the files dictionary with item paths and the snapshot time.
            foreach (var snapshot in snapshots)
            {
                ExitIfCancelled();

                if (int.TryParse(snapshot, out timestamp))
                {
                    var date = Date.FromUnix(timestamp);

                    VersionSpec startFrom = Server.GetFirstVersionAfter(date);
                    var items = Server.GetItems(Path, startFrom, RecursionType.Full, DeletedState.NonDeleted, Item.ItemType.File, false);

                    AddItemsTo(files, items, date);
                }
                else
                {
                    Console.WriteLine("Skipped invalid snapshot timestamp: {0}.", snapshot);
                }
            }

            return files;
        }

        protected void AddItemsTo(Dictionary<string, DateTime> files, Item[] items, DateTime date)
        {
            foreach (var item in items)
            {
                if (item.itemType.Equals(Item.ItemType.File))
                {
                    files[item.ServerItem] = date;
                }
            }
        }
    }
}
