#*****************************************************************************************
# File			: pextAuthor.py
# Description	: An example plugin for visualizing acquired file size (in lines of text) information.
#				  This should be used in combination with the mextLOT.py plugin when developing custom plugins.
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

#------------------------------------------------------------ Global imports ---
import wx
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GLU import *

import pysqlite2.dbapi2 as sqlite

#------------------------------------------------------------- Local imports ---
import sta_globals
import sta_plugin
import sta_utils
import sta_gl_list
import sta_gl_mixerunit

	# EXPLANATION
	# This is an example and code template for building metric calculators

	# EXPLANATION
	# Markers like this are used in the source code of this template to indicate the places
	# that need to be modified when implementing a new metric calculator. The used markers are:
	# EXPLANATION	: no action required. This is only information to help understanding the code
	# TEMPLATE 		: the marked code needs to be updated with calculator specific data.
	# CODE			: the marked code needs to be replaced with calculator specific one.
	# MULTI			: the marked code needs to be augmented only when more tables are generated.

#===============================================================================
#																Authors plugin
#===============================================================================
class Plugin(sta_plugin.Plugin):

	def __init__(self,*args, **kwds):
		sta_plugin.Plugin.__init__(self,*args, **kwds)

		# TEMPLATE
		#--------------------------------
		self.sClass					= 'pextLOT'			# Name of the plugin file
		self.sMetricID				= 'LOT'				# Identifier to store the visualized metric internaly.
														# Like the name of the plugin file, it needs to be unique
		self.sName					= 'Lines of text'   # Name to display in a list
		self.bHasMetricValue		= True				# True for numeric metrics
														# (as opposed to categorical ones, such as Author ID)
		# MULTI
		#--------------------------------
		# Actions:
		#	- declare more required tables here


		self.sTable				   	= 'M_SS2008021201'  # Unique ID of the table where metric is stored
														# Needs to be the same with the one used by the metric calculator
														# If more tables are involved they need be declared here (e.g., self.sTable2,...)
		self.lTables.append(self.sTable)				# Append to this list all tables that are used

		#--------------------------------
		# END MULTI
		# END TEMPLATE

		#----------------------------------------------------- Specific processing

		self.iLevel					= 80	# The interval level
		self.iLevelFactor			= 10	# Controls the amount of increase on the slider(CFG)
		self.iIntervals				= 10	# Number of intervals (CFG)
		self.iNrLevels				= 100	# Number of slider levels (CFG)

		self.iIntervalSize			= (self.iLevel * self.iLevelFactor) / self.iIntervals

		self.lLocalColors		   = []		# Color map (CFG)

#-------------------------------------------------------------------------------
#																 Filter GUI
#-------------------------------------------------------------------------------

#----------------------------------------------------------------- Build GUI ---
	def buildFilterGUI(self,pMasterGUI):

		self.cGUIMaster = pMasterGUI

		self.cSizer = wx.BoxSizer(wx.VERTICAL)
		l_cConfigSizer = wx.BoxSizer(wx.HORIZONTAL)

		l_cLabelSizer = wx.BoxSizer(wx.HORIZONTAL)
		l_sLabel = wx.StaticText(pMasterGUI.project_filters, -1, '  '+self.sName)
		self.g_txtMetric = wx.StaticText(pMasterGUI.project_filters, -1, style = wx.ALIGN_RIGHT )
		l_cLabelSizer.Add(l_sLabel, 0, wx.ADJUST_MINSIZE, 0)
		l_cLabelSizer.Add(self.g_txtMetric, 0, wx.ALIGN_RIGHT, 0)

		self.cGUIMode = wx.ComboBox(pMasterGUI.project_filters, -1, choices=[], style=wx.CB_DROPDOWN|wx.CB_READONLY)

		# CODE
		#--------------------------------
		# A color encoder can handle more variations/aggregations of the same metric
		# For example in the case of complexitry metrics, one can visualize the absolute value (MCB)
		# or the value reported to a certain number of lines of code (MCB / 100 LOC).
		# Actions:
		#	- Append the name of more variations/aggregations of the metric here
		#		The implementation part will come later in the code

		self.cGUIMode.Append('LOT')
		#--------------------------------
		# END CODE

		self.cGUIMode.SetSelection(0)

		self.cGUIList = sta_gl_list.wxColorCheckListBox(self.sName,self.sClass,pMasterGUI.project_filters, style=wx.BORDER_SUNKEN|wx.LB_MULTIPLE|wx.LB_SORT)
		self.cGUIList.lObservers.append(self)

		for i in range(self.iIntervals):
			self.lLocalColors.append(sta_utils.getRainbowColor(float(i+1)/self.iIntervals))
		self.cGUIList.SetData(self.lLocalColors)

		self.cGUIScale = wx.Slider(pMasterGUI.project_filters, -1, self.iLevel, 1, self.iNrLevels)
		self.cGUIConfig = wx.Button(pMasterGUI.project_filters, -1, 'cfg')
		self.cGUIConfig.Disable()
		self.cGUIScale.SetOwnBackgroundColour(wx.Colour(150,150,180))

		self.cGUIList.AddMetric("# files",self.MetricIntervalFrequency)

		self.cGUIList.Bind(wx.EVT_LISTBOX,self.cbListSelect)
		self.cGUIScale.Bind(wx.EVT_SLIDER,self.cbLevelSelect)
		self.cGUIMode.Bind(wx.EVT_COMBOBOX,self.cbModeSelect)

		self.cGUIList.SetMinSize((140,-1))
		self.cGUIConfig.SetMinSize((30,15))
		self.cGUIScale.SetMinSize((-1,15))

		self.cSizer.Add(l_cLabelSizer, 0, wx.EXPAND, 0)
		self.cSizer.Add(self.cGUIMode, 0, wx.EXPAND, 0)
		self.cSizer.Add(self.cGUIList, 1, wx.EXPAND|wx.ADJUST_MINSIZE, 0)
		l_cConfigSizer.Add(self.cGUIScale, 1, wx.EXPAND, 0)
		l_cConfigSizer.Add(self.cGUIConfig, 0, wx.ADJUST_MINSIZE, 0)
		self.cSizer.Add(l_cConfigSizer, 0, wx.EXPAND, 0)

		pMasterGUI.cListsSizer_Project.Add(self.cSizer, 0, wx.EXPAND, 0)

		self.showFilterGUI(False)
		self.doBindings()

#-------------------------------------------------------------------------------
#														 Filter GUI callback
#-------------------------------------------------------------------------------

	def cbListSelect_aux(self, files):

		if len(self.lIdx) == 0:
			for i in files:
				for j in i.lRevs:
					j.lValues[self.sClass] = int(j.lValues[self.sMetricID] / self.iIntervalSize)
					if j.lValues[self.sClass] < 0:
						j.lValues[self.sClass] = -1
					if j.lValues[self.sClass] >= self.iIntervals:
						j.lValues[self.sClass] = self.iIntervals-1
		else:
			for i in files:
				for j in i.lRevs:
					l_iIdx = int(j.lValues[self.sMetricID] / self.iIntervalSize)
					if l_iIdx >= self.iIntervals:
						l_iIdx = self.iIntervals-1
					j.lValues[self.sClass] = -1
					if l_iIdx in self.lIdx:
						j.lValues[self.sClass] = l_iIdx

	def getSelectedIndexes(self):
		l_lsSize = self.cGUIList.GetSelections()
		self.clearSelection()
		for i in range(self.iIntervals):
			if i<self.iIntervals-1:
				l_sItem = str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1)
			else:
				l_sItem = str(i*self.iIntervalSize)+' - ...'
			if l_sItem in l_lsSize:
				self.lIdx.append(i)
		self.bSelected = len(self.lIdx)>0

	def cbLevelSelect(self,evt):

		self.unloadMetric()
		self.iLevel = self.cGUIScale.GetValue()
		self.iIntervalSize = (self.iLevel * self.iLevelFactor) / self.iIntervals

		self.loadMetric()
		sta_plugin.Plugin.updateCanvases(self)

	def cbModeSelect(self,evt):
		self.cGUIList.DeselectAll()
		self.reloadMetric()
		sta_plugin.Plugin.updateCanvases(self)

#-------------------------------------------------------------------------------
#															Filter GUI metrics
#-------------------------------------------------------------------------------
#--------------------------------------------------------- Number of commits ---
	def MetricIntervalFrequency(self,sName):

		l_iLevel = -1
		for i in range(self.iIntervals):
			if i<self.iIntervals-1:
				l_sItem = str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1)
			else:
				l_sItem = str(i*self.iIntervalSize)+' - ...'
			if l_sItem == sName:
				l_iLevel = i
				break

		l_iNr = 0
		if (l_iLevel >=0):
			if (l_iLevel <self.iIntervals-1):
				for i in sta_globals.lSelectedFiles:
					for j in i.lRevs:
						if (j.lValues[self.sMetricID] >= l_iLevel*self.iIntervalSize) and (j.lValues[self.sMetricID] < (l_iLevel+1)*self.iIntervalSize):
							l_iNr = l_iNr+1
							break
			else:
				for i in sta_globals.lSelectedFiles:
					for j in i.lRevs:
						if (j.lValues[self.sMetricID] >= l_iLevel*self.iIntervalSize):
							l_iNr = l_iNr+1
							break
		return l_iNr

#-------------------------------------------------------------------------------
#																Data management
#-------------------------------------------------------------------------------
#--------------------------------------------------------------- Load metric ---
	def loadMetric(self):

		l_sDBPath = sta_globals.cSTAPrj.storagePath()+"/history.db"

		if not sta_utils.isValidDB(l_sDBPath):
			print 'DATA: no history information found'
			return

		cDB = sqlite.connect(l_sDBPath)
		cCursorDB = cDB.cursor()

		# Check if all required information is available
		for i in self.lTables:
			if not sta_utils.isValidTable(cCursorDB,i):
				print 'DATA: metric information found'
				cCursorDB.close()
				cDB.close()
				return

		self.clearSelection()
		l_lMetric = {}

		# CODE
		# TEMPLATE
		# MULTI
		#--------------------------------
		# This specifies what will be consdered as metric value by the color encoder
		# If there are more variations possible, tyhey have to be specified here
		# If therere are more tables involved they have to be corelated here

		#------ Lines of code ---
		if (self.cGUIMode.GetSelection() == 0):

			sCmd = "select ID,LOT from "+self.sTable
			cCursorDB.execute(sCmd)
			for i in cCursorDB:
				l_lMetric[i[0]]=i[1]

		#--- Other variations ---
		else:
			pass

		#--------------------------------
		# END MULTI
		# END TEMPLATE
		# END CODE

		#------------------------------------------ General ---
		# spread the actual value over a number of intervals
		# These intervals will be displayed by the color encoder
		for i in sta_globals.lFiles:
			for j in i.lRevs:
				try:
					l_iMetricValue = l_lMetric[j.iDBid]
					j.lValues[self.sMetricID]=l_iMetricValue
					j.lValues[self.sClass]= int(l_iMetricValue / self.iIntervalSize)
					if j.lValues[self.sClass] < 0:
						j.lValues[self.sClass] = -1
					if j.lValues[self.sClass] >= self.iIntervals:
						j.lValues[self.sClass] = self.iIntervals-1
				except:
					j.lValues[self.sMetricID]= -1
					j.lValues[self.sClass]= -1
		l_lMetric = {}

		# Display intervals
		for i in range(self.iIntervals):
			if (i < self.iIntervals-1):
				self.cGUIList.Append(str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1),i)
			else:
				self.cGUIList.Append(str(i*self.iIntervalSize)+' - ...',i)
			self.lData.append(i)
		#----------------------------------------------------

		cCursorDB.close()
		cDB.close()

#------------------------------------------------------------- Unload metric ---
	def unloadMetric(self):

		# CODE
		#-----------------------------------
		# Additional clean up code goes here (none in this case)
		pass
		#-----------------------------------
		# END CODE

		sta_plugin.Plugin.unloadMetric(self)

#===============================================================================
#												Authors filter mixer unit
#===============================================================================
class MixerUnit(sta_gl_mixerunit.MixerUnit):

	def __init__(self, *args, **kwds):
		sta_gl_mixerunit.MixerUnit.__init__(self, *args, **kwds)

		# TEMPLATE
		#-------------------------------------
		# Choose the type of metric to be supported (0 for version, 1 for file)
		# Choose the type of representation (0 for squares, 1 for horizontal lines)

		self.iAttrType		= 0		# A version metric
		self.iDrawType		= 1		# A squares representation

		#-------------------------------------
		# END TEMPLATE