#*****************************************************************************************
# Name			: TFS Work Items
# File			: pextTFSWorkItems.py
# Description	: A plugin for extracting version associated Work Items from TFS
# Version		: 1.0
# Copyright Joost Koehoorn
#*****************************************************************************************

#------------------------------------------------------------ Global imports ---
import wx
import wx.lib.mixins.listctrl as listmix
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GLU import *

import pysqlite2.dbapi2 as sqlite

import types

#------------------------------------------------------------- Local imports ---
import sta_globals
import sta_plugin
import sta_utils
import sta_gl_list
import sta_gl_mixerunit

class CWorkItemsTable(wx.Dialog, listmix.ColumnSorterMixin):
   def __init__(self, *args, **kwds):

      kwds["style"] = wx.CAPTION|wx.RESIZE_BORDER|wx.CLOSE_BOX
      wx.Dialog.__init__(self, *args, **kwds)

      self.SetTitle("Work Items")

      self.cTable = wx.ListCtrl(self, -1, style=wx.LC_REPORT);
      self.cTable.InsertColumn(0, "")
      self.cTable.InsertColumn(1, "ID")
      self.cTable.InsertColumn(2, "Date")
      self.cTable.InsertColumn(3, "Created By")
      self.cTable.InsertColumn(4, "Title")
      self.cTable.InsertColumn(5, "Description")
      self.cTable.InsertColumn(6, "State")
      self.cTable.InsertColumn(7, "Reason")
      self.cTable.InsertColumn(8, "Type")
      self.cTable.SetColumnWidth(0, 0)
      self.cTable.SetColumnWidth(1, 75)
      self.cTable.SetColumnWidth(2, 75)
      self.cTable.SetColumnWidth(3, 150)
      self.cTable.SetColumnWidth(4, 175)
      self.cTable.SetColumnWidth(5, 150)
      self.cTable.SetColumnWidth(6, 75)
      self.cTable.SetColumnWidth(7, 100)
      self.cTable.SetColumnWidth(8, 70)
      self.cTable.SetMinSize((300, 100))
      self.SetSize((900, 250))

      listmix.ColumnSorterMixin.__init__(self, 8)
      self.Bind(wx.EVT_LIST_COL_CLICK, self.OnColClick, self.cTable)

      sizer = wx.BoxSizer(wx.VERTICAL)
      sizer.Add(self.cTable, 1, wx.EXPAND|wx.ALL|wx.FIXED_MINSIZE, 0)
      self.SetSizer(sizer)
      sizer.Fit(self)
      sizer.SetSizeHints(self)
      self.Layout()

      self.iCurrentColorColumn = -1
      self.itemDataMap = {}
      self.itemIndexMap = {}
      self.highlighted = []

   def Clear(self):
      self.cTable.DeleteAllItems()
      self.itemDataMap = {}
      self.itemIndexMap = {}

   def InsertItem(self, index, row):
      date = sta_utils.getDate(row[1])
      self.itemDataMap[index] = ('', row[0], date, row[2], row[3], row[4], row[5], row[6], row[7])
      self.itemIndexMap[row[0]] = index

      self.cTable.InsertStringItem(index, '')
      self.cTable.SetStringItem(index, 1, str(row[0]))
      self.cTable.SetStringItem(index, 2, date)
      self.cTable.SetStringItem(index, 3, row[2])
      self.cTable.SetStringItem(index, 4, row[3])
      self.cTable.SetStringItem(index, 5, row[4])
      self.cTable.SetStringItem(index, 6, row[5])
      self.cTable.SetStringItem(index, 7, row[6])
      self.cTable.SetStringItem(index, 8, row[7])
      self.cTable.SetItemData(index, index)

   def CreateColorBitmaps(self, colors):
      bitmaps = []

      dc = wx.MemoryDC()
      dc.SetPen(wx.BLACK_PEN)
      dc.SetBackground(wx.Brush("white"))

      for color in colors:
         bmp = wx.EmptyBitmap(13, 12)
         brush = wx.Brush(wx.Colour(color[0] * 255, color[1] * 255, color[2] * 255))

         dc.SelectObject(bmp)
         dc.Clear()
         dc.SetBrush(brush)
         dc.DrawRectangle(1, 0, 12, 12)
         bitmaps.append(bmp)

      dc.SetBrush(wx.NullBrush)
      dc.SetBackground(wx.NullBrush)
      dc.SetPen(wx.NullPen)
      dc.SelectObject(wx.NullBitmap)

      return bitmaps

   def Recolor(self, indices):
      # Not initialized yet
      if self.iCurrentColorColumn == -1:
         return

      for index, values in self.itemDataMap.iteritems():
         value = str(values[self.iCurrentColorColumn])
         try:
            self.cTable.SetItemColumnImage(index, self.iCurrentColorColumn, self.imageitem[indices[value]])
         except:
            pass

   def SetColors(self, column, colors):
      if self.iCurrentColorColumn != -1:
         for index, values in self.itemDataMap.iteritems():
            self.cTable.SetItemColumnImage(index, self.iCurrentColorColumn, -1)

      self.iCurrentColorColumn = column

      self.imageitem = []
      bitmaps = self.CreateColorBitmaps(colors)

      imagelist = wx.ImageList(13, 12)

      for bitmap in bitmaps:
         self.imageitem.append(imagelist.Add(bitmap))

      self.cTable.AssignImageList(imagelist, wx.IMAGE_LIST_SMALL)

   def UnhighlightAll(self):
      for item in self.highlighted:
         try:
            index = self.cTable.FindItemData(-1, self.itemIndexMap[item])
            if index != wx.NOT_FOUND:
               self.cTable.SetItemBackgroundColour(index, "white")
         except:
            pass
      self.highlighted = []

   def HighlightWorkItem(self,item):
      try:
         index = self.cTable.FindItemData(-1, self.itemIndexMap[item])
         if index != wx.NOT_FOUND:
            self.highlighted.append(item)
            self.cTable.SetItemBackgroundColour(index, "yellow")
            self.cTable.EnsureVisible(index)
      except Exception as e:
         print e

   def GetListCtrl(self):
      return self.cTable

   def OnColClick(self, event):
      event.Skip()

#===============================================================================
#                                                         TFS Work Items plugin
#===============================================================================
class Plugin(sta_plugin.Plugin):

   def __init__(self,*args, **kwds):
      sta_plugin.Plugin.__init__(self,*args, **kwds)
      self.sClass                  = 'pextTFSWorkItems' # Name of the plugin file
      self.sName                   = 'TFS Work Items'     # Name to display in a list
      self.sVersion                = '1.0'         # Version
      self.sMetricID               = 'TFSWorkItems'

      self.sTable1                 = 'D_JK2013051501'  # Name of the table where metric is stored
      self.sTable2                 = 'M_JK2013051501'  # Name of the table where metric is stored
      self.lTables.append(self.sTable1)			# Append to this list all tables that are used
      self.lTables.append(self.sTable2)

      #----------------------------------------------------- Specific processing

      self.dMapping               = {}  # Type values
      self.lBugTypeColors         = []  # Colors for GUI list items
      self.multipleLabel          = "Multiple"

#-------------------------------------------------------------------------------
#                                                                 Filter GUI
#-------------------------------------------------------------------------------

#----------------------------------------------------------------- Build GUI ---
   def buildFilterGUI(self,pMasterGUI):

      self.cGUIMaster = pMasterGUI

      self.cSizer = wx.BoxSizer(wx.VERTICAL)
      l_cConfigSizer = wx.BoxSizer(wx.HORIZONTAL)

      l_cLabelSizer = wx.BoxSizer(wx.HORIZONTAL)
      l_sLabel = wx.StaticText(pMasterGUI.project_filters, -1, '  '+self.sName)
      self.g_txtMetric = wx.StaticText(pMasterGUI.project_filters, -1, style = wx.ALIGN_RIGHT )
      l_cLabelSizer.Add(l_sLabel, 0, wx.ADJUST_MINSIZE, 0)
      l_cLabelSizer.Add(self.g_txtMetric, 0, wx.ALIGN_RIGHT, 0)

      self.cGUIMode = wx.ComboBox(pMasterGUI.project_filters, -1, choices=[], style=wx.CB_DROPDOWN|wx.CB_READONLY)
      self.cGUIMode.Append('ID')
      self.cGUIMode.Append('Created By')
      self.cGUIMode.Append('State')
      self.cGUIMode.Append('Reason')
      self.cGUIMode.Append('Type')
      self.cGUIMode.SetSelection(4)

      self.cGUIList = sta_gl_list.wxColorCheckListBox(self.sName,self.sClass,pMasterGUI.project_filters, style=wx.BORDER_SUNKEN|wx.LB_MULTIPLE|wx.LB_SORT)
      self.cGUIList.lObservers.append(self)

      self.cGUIList.AddMetric("# versions",self.MetricNumberOfVersions)

      self.g_butShowTable     = wx.Button(pMasterGUI.project_filters, -1, "Show Table")

      self.cWorkItemsTable = CWorkItemsTable(sta_globals.GUI.mainFrame)

      self.cGUIList.Bind(wx.EVT_LISTBOX,self.cbListSelect)
      self.cGUIMode.Bind(wx.EVT_COMBOBOX,self.cbModeSelect)
      self.g_butShowTable.Bind(wx.EVT_BUTTON, self.cbToggleTable)

      self.cSizer.Add(l_cLabelSizer, 0, wx.EXPAND, 0)
      self.cSizer.Add(self.cGUIMode, 0, wx.EXPAND, 0)
      self.cSizer.Add(self.cGUIList, 1, wx.EXPAND|wx.ADJUST_MINSIZE)
      self.cSizer.Add(self.g_butShowTable, 0, wx.EXPAND|wx.ADJUST_MINSIZE, 0)
      pMasterGUI.cListsSizer_Project.Add(self.cSizer, 0, wx.EXPAND, 0)

      self.cGUIList.SetMinSize((160,-1))

      self.showFilterGUI(False)
      self.doBindings()

#-------------------------------------------------------------------------------
#						         Filter GUI callback
#-------------------------------------------------------------------------------

   def cbToggleTable(self, event):
      if self.cWorkItemsTable.IsShown():
         self.cWorkItemsTable.Show(False)
         self.g_butShowTable.SetLabel("Show Table")
      else:
         self.SetTableColors()
         self.cWorkItemsTable.Show()
         self.ReloadTable()
         self.g_butShowTable.SetLabel("Hide Table")

   def cbListSelect_aux(self, files):

      if len(self.lIdx) == 0:
        for i in files:
           for j in i.lRevs:
               j.lValues[self.sClass] = self.CommitClass(j)
      else:
        for i in files:
           for j in i.lRevs:
               iClass = self.CommitClass(j)
               j.lValues[self.sClass] = iClass if iClass in self.lIdx else -1

   def CommitClass(self,commit):
      try:
         l_lWorkItems = commit.lValues[self.sMetricID]
         if len(l_lWorkItems) == 0:
            return -1
         elif len(l_lWorkItems) > 1:
            # Figure out if the work items are in different classes
            classes = {}
            for item in l_lWorkItems:
               classes[item] = True

            # If they are, set to Multiple class with index 0 or just use the classes
            if len(classes) > 1:
               return 0
            else:
               return l_lWorkItems[0]
         else:
            return l_lWorkItems[0]
      except:
         return -1

   def getSelectedIndexes(self):
      l_lsType = self.cGUIList.GetSelections()
      self.clearSelection()
      for i in l_lsType:
         self.lIdx.append(self.dMapping[i])
      self.bSelected = len(self.lIdx)>0

   def cbModeSelect(self,evt):
      self.cGUIList.DeselectAll()
      self.unloadMetric()
      self.loadMetric()
      self.SetTableColors()

      sta_plugin.Plugin.updateCanvases(self)

   def Refresh(self):
      sta_plugin.Plugin.Refresh(self)
      self.RecolorTable()

   def SetTableColors(self):
      #----------------------------------- ID ---
      if (self.cGUIMode.GetSelection() == 0):
         column = 1

      #----------------------------------- Author ---
      elif (self.cGUIMode.GetSelection() == 1):
         column = 3

      #----------------------------------- State ---
      elif (self.cGUIMode.GetSelection() == 2):
         column = 6

      #----------------------------------- Reason ---
      elif (self.cGUIMode.GetSelection() == 3):
         column = 7

      #----------------------------------- Type ---
      elif (self.cGUIMode.GetSelection() == 4):
         column = 8

      self.cWorkItemsTable.SetColors(column, self.cGUIList.lColors)

      self.RecolorTable()

   def RecolorTable(self):
      indices = {}
      for value, index in self.cGUIList.dItems.iteritems():
         indices[value] = self.cGUIList.findColor(index)

      self.cWorkItemsTable.Recolor(indices)

   def UnhighlightAll(self):
      self.cWorkItemsTable.UnhighlightAll()

   def HighlightWorkItems(self,commit):
      if self.cWorkItemsTable.IsShown():
         self.cWorkItemsTable.UnhighlightAll()
         for workItem in commit.lValues[self.sMetricID + "_IDs"]:
            self.cWorkItemsTable.HighlightWorkItem(workItem)

#-------------------------------------------------------------------------------
#						            Filter GUI metrics
#-------------------------------------------------------------------------------
#--------------------------------------------------------- Number of commits ---
   def MetricNumberOfVersions(self,p_sName):

      l_iValue = self.dMapping[p_sName]
      l_iNr = 0

      for i in sta_globals.lSelectedFiles:
         for j in i.lRevs:
            if l_iValue in j.lValues[self.sMetricID]:
               l_iNr += 1
      return l_iNr

#-------------------------------------------------------------------------------
#                                                                Data management
#-------------------------------------------------------------------------------
#--------------------------------------------------------------- Load metric ---

   def ReloadTable(self):
      if not self.cWorkItemsTable.IsShown():
         return

      l_sDBPath = sta_globals.cSTAPrj.storagePath()+"/history.db"

      #---------------------------- Check existance of data ---
      if not sta_utils.isValidDB(l_sDBPath):
         print 'DATA: no history information found'
         return

      cDB = sqlite.connect(l_sDBPath)
      cCursorDB = cDB.cursor()

      if not sta_utils.isValidTable(cCursorDB,self.sTable1):
         print 'DATA: no work item information found (1)'
         cCursorDB.close()
         cDB.close()
         return

      if not sta_utils.isValidTable(cCursorDB,self.sTable2):
         print 'DATA: no work item information found (2)'
         cCursorDB.close()
         cDB.close()
         return

      fileIDs = ','.join([str(f.iDBid) for f in sta_globals.lSelectedFiles])

      sCmd = ("select ID, CreationDate, Author, Title, Description, State, Reason, Type from %(items)s " + \
             "where ID in (" + \
               "select distinct %(pivot)s.WorkItem from %(pivot)s "
               "join Versions on Versions.Name = %(pivot)s.Version " + \
               ("where Versions.File in (%(ids)s)" if len(sta_globals.lSelectedFiles) > 0 else "") + \
             ") order by ID") % \
             { "items": self.sTable1, "pivot": self.sTable2, "ids": fileIDs }

      cCursorDB.execute(sCmd)

      self.cWorkItemsTable.Clear()

      index = 0
      for row in cCursorDB:
         #for i in range(1000):
         self.cWorkItemsTable.InsertItem(index, row)
         index += 1

      cCursorDB.close()
      cDB.close()

      self.RecolorTable()

   def unloadMetric(self):
      for i in sta_globals.lFiles:
         for j in i.lRevs:
            try:
               del j.lValues[self.sMetricID + "_IDs"]
            except:
               pass

      sta_plugin.Plugin.unloadMetric(self)

   def loadMetric(self):

      l_sDBPath = sta_globals.cSTAPrj.storagePath()+"/history.db"

      #---------------------------- Check existance of data ---
      if not sta_utils.isValidDB(l_sDBPath):
         print 'DATA: no history information found'
         return

      cDB = sqlite.connect(l_sDBPath)
      cCursorDB = cDB.cursor()

      if not sta_utils.isValidTable(cCursorDB,self.sTable1):
         print 'DATA: no work item information found (1)'
         cCursorDB.close()
         cDB.close()
         return

      if not sta_utils.isValidTable(cCursorDB,self.sTable2):
         print 'DATA: no work item information found (2)'
         cCursorDB.close()
         cDB.close()
         return

      #----------------------------------- ID ---
      if (self.cGUIMode.GetSelection() == 0):
         l_sColumn = "ID"

      #----------------------------------- Author   ---
      elif (self.cGUIMode.GetSelection() == 1):
         l_sColumn = "Author"

      #----------------------------------- State ---
      elif (self.cGUIMode.GetSelection() == 2):
         l_sColumn = "State"

      #----------------------------------- Reason ---
      elif (self.cGUIMode.GetSelection() == 3):
         l_sColumn = "Reason"

      #----------------------------------- Type ---
      elif (self.cGUIMode.GetSelection() == 4):
         l_sColumn = "Type"

      # Select all possible values first
      sCmd = "select distinct cast(%(column)s as text) from %(items)s" % { "items": self.sTable1, "column": l_sColumn }

      cCursorDB.execute(sCmd)

      #--------------- Append info to GUI ---
      self.dMapping = {self.multipleLabel: 0}
      lOrdered = [self.multipleLabel]
      self.lData.append(0)
      l_iIdx = 1
      for i in cCursorDB:
         self.lData.append(l_iIdx)
         self.dMapping[i[0]] = l_iIdx
         lOrdered.append(i[0])
         l_iIdx += 1

      self.lBugTypeColors = []
      l_iLen = l_iIdx-1
      self.lBugTypeColors.append((0.2,0.2,0.2)) # First index is dark gray, for Multiple types
      for i in range(l_iLen):
         self.lBugTypeColors.append(sta_utils.getRainbowColor(float(i+1)/l_iLen))
      for i in range(l_iLen,10):
         self.lBugTypeColors.append((0.75,0.75,0.75))
      self.cGUIList.SetData(self.lBugTypeColors)

      for l_sValue in lOrdered:
         self.cGUIList.Append(l_sValue,self.dMapping[l_sValue])

      # Then select the version associated with the value
      sCmd = ("select distinct %(pivot)s.Version, cast(%(items)s.%(column)s as text), %(items)s.ID from %(items)s " + \
             "join %(pivot)s on %(items)s.ID = %(pivot)s.WorkItem") % \
             { "items": self.sTable1, "pivot": self.sTable2, "column": l_sColumn }

      cCursorDB.execute(sCmd)

      l_lMetric = {}
      for i in cCursorDB:
         if i[0] not in l_lMetric:
            l_lMetric[i[0]] = [[], []]

         l_iIdx = self.dMapping[i[1]]
         l_lMetric[i[0]][0].append(l_iIdx) # Add class index to first metric index
         l_lMetric[i[0]][1].append(i[2]) # Add WorkItem ID to second metric index

      #-------------------------------------- Assign values ---
      self.clearSelection()

      bMixed = False
      for i in sta_globals.lFiles:
         for j in i.lRevs:
            try:
               l_lWorkItems = l_lMetric[int(j.sID)][0]
               j.lValues[self.sMetricID] = l_lWorkItems
               j.lValues[self.sMetricID + "_IDs"] = l_lMetric[int(j.sID)][1]
               j.lValues[self.sClass] = self.CommitClass(j)

               if j.lValues[self.sClass] == 0:
                  bMixed = True
            except:
               j.lValues[self.sMetricID]= []
               j.lValues[self.sMetricID + "_IDs"] = []
               j.lValues[self.sClass]= -1
      l_lMetric = {}

      if not bMixed:
         self.cGUIList.Delete(self.multipleLabel)

         # Also delete the color from the list to avoid it from appearing in the mixerunits
         del self.cGUIList.lColors[0]
         for item,idx in self.cGUIList.dColors.iteritems():
            self.cGUIList.dColors[item] = idx - 1

      cCursorDB.close()
      cDB.close()

      self.RecolorTable()

#===============================================================================
#						Authors filter mixer unit
#===============================================================================
class MixerUnit(sta_gl_mixerunit.MixerUnit):

	def __init__(self, *args, **kwds):
		sta_gl_mixerunit.MixerUnit.__init__(self, *args, **kwds)

		self.iAttrType		= 0
		self.iDrawType		= 0
