#*****************************************************************************************
# Name			: Evolution Similarity
# File			: cextEvolution.py
# Description	: A plugin for evolution based clustering of files 
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

import sta_globals
import os
import os.path
import math
import sta_clusters
import sys
import wx

#//////////////////////////////////////////////////////////////////////////////////////////
#																Clusters Engine Class
#//////////////////////////////////////////////////////////////////////////////////////////

class CClusterEngineDlg(wx.Dialog):
	def __init__(self,p_cParent,*args, **kwds):
		
		# Data
		self.cParent		= p_cParent
		
		wx.Dialog.__init__(self, *args, **kwds)
		
		# Commit radius in hours
		self.gRadius = wx.SpinCtrl(self, -1, "", (30, 50))
		self.gRadius.SetRange(1,100)
		self.gRadius.SetValue(int(self.cParent.iCommitRadius/3600))
		l_gRadius = wx.StaticText(self, -1, 'Commit time window (hours)')
		
		# Reference value (commits)
		self.gCommits = wx.SpinCtrl(self, -1, "", (30, 50))
		self.gCommits.SetRange(1,100)
		self.gCommits.SetValue(int(1.0/self.cParent.fRefValue))
		l_gCommits = wx.StaticText(self, -1, 'Reference joint modifications (commits)')
		
		# Cluster size (files)
		self.gFiles = wx.SpinCtrl(self, -1, "", (30, 50))
		self.gFiles.SetRange(1,100)
		self.gFiles.SetValue(self.cParent.fRefClusterSize)
		l_gFiles = wx.StaticText(self, -1, 'Reference cluster size (files)')
		
		szSettings = wx.FlexGridSizer(3,2,5,5)
		szSettings.AddGrowableCol(0)
		szSettings.Add(l_gRadius,1,wx.EXPAND|wx.LEFT,10)
		szSettings.Add(self.gRadius,0,wx.FIXED_MINSIZE|wx.RIGHT,10)
		szSettings.Add(l_gCommits,1,wx.EXPAND|wx.LEFT,10)
		szSettings.Add(self.gCommits,0,wx.FIXED_MINSIZE|wx.RIGHT,10)
		szSettings.Add(l_gFiles,1,wx.EXPAND|wx.LEFT,10)
		szSettings.Add(self.gFiles,0,wx.FIXED_MINSIZE|wx.RIGHT,10)
		
		#--- Ok/Cancel buttons
		l_gButOk 		= wx.Button(self,-1,'Apply')
		l_gButCancel 	= wx.Button(self,-1,'Cancel')
		szButtons = wx.BoxSizer(wx.HORIZONTAL)
		szButtons.Add(l_gButOk,0,wx.FIXED_MINSIZE|wx.ALL,2) 
		szButtons.Add(l_gButCancel,0,wx.FIXED_MINSIZE|wx.ALL,2) 
		
		#--- Main sizer
		szMain			= wx.BoxSizer(wx.VERTICAL)
		szMain.Add(szSettings,1, wx.EXPAND |wx.ALL, 10)
		szMain.Add(szButtons,0, wx.ALIGN_CENTER |wx.ALL, 10)
		
		self.SetSizer(szMain)
		self.SetAutoLayout(True)
		szMain.Fit(self)
		self.Layout()
		
		# Callbacks
		self.Bind(wx.EVT_BUTTON, self.OnOk, l_gButOk)
		self.Bind(wx.EVT_BUTTON, self.OnCancel, l_gButCancel)
		
	def OnOk(self,evt):
		self.cParent.iCommitRadius		= self.gRadius.GetValue()*3600
		self.cParent.fRefValue			= 1.0/self.gCommits.GetValue()
		self.cParent.fRefClusterSize	= self.gFiles.GetValue()
		self.Show(False)
	
	def OnCancel(self,evt):
		self.gRadius.SetValue(int(self.cParent.iCommitRadius/3600))
		self.gCommits.SetValue(int(1.0/self.cParent.fRefValue))
		self.gFiles.SetValue(self.cParent.fRefClusterSize)
		self.Show(False)
		

class CClusterEngine(sta_clusters.CClusterEngine):
#==========================================================================================
#																		Construction
#==========================================================================================
	def __init__(self,*args,**kwds):
		sta_clusters.CClusterEngine.__init__(self,*args,**kwds)
		self.sName					= 'Evolution clustering'
		self.sID					= 'Evolution'
		self.bHasGUI				= True
		#---
		self.fRefValue				= 1.00				# Reference value for distance (1/nr common commits)
		self.fRefClusterSize		= 2.00				# Reference value for cluster size (min nr files in a cluster)
		self.iCommitRadius			= 2*3600			# Neighbouring interval (2 hours)
		self.lPropagationClusters	= []				# Used to calculate the cluster score
	
	def SetGUI(self,p_cGUI):
		sta_clusters.CClusterEngine.SetGUI(self,p_cGUI)
		if (self.cParentGUI != None):
			self.cGUI			= CClusterEngineDlg(self,self.cParentGUI,-1,'Evolution clustering settings')
#==========================================================================================
#																				Compute
#==========================================================================================
	
	#------------------------------------------------------------------------------
	#																Check skip
	#------------------------------------------------------------------------------
	def pCheckSkip(self,p_cCluster):
		
		if (p_cCluster.iType == 1):
			return
		
		l_iCommits	= 0
		l_cFile 	= p_cCluster.pNode
		for i in l_cFile.lRevs:
			if (i.iTime >= self.iStartTime) and (i.iTime <= self.iEndTime):
				l_iCommits += 1
		
		if (l_iCommits < int(1.0/self.fRefValue)):
			p_cCluster.bSkip = True
		else:
			p_cCluster.bSkip = False
	
	#------------------------------------------------------------------------------
	#																Compact cluster
	#------------------------------------------------------------------------------
	def pCompactCluster(self,p_cCluster):
		l_bSwitch = True
		while l_bSwitch:
			l_bSwitch = False
			l_lRevs = []
			l_iLen = len(p_cCluster.lData)-1
			i = 0
			while i < l_iLen:
				if (p_cCluster.lData[i+1] - p_cCluster.lData[i]) <= self.iCommitRadius:
					l_bSwitch = True
					l_fTime = (p_cCluster.lData[i+1] + p_cCluster.lData[i]) / 2
					l_lRevs.append(l_fTime)
					i = i+1
				else:
					l_lRevs.append( p_cCluster.lData[i])
				i = i+1
			if l_bSwitch:
				p_cCluster.lData = l_lRevs
		
	#------------------------------------------------------------------------------
	#														Construct new cluster  
	#------------------------------------------------------------------------------
	def pConstructCluster(self,p_cFile):
		l_cNew				 = sta_clusters.CCluster()
		l_cNew.iType		 = 0
		
		for j in p_cFile.lRevs:
			if (j.iTime >= self.iStartTime) and (j.iTime <= self.iEndTime):
				l_cNew.lData.append(j.iTime)
			
		l_cNew.pNode = p_cFile
		self.pCompactCluster(l_cNew)
		self.pCheckSkip(l_cNew)
		self.pAppendElement(l_cNew)
		
	#------------------------------------------------------------------------------
	#																Merge clusters 
	#------------------------------------------------------------------------------
	def pMergeClusters(self,p_fMeasure,p_iP1,p_iP2):
	
		l_cNew					= sta_clusters.CCluster()
		l_cNew.pElement1		= self.dClusters[p_iP1]
		l_cNew.pElement2		= self.dClusters[p_iP2]
		l_cNew.fMeasure			= p_fMeasure
		l_cNew.iFileSize		= l_cNew.pElement1.iFileSize + l_cNew.pElement2.iFileSize
		
		#--- combine sets ---
		l_lCommits = []		
		for i in l_cNew.pElement1.lData: 
				l_lCommits.append((1,i))
		for i in l_cNew.pElement2.lData: 
			l_lCommits.append((2,i))
		l_lCommits.sort(cmp=lambda x,y: cmp(x[1], y[1]))
		
		#--- find centers ---
		i = 0
		l_iLen = len(l_lCommits)-1
		while (i<l_iLen):
			if (l_lCommits[i][0] != l_lCommits[i+1][0]):
				l_iDistance = l_lCommits[i+1][1] - l_lCommits[i][1]
				if ( l_iDistance > sta_globals.iCommitRadius):
					pass
				elif (l_iDistance == 0):
					l_fTime   = l_lCommits[i][1]
					l_cNew.lData.append(l_fTime)
					i = i+1
				else:
					l_fTime = (l_lCommits[i+1][1] + l_lCommits[i][1]) / 2
					l_cNew.lData.append(l_fTime)
					i = i+1
			i = i+1
		
		self.pCompactCluster(l_cNew)
		return l_cNew
		
	#------------------------------------------------------------------------------
	#												 Evolution similarity distance 
	#------------------------------------------------------------------------------
	
	def pDelta(self,p_cCluster1,p_cCluster2):
		l_iDelta = 0
		l_iPoints = 0
		if (p_cCluster1 == p_cCluster2):
			return l_iDelta   
		#------------------------------------- generate combined set ---
		l_lCommits = []
		for i in p_cCluster1.lData:
			l_lCommits.append((1,i))
		for i in p_cCluster2.lData:
			l_lCommits.append((2,i))
		l_lCommits.sort(cmp=lambda x,y: cmp(x[1], y[1]))
		l_iLen = len(l_lCommits)-1
		#--------------------------------------------- compute delta ---
		i = 0
		while (i<l_iLen):
			if (l_lCommits[i][0] != l_lCommits[i+1][0]):
				l_iDistance = l_lCommits[i+1][1] - l_lCommits[i][1]
				if ( l_iDistance < self.iCommitRadius):
					l_iPoints = l_iPoints+1
					i = i+1
			i = i+1
		#---------------------------------------------------------------
		if (l_iPoints == 0):
			return sys.maxint
		
		return 1.0 / l_iPoints
	#------------------------------------------------------------------------------
	#												 Clustering score
	#------------------------------------------------------------------------------
	def GetFileList(self,p_cCluster,p_lFileList):
		if (p_cCluster.iType == 1):
			self.GetFileList(p_cCluster.pElement1,p_lFileList)
			self.GetFileList(p_cCluster.pElement2,p_lFileList)
		else:
			p_lFileList.append(p_cCluster.pNode.sPath)
	
	def pScore_aux(self,p_cCluster):
		
		if (p_cCluster.fMeasure < 1.0):
			if (p_cCluster.iFileSize >= self.fRefClusterSize):
				self.lPropagationClusters.append(p_cCluster)
		else:
			if (p_cCluster.iType == 1):
				self.pScore_aux(p_cCluster.pElement1)
				self.pScore_aux(p_cCluster.pElement2)
	
	def pScore(self):
		self.lPropagationClusters = []
		l_sResult = 'Not available' 
		if self.HasClusters():
			self.pScore_aux(self.cClusterRoot)
			l_dFolders = {}
			l_iScore = 0
			for l_cCluster in self.lPropagationClusters:
				l_iFiles = l_cCluster.iFileSize
				l_dFolders.clear()
				l_lFileList = []
				self.GetFileList(l_cCluster,l_lFileList)
				for l_sFile in l_lFileList:
					l_sFolder = os.path.dirname(l_sFile)
					try:
						l_dFolders[l_sFolder] += 1
					except:
						l_dFolders[l_sFolder] = 1
				l_iScore += l_iFiles + (l_iFiles - max(l_dFolders.values()))*len(l_dFolders.keys())
			l_sResult = '   Change propagation = %d   '%l_iScore
		return l_sResult
		
