#*****************************************************************************************
# Name			: CCCC integrator
# File			: mextCCCC.py
# Description	: A plugin for integration with the CCCC (version 3.pre84) metric calculator
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

import sta_metricgenerator
import sys
import os
import time
import subprocess
import sta_utils
import sta_globals

import pysqlite2.dbapi2 as sqlite

#===============================================================================
#														CCCC output XML parser
#===============================================================================
from xml.sax import make_parser
from xml.sax.handler import ContentHandler

class CCCCHandler(ContentHandler):

#-------------------------------------------------------------------------------
#																	   Init
#-------------------------------------------------------------------------------

	def __init__ (self):
		self.iID_1		  = 0		 # Parser position markers
		self.iID_2		  = 0		 #
		self.iID_3		  = 0		 #
		self.iID_4		  = 0		 #

		self.sValue		 = ''		# Reconstruct character values

		self.cCursorDB	  = 0		 # DB cursor
		self.iDBid		  = ''		# Version ID in the database

		self.init()

	def init(self):
		#------------------------------- Summmary
		self.iModules	   = -1		# Number of modules
		self.iMethods	   = -1		# Number of methods
		self.iLOC		   = -1		# Lines of code
		self.iCOM		   = -1		# Lines of comment
		self.iMCB		   = -1		# McCabes complexity
		self.iIF4		   = -1		# Information flow
		self.iIF4v		  = -1		# Information flow (visible)
		self.iIF4c		  = -1		# Information flow (concrete)

		self.init_module1()
		self.init_module2()
		self.init_module3()
		self.tdModule	   = {}		# Modules info table

		self.init_function()
		self.tdFunction	 = {}		# Functions info table
		self.tdFunctionItem = {}		# Functions info table

	def init_module1(self):
		#------------------------------- Per module 1
		self.sM_Name	= ''		# Module name
		self.iM_LOC		= -1		# Lines of code
		self.iM_MTD		= -1		# Number of methods
		self.iM_MCB		= -1		# McCabes complexity
		self.iM_COM		= -1		# Lines of comment

	def init_module2(self):
		#------------------------------- Per module 2
		self.iM_WMC1	= -1		# Weighted methods per class (all)
		self.iM_WMCv	= -1		# Weighted methods per class (visible)
		self.iM_DIT		= -1		# Depth of inheritance tree
		self.iM_NOC		= -1		# Number of children
		self.iM_CBO		= -1		# Coupling between objects

	def init_module3(self):
		#------------------------------- Per module 3
		self.iM_FIv		= -1		# Fan in visible
		self.iM_FIc		= -1		# Fan in concrete
		self.iM_FI		= -1		# Fan in
		self.iM_FOv		= -1		# Fan out visible
		self.iM_FOc		= -1		# Fan out concrete
		self.iM_FO		= -1		# Fan out
		self.iM_IF4v	= -1		# Information flow visible
		self.iM_IF4c	= -1		# Information flow concrete
		self.iM_IF4		= -1		# Information flow

	def init_function(self):
		#------------------------------- Per function
		#self.sF_ModuleName  = ''	   # Module name
		self.sF_FunctionName = ''	   # Function name
		self.iF_LOC		= -1		# Lines of code
		self.iF_MCB		= -1		# McCabes complexity
		self.iF_COM		= -1		# Lines of comment

		#--------------------------------

	def setData(self,p_cCtrl,p_cCursorDB,p_iDBid):
		self.cCursorDB  = p_cCursorDB
		self.iDBid	  	= p_iDBid
		self.cParent	= p_cCtrl

	def getIntValue(self,p_sValue):
		try:
			l_iValue = int(p_sValue)
		except:
			l_iValue = -1
		return l_iValue

#-------------------------------------------------------------------------------
#																 Start element
#-------------------------------------------------------------------------------
	def startElement(self, name, attrs):
		#------------------------------ first level ---
		if (self.iID_1 == 0):
			if (name == 'project_summary'):
				self.iID_1 = 1
				self.init()
			elif (name == 'procedural_summary'):
				self.iID_1 = 2
			elif (name == 'procedural_detail'):
				self.iID_1 = 3
			elif (name == 'oo_design'):
				self.iID_1 = 4
			elif (name == 'structural_summary'):
				self.iID_1 = 5
		#------------------------- project summmary ---
		elif (self.iID_1 == 1):
			if (name == 'number_of_modules'):
				self.iModules = self.getIntValue(attrs.getValue('value'))
			elif (name == 'lines_of_code'):
				self.iLOC = self.getIntValue(attrs.getValue('value'))
			elif (name == 'McCabes_cyclomatic_complexity'):
				self.iMCB = self.getIntValue(attrs.getValue('value'))
			elif (name == 'lines_of_comment'):
				self.iCOM = self.getIntValue(attrs.getValue('value'))
			elif (name == 'IF4'):
				self.iIF4 = self.getIntValue(attrs.getValue('value'))
			elif (name == 'IF4_visible'):
				self.iIF4v = self.getIntValue(attrs.getValue('value'))
			elif (name == 'IF4_concrete'):
				self.iIF4c = self.getIntValue(attrs.getValue('value'))

		#---------------------- procedural summmary ---
		elif (self.iID_1 == 2):
			if (self.iID_2 == 0):
				if (name == 'module'):
					self.iID_2 = 1
					self.init_module1()
			else:
				if (name == 'name'):
					self.iID_3 = 1
					self.sValue = ''
				elif (name == 'lines_of_code'):
					self.iM_LOC = self.getIntValue(attrs.getValue('value'))
				elif (name == 'McCabes_cyclomatic_complexity'):
					self.iM_MCB = self.getIntValue(attrs.getValue('value'))
				elif (name == 'lines_of_comment'):
					self.iM_COM = self.getIntValue(attrs.getValue('value'))

		#------------------------ procedural detail ---
		elif self.iID_1 == 3:
			if (self.iID_2 == 0):
				if (name == 'module'):
					self.iID_2 = 1
					self.tdFunctionItem = {}
					self.sF_ModuleName  = ''
			else:
				if (self.iID_3 == 0): # Not in member function
					if (name == 'name'):
						self.iID_4 = 1
						self.sValue = ''
					elif (name == 'member_function'):
						self.iID_3 = 1
						self.init_function()
				elif (self.iID_3 == 1):
					if (name == 'name'):
						self.iID_4 = 1
						self.sValue = ''
					elif (name == 'lines_of_code'):
						self.iF_LOC = self.getIntValue(attrs.getValue('value'))
					elif (name == 'McCabes_cyclomatic_complexity'):
						self.iF_MCB = self.getIntValue(attrs.getValue('value'))
					elif (name == 'lines_of_comment value'):
						self.iF_COM = self.getIntValue(attrs.getValue('value'))
		#-------------------------------- oo design ---
		elif self.iID_1 == 4:
			if (self.iID_2 == 0):
				if (name == 'module'):
					self.iID_2 = 1
					self.init_module2()
			else:
				if (name == 'name'):
					self.iID_3 = 1
					self.sValue = ''
				elif (name == 'weighted_methods_per_class_unity'):
					self.iM_WMC1 = self.getIntValue(attrs.getValue('value'))
				elif (name == 'weighted_methods_per_class_visibility'):
					self.iM_WMCv = self.getIntValue(attrs.getValue('value'))
				elif (name == 'depth_of_inheritance_tree'):
					self.iM_DIT = self.getIntValue(attrs.getValue('value'))
				elif (name == 'number_of_children'):
					self.iM_NOC = self.getIntValue(attrs.getValue('value'))
				elif (name == 'coupling_between_objects'):
					self.iM_CBO = self.getIntValue(attrs.getValue('value'))

		#---------------------- structural summmary ---
		elif self.iID_1 == 5:
			if (self.iID_2 == 0):
				if (name == 'module'):
					self.iID_2 = 1
					self.init_module3()
			else:
				if (name == 'name'):
					self.iID_3 = 1
					self.sValue = ''
				elif (name == 'fan_out_visible'):
					self.iM_FOv  = self.getIntValue(attrs.getValue('value'))
				elif (name == 'fan_out_concrete'):
					self.iM_FOc  = self.getIntValue(attrs.getValue('value'))
				elif (name == 'fan_out'):
					self.iM_FO  = self.getIntValue(attrs.getValue('value'))
				elif (name == 'fan_in_visible'):
					self.iM_FIv = self.getIntValue(attrs.getValue('value'))
				elif (name == 'fan_in_concrete'):
					self.iM_FIc = self.getIntValue(attrs.getValue('value'))
				elif (name == 'fan_in'):
					self.iM_FI = self.getIntValue(attrs.getValue('value'))
				elif (name == 'IF4_visible'):
					self.iM_IF4v = self.getIntValue(attrs.getValue('value'))
				elif (name == 'IF4_concrete'):
					self.iM_IF4c = self.getIntValue(attrs.getValue('value'))
				elif (name == 'IF4'):
					self.iM_IF4 = self.getIntValue(attrs.getValue('value'))
		#----------------------------------------------

#-------------------------------------------------------------------------------
#																	Characters
#-------------------------------------------------------------------------------
	def characters (self, ch):
		self.sValue = self.sValue + ch

#-------------------------------------------------------------------------------
#																   End element
#-------------------------------------------------------------------------------
	def endElement(self, name):
		#------------------------------------- project summary ---
		if (self.iID_1 == 1):
			if  (name == 'project_summary'):
				self.iID_1 = 0
		#---------------------------------- procedural summary ---
		elif (self.iID_1 == 2):
			if (self.iID_2 == 0):
				if  (name == 'procedural_summary'):
					self.iID_1 = 0
			elif (self.iID_2 == 1):
				if (self.iID_3 == 1):
					if (name == 'name'):
						self.sM_Name = self.sValue
						self.iID_3 = 0
				elif (self.iID_3 == 0):
					if (name == 'module'):
						self.iID_2 = 0
						self.tdModule[self.sM_Name] = [self.iM_LOC,self.iM_MCB,self.iM_COM]

		#----------------------------------- procedural detail ---
		elif (self.iID_1 == 3):
			if (self.iID_2 == 0):
				if  (name == 'procedural_detail'):
					self.iID_1 = 0
			elif (self.iID_2 == 1):
				if (self.iID_3 == 0): # Not in a memeber function
					if (self.iID_4 == 1):
						if (name == 'name'):
							self.sF_ModuleName = self.sValue
							self.iID_4 = 0
					elif (self.iID_4 == 0):
						if (name == 'module'):
							self.iID_2 = 0
							try:
								if self.tdModule[self.sF_ModuleName][0] > 0:
									self.tdFunction[self.sF_ModuleName] = self.tdFunctionItem
									self.iM_MTD = len(self.tdFunctionItem.keys())
									self.iMethods = self.iMethods + self.iM_MTD
									self.tdModule[self.sF_ModuleName].insert(1,self.iM_MTD)
							except:
								# Some modules (e.g. structure) are only detected here
								pass
				elif (self.iID_3 == 1): # In a member function
					if (self.iID_4 == 1):
						if (name == 'name'):
							self.sF_FunctionName = self.sValue
							l_iIdx = self.sF_FunctionName.find('(')
							if l_iIdx > 0:
								self.sF_FunctionName = self.sF_FunctionName[:l_iIdx]
							self.iID_4 = 0
					elif (self.iID_4 == 0):
						if (name == 'member_function'):
							self.iID_3 = 0
							self.tdFunctionItem[self.sF_FunctionName] = [self.iF_LOC,self.iF_MCB,self.iF_COM]

		#------------------------------------------- oo design ---
		elif (self.iID_1 == 4):
			if (self.iID_2 == 0):
				if  (name == 'oo_design'):
					self.iID_1 = 0
			elif (self.iID_2 == 1):
				if (self.iID_3 == 1):
					if (name == 'name'):
						self.sM_Name = self.sValue
						self.iID_3 = 0
				elif (self.iID_3 == 0):
					if (name == 'module'):
						self.iID_2 = 0
						self.tdModule[self.sM_Name].extend([self.iM_WMC1,self.iM_WMCv,self.iM_DIT,self.iM_NOC,self.iM_CBO])

		#---------------------------------- structural summary ---
		elif (self.iID_1 == 5):
			if (self.iID_2 == 0):
				if  (name == 'structural_summary'):
					self.iID_1 = 0
			elif (self.iID_2 == 1):
				if (self.iID_3 == 1):
					if (name == 'name'):
						self.sM_Name = self.sValue
						self.iID_3 = 0
				elif (self.iID_3 == 0):
					if (name == 'module'):
						self.iID_2 = 0
						self.tdModule[self.sM_Name].extend([self.iM_FOv,self.iM_FOc,self.iM_FO,self.iM_FIv,self.iM_FIc,self.iM_FI,self.iM_IF4v,self.iM_IF4c,self.iM_IF4])
		#------------------------------------ SAVE to database ---
		else:
			if (name == 'CCCC_Project'):

				"""
				# Test parsing results
				#----------------------------------------------- DB --
				fileHandle=open('test.txt','w')

				fileHandle.write('\nModules\n\n')
				for i in self.tdModule.keys():
					fileHandle.write(i+' '+str(self.tdModule[i])+'\n')

				fileHandle.write('\nFunctions\n\n')
				for i in self.tdFunction.keys():
					l_dItem = self.tdFunction[i]
					for j in l_dItem.keys():
						fileHandle.write(i+' '+j+' '+str(l_dItem[j])+'\n')
				fileHandle.close()
				#-----------------------------------------------------
				"""

				#--- Save information ---

				# Summary
				if self.iMethods >=0:
					self.iMethods = self.iMethods + 1

				if not(self.cParent.bComputed_1):
					sCmdRoot = "insert into "+self.cParent.sTableID_1+"(ID,MOD,MTD,LOC,IDEP,MCB,COM,IF4,IF4v,IF4c) values (%s,%s,%s,%s,0.%s,%s,%s,%s,%s)"
					sParam = [self.iDBid,self.iModules,self.iMethods,self.iLOC,self.iMCB,self.iCOM,self.iIF4,self.iIF4v,self.iIF4c]
					sCmd = sCmdRoot%tuple(sParam)
					self.cCursorDB.execute(sCmd)

				# Modules

				if not(self.cParent.bComputed_2):
					sCmdRoot = "insert into "+self.cParent.sTableID_2+"(ID,Module,LOC,MTD,MCB,PROP,COM,WMC1,WMCv,DIT,NOC,CBO,FIv,FIc,FI,FOv,FOc,FO,IF4v,IF4c,IF4) values (%s,'%s',%s,%s,%s,0,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s)"
					for i in self.tdModule.keys():
						if self.tdModule[i][0] > 0:
							sParam = [self.iDBid,i]
							sParam.extend(self.tdModule[i])
							sCmd = sCmdRoot%tuple(sParam)
							self.cCursorDB.execute(sCmd)

				# Methods
				if not(self.cParent.bComputed_3):
					sCmdRoot = "insert into "+self.cParent.sTableID_3+"(ID,Module,Method,SSIZE,LOC,MCB,COM,INV,STC) values (%s,'%s','%s',0,%s,%s,%s,0,0)"
					for i in self.tdFunction.keys():
						l_dItem = self.tdFunction[i]
						for j in l_dItem.keys():
							sParam = [self.iDBid,i,j]
							sParam.extend(l_dItem[j])
							sCmd = sCmdRoot%tuple(sParam)
							self.cCursorDB.execute(sCmd)

		#---------------------------------------------------------

#===============================================================================
#												  Basic metric generator plugin
#===============================================================================
class MetricGenerator(sta_metricgenerator.MetricGenerator):
	def __init__(self,*args, **kwds):
		sta_metricgenerator.MetricGenerator.__init__(self,*args, **kwds)
		self.sName				= 'Basic metrics calculator'
		self.sAuthor			= 'SolidSource'
		self.sVersion			= '1.0'
		self.sID				= 'cccc'

		self.sTableID_1			= 'M_SS2007010201'
		self.sTableName_1		= 'Basic metrics'
		self.sTableDescription_1= 'Basic code metrics.<br />Computed using the basic metrics calculator.'
		self.sTableDependencies = 'D_SS2007010201_1:D_SS2007010201_2'
		self.bComputed_1		= False

		self.sTableID_2			= 'D_SS2007010201_1'
		self.bComputed_2		= False

		self.sTableID_3			= 'D_SS2007010201_2'
		self.bComputed_3		= False


		self.sInfo		= '<strong>Description:</strong><br />'
		self.sInfo		= self.sInfo + 'Computes a number of basic code metrics (LOC, McCabe, COM, No. Methods, Fan-out, Fan-in).<br /><br />'

		self.sInfo		= self.sInfo + '<strong>Requires:</strong><br />'
		self.sInfo		= self.sInfo + 'Source code (C,C++,C#,Java).<br /><br />'

		self.sInfo		= self.sInfo + '<strong>Output metrics:</strong><br />'
		self.sInfo		= self.sInfo + 'Basic metrics<br />'
		#----------------------------------------------------------
		self.dExtensions = {'c':'c',\
						  'h':'c',\
						  'cc':'c++',\
						  'cpp':'c++',\
						  'cxx':'c++',\
						  'c++':'c++',\
						  'hh':'c++',\
						  'hpp':'c++',\
						  'hxx':'c++',\
						  'h++':'c++',\
						  'j':'java',\
						  'jav':'java',\
						  'java':'java'}
		self.sExtensionsList = ['c','cc','cpp','cxx','c++','h','hh','hpp','hxx','h++','j','jav','java']
		self.cParser = make_parser()
		self.cHandler = CCCCHandler()
		self.cParser.setContentHandler(self.cHandler)

#===============================================================================
#																	 Generate
#===============================================================================

	def generate_TFS(self, p_cCursorDB):
		l_CSharpFiles = []

		#--------------------------- Task folders ---
		l_sPath = sta_globals.cSTAPrj.storagePath()+"/taskbuffer/"+self.sID
		if not os.path.exists(l_sPath):
			os.makedirs(l_sPath)
		l_sDestPath = l_sPath+"/content.dat"

		for l_cFile in sta_globals.lSelectedFiles:

			#--- Skip unsupported files
			l_sFile,l_sExtension = os.path.splitext(l_cFile.sPath)
			if (l_sExtension != ''):
				l_sExtension = l_sExtension[1:]
			l_sExtension = l_sExtension.lower()

			# C# analyzer is built into SolidTFSImporter, they are processed in one go
			if l_sExtension == 'cs':
				l_CSharpFiles.append(l_cFile)
				continue

			if not((len(self.sExtensionsList)==0) or (l_sExtension in self.sExtensionsList)):
				sta_globals.GUI.MainThreadExec('self.cbGeneratorProgressUpdate()')
				continue

			zipFile = sta_globals.cSTAPrj.storagePath()+"/extra/files/"+str(l_cFile.iDBid)+".zip"
			if not os.path.exists(zipFile):
				sta_globals.GUI.MainThreadExec('self.cbGeneratorProgressUpdate()')
				continue

			try:
				versions = sta_utils.listArchive(p_sSrcArchive)
			except:
				sta_globals.GUI.MainThreadExec('self.cbGeneratorProgressUpdate()')
				continue

			for version in versions:
				#--- Test whether metric is computed
				sCmd = "select ID from "+self.sTableID_1+" where ID="+version
				p_cCursorDB.execute(sCmd)
				self.bComputed_1 = (p_cCursorDB.fetchone() != None)

				sCmd = "select ID from "+self.sTableID_2+" where ID="+version
				p_cCursorDB.execute(sCmd)
				self.bComputed_2 = (p_cCursorDB.fetchone() != None)

				sCmd = "select ID from "+self.sTableID_3+" where ID="+version
				p_cCursorDB.execute(sCmd)
				self.bComputed_3 = (p_cCursorDB.fetchone() != None)

				if (self.bComputed_1) and (self.bComputed_2) and (self.bComputed_3):
					continue

				#--- Compute and save metrics
				try:
					l_sContent = sta_utils.readFile(zipFile,version)
				except:
					sta_utils.dprint("%s: Could not read %s <%s>\n"%(self.sID, l_cFile.sPath, version))
					continue

				fileHandle = open(l_sDestPath,"w")
				fileHandle.write(l_sContent)
				fileHandle.close()

				l_lCmd = ['./support/cccc/cccc']
				l_lCmd.append('--lang='+self.dExtensions[l_sExtension])
				l_lCmd.append('--outdir='+l_sPath)
				l_lCmd.append('--report_mask=sorpP')
				l_lCmd.append(l_sDestPath)

				l_cProcess = subprocess.call(l_lCmd)
				if l_cProcess == 0:
					self.cHandler.setData(self,p_cCursorDB,int(version))
					l_sXMLFile = l_sPath+'/cccc.xml'
					fileHandle=open(l_sXMLFile,"r")
					self.cParser.parse(fileHandle)
					fileHandle.close()

			#--- Progress update
			sta_globals.GUI.MainThreadExec('self.cbGeneratorProgressUpdate()')

		return l_CSharpFiles

	#---------------------------------------------- Auxiliary entry function ---
	def generate_aux(self,p_sSrcArchive,p_iTime,p_cCursorDB):

		#--------------------------- Task folders ---
		l_sPath = sta_globals.cSTAPrj.storagePath()+"/taskbuffer/"+self.sID
		if not os.path.exists(l_sPath):
			os.makedirs(l_sPath)
		l_sDestPath = l_sPath+"/content.dat"

		l_lFiles = sta_utils.listArchive(p_sSrcArchive)

		for l_cFile in sta_globals.lSelectedFiles:

			sta_globals.GUI.MainThreadExec('self.logCommand(" - "+$STA$+"\\n")',lParams=[l_cFile.sName])

			#--- Check if cancel pressed
			if sta_globals.GUI.bMetricCancel:
				break

			#--- Skip non-existing files
			l_sFileName = str(l_cFile.iDBid)
			if (l_sFileName not in l_lFiles):
				continue

			#--- Skip unsupported files
			l_sFile,l_sExtension = os.path.splitext(l_cFile.sPath)
			if (l_sExtension != ''):
				l_sExtension = l_sExtension[1:]
			l_sExtension = l_sExtension.lower()

			if not((len(self.sExtensionsList)==0) or (l_sExtension in self.sExtensionsList)):
				continue

			#--- Get snapshot revision
			l_cCommit = None
			for l_cRev in l_cFile.lRevs:	# Find first revision before snapshot
				if (l_cRev.iTime <= p_iTime):
					l_cCommit = l_cRev
				else:
					break
			if not (l_cCommit is None):
				if (p_iTime > l_cFile.lRevs[-1].iTime) and (l_cFile.iStatus != 0):
					l_cCommit = None
			if (l_cCommit is None):
				continue

			#--- Test whether metric is computed
			sCmd = "select ID from "+self.sTableID_1+" where ID="+str(l_cCommit.iDBid)
			p_cCursorDB.execute(sCmd)
			l_iID = p_cCursorDB.fetchone()
			if (l_iID != None):
				self.bComputed_1 = True
			else:
				self.bComputed_1 = False

			sCmd = "select ID from "+self.sTableID_2+" where ID="+str(l_cCommit.iDBid)
			p_cCursorDB.execute(sCmd)
			l_iID = p_cCursorDB.fetchone()
			if (l_iID != None):
				self.bComputed_2 = True
			else:
				self.bComputed_2 = False

			sCmd = "select ID from "+self.sTableID_3+" where ID="+str(l_cCommit.iDBid)
			p_cCursorDB.execute(sCmd)
			l_iID = p_cCursorDB.fetchone()
			if (l_iID != None):
				self.bComputed_3 = True
			else:
				self.bComputed_3 = False

			if (self.bComputed_1) and (self.bComputed_2) and (self.bComputed_3):
				continue

			#--- Compute and save metrics
			try:
				l_sContent = sta_utils.readFile(p_sSrcArchive,l_sFileName)
			except:
				sta_utils.dprint("%s: Could not read %s <%s>\n"%(self.sID,l_cFile.sPath,l_sFileName))
				continue

			fileHandle=open(l_sDestPath,"w")
			fileHandle.write(l_sContent)
			fileHandle.close()

			l_lCmd = ['./support/cccc/cccc']
			l_lCmd.append('--lang='+self.dExtensions[l_sExtension])
			l_lCmd.append('--outdir='+l_sPath)
			l_lCmd.append('--report_mask=sorpP')
			l_lCmd.append(l_sDestPath)

			l_cProcess = subprocess.call(l_lCmd)
			if l_cProcess == 0:
				self.cHandler.setData(self,p_cCursorDB,l_cCommit.iDBid)
				l_sXMLFile = l_sPath+'/cccc.xml'
				fileHandle=open(l_sXMLFile,"r")
				self.cParser.parse(fileHandle)
				fileHandle.close()

			#--- Progress update
			sta_globals.GUI.MainThreadExec('self.cbGeneratorProgressUpdate()')

	#---------------------------------------------------------------- Main entry ---
	def generate(self,p_sDBPath):

		if not sta_utils.isValidDB(p_sDBPath):
			sta_globals.GUI.MainThreadExec('self.logCommand(" ERROR: no history information found\\n")')
			sta_globals.GUI.MainThreadExec('self.postGenerator()')
			return

		cDB = sqlite.connect(p_sDBPath)
		cCursorDB = cDB.cursor()
		#------------------------------------------ Prepare database ---
		if not sta_utils.isValidTable(cCursorDB,self.sTableID_1):
			sCmd = "create table "+self.sTableID_1+"(ID integer primary key not null,"\
				+ "MOD integer,"\
				+ "MTD integer,"\
				+ "LOC integer,"\
				+ "MCB integer,"\
				+ "INV integer,"\
				+ "COM integer,"\
				+ "IDEP integer,"\
				+ "IF4 integer,"\
				+ "IF4v integer,"\
				+ "IF4c integer,"\
				+ "foreign key(ID) references Versions(ID) on delete cascade)"
			cCursorDB.execute(sCmd)

			sCmdRoot = "insert into Metrics(ID,Name,Type,Description,Dependencies) values ('%s','%s',1,'%s','%s')"
			sParam = [self.sTableID_1,self.sTableName_1,self.sTableDescription_1,self.sTableDependencies]
			sCmd = sCmdRoot%tuple(sParam)
			cCursorDB.execute(sCmd)

		if not sta_utils.isValidTable(cCursorDB,self.sTableID_2):
			sCmd = "create table "+self.sTableID_2+"(ID integer,"\
				+ "Module text,"\
				+ "LOC integer,"\
				+ "MTD integer,"\
				+ "MCB integer,"\
				+ "PROP integer,"\
				+ "INV integer,"\
				+ "COM integer,"\
				+ "WMC1 integer,"\
				+ "WMCv integer,"\
				+ "DIT integer,"\
				+ "NOC integer,"\
				+ "CBO integer,"\
				+ "FIv integer,"\
				+ "FIc integer,"\
				+ "FI integer,"\
				+ "FOv integer,"\
				+ "FOc integer,"\
				+ "FO integer,"\
				+ "IF4v integer,"\
				+ "IF4c integer,"\
				+ "IF4 integer,"\
				+ "foreign key(ID) references Versions(ID) on delete cascade)"
			cCursorDB.execute(sCmd)

		if not sta_utils.isValidTable(cCursorDB,self.sTableID_3):
			sCmd = "create table "+self.sTableID_3+"(ID integer,Module text,Method text,"\
				+ "SSIZE integer,"\
				+ "LOC integer,"\
				+ "MCB integer,"\
				+ "COM integer,"\
				+ "INV integer,"\
				+ "STC integer,"\
				+ "foreign key(ID) references Versions(ID) on delete cascade)"
			cCursorDB.execute(sCmd)

		#---------------------- calculate metrics for all snapshots ---
		# TFS imports data per file version
		if sta_globals.cSTAPrj.sType == 'GIT' or sta_globals.cSTAPrj.sType == 'TFS':
			l_CSharpFiles = self.generate_TFS(cCursorDB)

		# Other types only download file contents per snapshot
		else:
			l_CSharpFiles = []
			l_sSrcArchiveBase	= sta_globals.cSTAPrj.storagePath()+"/extra/snapshots"
			for l_cSnapshot in sta_globals.cSTAPrj.lSnapshots:

				if sta_globals.GUI.bMetricCancel:
					sta_globals.GUI.bMetricCancel = False
					break

				l_sDate = time.strftime("%Y%m%d_%H%S%M",time.gmtime(l_cSnapshot[0]))
				l_sSrcArchive = l_sSrcArchiveBase+"/"+l_sDate+".zip"
				l_sSrcArchive = l_sSrcArchive.replace('\\','/')
				self.generate_aux(l_sSrcArchive,l_cSnapshot[0],cCursorDB)

				sta_globals.GUI.MainThreadExec('self.logCommand("Snapshot: "+$STA$+" generation completed\\n")',lParams=[l_cSnapshot[1]])

		#-------------------------------------------- close database ---
		cDB.commit()
		cCursorDB.close()
		cDB.close()

		if len(l_CSharpFiles) > 0:
			sta_globals.GUI.TFSImporter('metrics', inputargs=[str(f.iDBid) for f in l_CSharpFiles])

		sta_globals.GUI.MainThreadExec('self.UpdateFiltersList()')
		sta_globals.GUI.MainThreadExec('self.logCommand(" Finished: "+$STA$+" generation completed\\n")',lParams=[self.sName])
		sta_globals.GUI.MainThreadExec('self.postGenerator()')
