﻿// Copyright (c) AlphaSierraPapa for the SharpDevelop Team
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy of this
// software and associated documentation files (the "Software"), to deal in the Software
// without restriction, including without limitation the rights to use, copy, modify, merge,
// publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons
// to whom the Software is furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in all copies or
// substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
// INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE
// FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

using System;
using System.Collections.Generic;
using System.Diagnostics.Contracts;
using ICSharpCode.NRefactory.Documentation;

namespace ICSharpCode.NRefactory.TypeSystem
{
	/// <summary>
	/// Represents an unresolved entity.
	/// </summary>
	public interface IUnresolvedEntity : INamedElement, IHasAccessibility
	{
		/// <summary>
		/// Gets the entity type.
		/// </summary>
		EntityType EntityType { get; }
		
		/// <summary>
		/// Gets the complete entity region (including header+body)
		/// </summary>
		DomRegion Region { get; }
		
		/// <summary>
		/// Gets the entity body region.
		/// </summary>
		DomRegion BodyRegion { get; }
		
		/// <summary>
		/// Gets the declaring class.
		/// For members, this is the class that contains the member.
		/// For nested classes, this is the outer class. For top-level entities, this property returns null.
		/// </summary>
		IUnresolvedTypeDefinition DeclaringTypeDefinition { get; }
		
		/// <summary>
		/// Gets the parsed file in which this entity is defined.
		/// Returns null if this entity wasn't parsed from source code (e.g. loaded from a .dll with CecilLoader).
		/// </summary>
		IUnresolvedFile UnresolvedFile { get; }
		
		/// <summary>
		/// Gets the attributes on this entity.
		/// </summary>
		IList<IUnresolvedAttribute> Attributes { get; }
		
		/// <summary>
		/// Gets whether this entity is static.
		/// Returns true if either the 'static' or the 'const' modifier is set.
		/// </summary>
		bool IsStatic { get; }
		
		/// <summary>
		/// Returns whether this entity is abstract.
		/// </summary>
		/// <remarks>Static classes also count as abstract classes.</remarks>
		bool IsAbstract { get; }
		
		/// <summary>
		/// Returns whether this entity is sealed.
		/// </summary>
		/// <remarks>Static classes also count as sealed classes.</remarks>
		bool IsSealed { get; }
		
		/// <summary>
		/// Gets whether this member is declared to be shadowing another member with the same name.
		/// </summary>
		bool IsShadowing { get; }
		
		/// <summary>
		/// Gets whether this member is generated by a macro/compiler feature.
		/// </summary>
		bool IsSynthetic { get; }
	}
	
	/// <summary>
	/// Represents a resolved entity.
	/// </summary>
	public interface IEntity : ICompilationProvider, INamedElement, IHasAccessibility
	{
		/// <summary>
		/// Gets the entity type.
		/// </summary>
		EntityType EntityType { get; }
		
		/// <summary>
		/// Gets the complete entity region (including header+body)
		/// </summary>
		DomRegion Region { get; }
		
		/// <summary>
		/// Gets the entity body region.
		/// </summary>
		DomRegion BodyRegion { get; }
		
		/// <summary>
		/// Gets the declaring class.
		/// For members, this is the class that contains the member.
		/// For nested classes, this is the outer class. For top-level entities, this property returns null.
		/// </summary>
		ITypeDefinition DeclaringTypeDefinition { get; }
		
		/// <summary>
		/// Gets/Sets the declaring type (incl. type arguments, if any).
		/// This property never returns null -- for top-level entities, it returns SharedTypes.UnknownType.
		/// If this is not a specialized member, the value returned is equal to <see cref="DeclaringTypeDefinition"/>.
		/// </summary>
		IType DeclaringType { get; }
		
		/// <summary>
		/// The assembly in which this entity is defined.
		/// This property never returns null.
		/// </summary>
		IAssembly ParentAssembly { get; }
		
		/// <summary>
		/// Gets the attributes on this entity.
		/// </summary>
		IList<IAttribute> Attributes { get; }
		
		/// <summary>
		/// Gets the documentation for this entity.
		/// </summary>
		DocumentationComment Documentation { get; }
		
		/// <summary>
		/// Gets whether this entity is static.
		/// Returns true if either the 'static' or the 'const' modifier is set.
		/// </summary>
		bool IsStatic { get; }
		
		/// <summary>
		/// Returns whether this entity is abstract.
		/// </summary>
		/// <remarks>Static classes also count as abstract classes.</remarks>
		bool IsAbstract { get; }
		
		/// <summary>
		/// Returns whether this entity is sealed.
		/// </summary>
		/// <remarks>Static classes also count as sealed classes.</remarks>
		bool IsSealed { get; }
		
		/// <summary>
		/// Gets whether this member is declared to be shadowing another member with the same name.
		/// </summary>
		bool IsShadowing { get; }
		
		/// <summary>
		/// Gets whether this member is generated by a macro/compiler feature.
		/// </summary>
		bool IsSynthetic { get; }
	}
}
