﻿using System;
using Ionic.Zip;
using CSharpMetrics;
using CSharpMetrics.Scopes;

namespace SolidTA.SCMImporter.Commands
{
    class CalculateMetrics : TfsCommand
    {
        protected Analyzer Analyzer = new Analyzer();

        public CalculateMetrics()
        {
            IsCommand("metrics", "Calculates simple metrics of C# files");
        }

        public override int Run(string[] remainingArguments)
        {
            var inputArguments = ParseInputArguments();

            InterceptSignals();

            MayExit = false;

            OpenDatabase();
            CreateTablesIfNecessary();

            ExitIfCancelled();
            stopWatch.Start();
            Execute(remainingArguments, inputArguments);
            stopWatch.Stop();

            LogExecutionTime(stopWatch.Elapsed);

            return (int)ExitCode.Success;
        }

        protected override void Execute(string[] args, string[] ids)
        {
            // Do *not* initialize the progress bar, its length is already set and progress for
            // non C# files has already been reported, so we should not reset the progress bar.
            // InitProgress(ids.Length, 2);

            Database.BeginTransaction();

            foreach (var id in ids)
            {
                ExitIfCancelled();

                try
                {
                    CalculateFileMetrics(id);
                }
                catch (System.IO.FileNotFoundException)
                {
                    var file = Database.Find<Models.File>(id);

                    Console.WriteLine("No file contents downloaded for {0}", file != null ? file.Path : "unknown file");
                }
                catch (System.IO.DirectoryNotFoundException)
                {
                    Console.WriteLine("No file contents downloaded yet.");

                    break;
                }

                UpdateProgress(2);
            }

            Database.Commit();
        }

        protected void CalculateFileMetrics(string id)
        {
            var path = ArchivePathForFile(id);

            using (var zipFile = ZipFile.Read(path))
            {
                var versions = Database.Query<Models.Version>(@"
                    select V.* from Versions V
                    left join M_SS2007010201 M on M.ID = V.ID
                    where V.File = ? and M.ID is null", id);

                foreach (var version in versions)
                {
                    ExitIfCancelled();

                    if (zipFile.ContainsEntry(version.Name))
                    {
                        CalculateVersionMetrics(version, zipFile);
                    }
                }
            }
        }

        protected void CalculateVersionMetrics(Models.Version version, ZipFile zipFile)
        {
            using (var source = new System.IO.MemoryStream())
            {
                zipFile[version.Name].Extract(source);

                source.Seek(0, System.IO.SeekOrigin.Begin);

                File file = Analyzer.Analyze(null, source);

                // Determine LOC ourselves, it is broken in CSharpMetrics somehow...
                file.LOC = CalcLOC(source);


                InsertAnalyzeResults(version, file);
            }
        }

        protected int CalcLOC(System.IO.Stream s)
        {
            bool inMultiLineComment = false;
            // Rewind
            s.Position = 0;
            System.IO.StreamReader sr = new System.IO.StreamReader(s);
            int LOC = 0;
            string source = sr.ReadToEnd();
            // Normalize file endings
            source = source.Replace("\r\n", "\n"); // M$ overdoing it
            source = source.Replace("\r", "\n");   // Apple just being weird
            string[] lines = source.Split('\n');
            foreach (string line in lines)
            {
                string trimmed = line.Trim();

                if (!inMultiLineComment)
                {
                    if (trimmed.StartsWith("//"))
                    {
                        // found a single line comment
                        continue;
                    }
                    else if (trimmed.StartsWith("/*"))
                    {
                        if (!trimmed.Contains("*/"))
                        {
                            // Start of multiline comment
                            inMultiLineComment = true;
                            continue;
                        }
                        // Found a /* */ used a single line comment
                    }
                    else if (trimmed.Length > 0)
                    {
                        // Found a line of code :)
                        LOC++;
                    }
                    
                    if (trimmed.Contains("/*")) {
                        // Maybe they added a multiline comment after something usefull
                        if (!trimmed.Contains("*/"))
                        {
                            // Start of multiline comment
                            inMultiLineComment = true;
                            continue;
                        }
                    }
                }
                else
                {
                    // In multiline comment, look for ending
                    if (trimmed.Contains("*/"))
                    {
                        // check for another 
                        if (trimmed.Contains("/*"))
                        {
                            if (trimmed.IndexOf("/*")-trimmed.IndexOf("*/") > 2)
                            {
                                // Found some code in between
                                LOC++;
                            }
                        }
                        else
                        {
                            inMultiLineComment = false;
                            if (trimmed.IndexOf("*/") < trimmed.Length - 2)
                            {
                                // Found some code on this line
                                LOC++;
                            }
                        }
                    }
                }
            }
            return LOC;
        }

        protected void InsertAnalyzeResults(Models.Version version, File file)
        {
            Database.Insert(new Models.Size.File
            {
                ID = version.ID,
                MOD = file.NrClasses,
                MTD = file.NrMethods,
                LOC = file.LOC,
                MCB = file.Complexity,
                INV = file.Invocations,
                COM = file.CLOC,
                IDEP = file.NrImports,
                IF4 = 0,
                IF4c = 0,
                IF4v = 0,
            });

            InsertFileClasses(version, file);
        }

        protected void InsertFileClasses(Models.Version version, File file)
        {
            foreach (var ns in file.Namespaces.Values)
            {
                foreach (var klass in ns.Classes.Values)
                {
                    Database.Insert(new Models.Size.Class
                    {
                        ID = version.ID,
                        Module = klass.FullName,
                        LOC = klass.LOC,
                        MTD = klass.NrMethods,
                        PROP = klass.NrFields,
                        MCB = klass.Complexity,
                        INV = klass.Invocations,
                        COM = klass.CLOC,
                        WMC1 = 0,
                        WMCv = 0,
                        DIT = 0,
                        NOC = 0,
                        CBO = 0,
                        FIv = 0,
                        FIc = 0,
                        FI = 0,
                        FOv = 0,
                        FOc = 0,
                        FO = 0,
                        IF4v = 0,
                        IF4c = 0,
                        IF4 = 0,
                    });

                    InsertClassMethods(version, klass);
                }
            }
        }

        protected void InsertClassMethods(Models.Version version, Class klass)
        {
            foreach (var method in klass.Methods)
            {
                Database.Insert(new Models.Size.Method
                {
                    ID = version.ID,
                    Module = klass.FullName,
                    Prototype = method.Prototype,
                    SSIZE = method.NrParams,
                    LOC = method.LOC,
                    MCB = method.Complexity,
                    COM = method.CLOC,
                    INV = method.Invocations,
                    STC = method.MaxDepth,
                });
            }
        }

        protected string ArchivePathForFile(string id)
        {
            return string.Format("./extra/files/{0}.zip", id);
        }

        protected override void CreateTablesIfNecessary()
        {
            base.CreateTablesIfNecessary();

            if (Database.CreateTable<Models.Size.File>())
            {
                Database.Insert(new Models.Metric
                {
                    ID = "M_SS2007010201",
                    Name = "Basic metrics",
                    Type = Models.Metric.Types.Version,
                    Description = "Basic code metrics.<br />Computed using the basic metrics calculator.",
                    Dependencies = "D_SS2007010201_1:D_SS2007010201_2",
                });
            }

            Database.CreateTable<Models.Size.Class>();

            Database.CreateTable<Models.Size.Method>();
        }
    }
}
