﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using SQLite;
using CSharpDeps.Scopes;
using Microsoft.TeamFoundation.VersionControl.Client;

namespace SolidTA.SCMImporter.Commands.Deps
{
    class SolutionsProcessor : Processor
    {
        protected RelationsProcessor RelationsProcessor;

        protected MetricsProcessor MetricsProcessor;

        public override void PreProcess()
        {
            // Prepare RelationsMap processor
            Init(RelationsProcessor = new RelationsProcessor()).PreProcess();

            // Prepare Metrics processor
            Init(MetricsProcessor = new MetricsProcessor()).PreProcess();
        }

        public override void PostProcess()
        {
            // Once the whole solution has been analyzed, negate all relations that have not been
            // touched in order for them to be cancelled out.
            RelationsProcessor.PostProcess();

            // Only save metric results until after the full solution has been processed,
            // since e.g. Fan-In metric requires all files to have been processed.
            MetricsProcessor.PostProcess();
        }

        public override void Process(Solution solution)
        {
            var repoPath = RepositoryPathFromLocalPath(solution.Path);
            
            Solution = Database.Table<Models.Deps.Solution>().Where(s => s.Path == repoPath).FirstOrDefault();

            if (Solution == null)
            {
                Solution = new Models.Deps.Solution
                {
                    Path = repoPath,
                };

                Database.Insert(Solution);
            }

            RelationsProcessor.PrepareForSolution(Solution);

            base.Process(solution);
        }

        public override void Process(Solution solution, Project project)
        {
            var repoPath = RepositoryPathFromLocalPath(project.Path);

            Project = Database.Table<Models.Deps.Project>().Where(s => s.Solution == Solution.ID).Where(s => s.Path == repoPath).FirstOrDefault();

            if (Project == null)
            {
                Project = new Models.Deps.Project
                {
                    Solution = Solution.ID,
                    Path = repoPath,
                };

                Database.Insert(Project);
            }

            RelationsProcessor.PrepareForProject(Project);
            
            base.Process(solution, project);
        }

        public override void Process(Solution solution, Project project, File file)
        {
            var repoPath = RepositoryPathFromLocalPath(file.Path);
            
            File = Database.Table<Models.Deps.File>().Where(f => f.Project == Project.ID).Where(f => f.Path == repoPath).FirstOrDefault();

            if (File == null)
            {
                var tfsFile = Database.Table<Models.File>().Where(f => f.Path == repoPath).FirstOrDefault();

                File = new Models.Deps.File
                {
                    TFSFile = tfsFile != null ? tfsFile.ID : (int?)null,
                    Solution = Solution.ID,
                    Project = Project.ID,
                    Path = repoPath,
                };

                Database.Insert(File);
            }

            ProcessDependencies(solution, project, file);
        }

        protected void ProcessDependencies(Solution solution, Project project, File file)
        {
            Init(RelationsProcessor).Process(solution, project, file);
            Init(MetricsProcessor).Process(solution, project, file);
        }
    }
}
