﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text.RegularExpressions;
using CSharpDeps;
using IO = Delimon.Win32.IO;
using static SolidTA.SCMImporter.SCMServer;
using static SolidTA.SCMImporter.ChangeSet;

namespace SolidTA.SCMImporter.Commands
{
    class ExtractDependencies : TfsCommand
    {
        private static int COMMIT_CHUNK_SIZE = 100;
        private static string TMP_VERSION_PATH = "/extra/deps";

        protected Analyzer Analyzer = new Analyzer();

        protected Deps.SolutionsProcessor solutionsProcessor = new Deps.SolutionsProcessor();

        protected int? StartVersion;

        protected int AnalyzeEveryNthVersion = 1;

        protected Dictionary<string, Item> Changed = new Dictionary<string, Item>();

        public ExtractDependencies()
        {
            IsCommand("dependencies", "Extract dependency information using static analysis.");

            HasOption("v|version=", "Version to start analyzing from.", v => StartVersion = ParseOptionalInt(v));
            HasOption("e|every=", "Only analyze every N-th version", v => AnalyzeEveryNthVersion = ParseInt(v, 1));
        }

        protected int? ParseOptionalInt(string v)
        {
            int number;

            return int.TryParse(v, out number) ? (int?)number : null;
        }

        protected int ParseInt(string v, int def = 0)
        {
            int number;

            return int.TryParse(v, out number) ? number : def;
        }

        protected override void Execute(string[] args, string[] input)
        {
            Database.DisableForeignKeys();
            Database.BeginTransaction();

            if (!Directory.Exists(TMP_VERSION_PATH))
            {
                Directory.CreateDirectory(TMP_VERSION_PATH);
            }

            var start = LatestAnalyzedChangeset();

            // If a version to start analyzing from has been given, update to that version
            if (StartVersion.HasValue && StartVersion > start)
            {
                start = StartVersion.Value;
            }

            InitProgress(Server.GetLatestChangesetId() - start + 1);
            int? prevId = null;
            for (int startingCommit = start; startingCommit < Server.GetLatestChangesetId(); startingCommit += COMMIT_CHUNK_SIZE)
            {
                var changesets = Server.QueryFileHistory(Path, RecursionType.Full, new VersionSpec(startingCommit), COMMIT_CHUNK_SIZE, true, true);
                
                foreach (ChangeSet changeset in changesets)
                {
                    ExitIfCancelled(true);

                    TrackChangesInChangeset(changeset);
                    
                    if (((changeset.ChangesetId - start) % AnalyzeEveryNthVersion) == 0)
                    {
                        AnalyzeChangeset(changeset);
                    }

                    // Update progress. Because we do not know exactly how many changesets
                    // there are, possibly increment the progress bars multiple times.
                    if (!prevId.HasValue) prevId = changeset.ChangesetId - 1;

                    for (int i = 0; i < changeset.ChangesetId - prevId; ++i)
                    {
                        UpdateProgress();
                        UpdateProgress(2);
                    }

                    prevId = changeset.ChangesetId;
                }
            }

            Database.Commit();
        }

        protected int LatestAnalyzedChangeset()
        {
            var state = Database.Table<Models.Deps.State>().Where(s => s.ID == 1).FirstOrDefault();

            return state != null ? state.Changelog : 0;
        }

        protected int MostRecentChangeset()
        {
            return Server.GetLatestChangesetId();
        }

        protected void TrackChangesInChangeset(ChangeSet changeset)
        {
            foreach (var change in changeset.Changes)
            {
                switch (change.ChangeType)
                {
                    case ChangeType.Delete:
                        DeleteItem(change.Item);
                        break;
                    default:
                        TrackItem(change.Item);
                        break;
                }
            }
        }

        private void TrackItem(Item item)
        {
            var path = GetItemPath(item);

            if (path != null) Changed[path] = item;
        }

        protected void DeleteItem(Item item)
        {
            var path = LocalPathFromRepositoryPath(item.ServerItem);

            if (item.itemType == Item.ItemType.File)
            {
                if (IO.File.Exists(path))
                {
                    IO.File.Delete(path);
                }
            }
            else if (item.itemType == Item.ItemType.Folder)
            {
                if (IO.Directory.Exists(path))
                {
                    IO.Directory.Delete(path, true);
                }
            }
        }

        protected string GetItemPath(Item item)
        {
            if (item.itemType == Item.ItemType.File && IsRelevantItem(item))
            {
                return LocalPathFromRepositoryPath(item.ServerItem);
            }
            else
            {
                return null;
            }
        }

        protected string LocalPathFromRepositoryPath(string path)
        {
            return string.Format("{0}{1}{2}", Environment.CurrentDirectory, TMP_VERSION_PATH, Regex.Replace(path, "^\\$/", "")).Replace('/', '\\');
        }

        protected string RepositoryPathFromLocalPath(string path)
        {
            return Regex.Replace(path.Replace("\\", "/"), ".*"+ TMP_VERSION_PATH, "$/", RegexOptions.Compiled);
        }

        protected bool IsRelevantItem(Item item)
        {
            return !Regex.IsMatch(item.ServerItem, "\\.(?:png|gif|jpg|js|css|html?|docx?|xlsx?|pdf|as.x|resx|master)$", RegexOptions.IgnoreCase | RegexOptions.Compiled);
        }

        protected Deps.Processor InitProcessor(Deps.Processor processor, ChangeSet changeset)
        {
            return processor.Init(Database, Server, changeset);
        }

        protected void AnalyzeChangeset(ChangeSet changeset)
        {
            if (Changed.Count() > 0)
            {
                DownloadChangedItems();

                AnalyzeChanges(changeset);

                // Now that we have analyzed all changed files, clear the list
                Changed.Clear();
            }

            MarkChangesetAnalyzed(changeset.ChangesetId);
        }

        protected void DownloadChangedItems()
        {
            InitProgress(Changed.Count, 1);

            foreach (var pair in Changed)
            {
                DownloadItem(pair.Value, pair.Key);

                UpdateProgress(1);
            }
        }

        protected void DownloadItem(Item item, string path)
        {
            // Manually create the directory otherwise opening the file may fail.
            var dir = IO.Path.GetDirectoryName(path);

            if (!IO.Directory.Exists(dir))
            {
                var create = new List<string>();

                while (!IO.Directory.Exists(dir))
                {
                    create.Add(dir);
                    dir = IO.Path.GetDirectoryName(dir);
                }

                create.Reverse();

                foreach (var directory in create)
                {
                    IO.Directory.CreateDirectory(directory);
                }
            }

            using (var stream = item.fileStream)
            using (var file = IO.File.Open(path, IO.FileMode.Create, IO.FileAccess.Write))
            {
                stream.CopyTo(file);
            }
        }

        protected void AnalyzeChanges(ChangeSet changeset)
        {
            var solutions = SolutionsToAnalyze();
            
            InitProcessor(solutionsProcessor, changeset).PreProcess();
            InitProgress(solutions.Length, 1);

            foreach (var solution in solutions)
            {
                Console.WriteLine("Analyzing solution {0}:{1}", changeset.ChangesetId, solution);

                var results = Analyzer.Analyze(solution);

                if (results != null)
                {
                    ProcessAnalyzationResults(changeset, results);
                }
                else
                {
                    Console.WriteLine("Invalid solution {0}", solution);
                }

                UpdateProgress(1);
            }

            solutionsProcessor.PostProcess();
        }

        protected void ProcessAnalyzationResults(ChangeSet changeset, CSharpDeps.Scopes.Solution solution)
        {
            InitProcessor(solutionsProcessor, changeset).Process(solution);
        }

        protected string[] SolutionsToAnalyze()
        {
            var solutions = new HashSet<string>();

            foreach (var path in Changed.Keys)
            {
                var project = DetermineSolution(path);

                solutions.UnionWith(project);
            }

            return solutions.ToArray();
        }

        protected IEnumerable<string> DetermineSolution(string path)
        {
            if (path.EndsWith(".sln"))
            {
                yield return path;
                yield break;
            }

            var dbPath = RepositoryPathFromLocalPath(path);
            var file = Database.Table<Models.Deps.File>().Where(f => f.Path == dbPath).FirstOrDefault();
            var model = file != null ? Database.Find<Models.Deps.Solution>(file.Solution) : null;
            var local = model != null ? LocalPathFromRepositoryPath(model.Path) : null;

            if (local != null && IO.File.Exists(local))
            {
                yield return local;
            }
            else foreach (var solution in FindSolutionFile(path))
            {
                yield return solution;
            }
        }

        protected IEnumerable<string> FindSolutionFile(string path)
        {
            var dir = IO.Path.GetDirectoryName(path);

            while (dir.StartsWith(Environment.CurrentDirectory + "\\extra\\"))
            {
                var solutions = IO.Directory.GetFiles(dir, "*.sln");

                foreach (var solution in solutions)
                {
                    yield return solution;
                }

                if (solutions.Length > 0) yield break;

                dir = IO.Path.GetDirectoryName(dir);
            }
        }

        protected void MarkChangesetAnalyzed(int id)
        {
            Database.InsertOrUpdate(new Models.Deps.State
            {
                ID = 1,
                Changelog = id,
            });

            // Commit intermediary results
            Database.Commit();
            Database.BeginTransaction();
        }

        protected override void CreateTablesIfNecessary()
        {
            base.CreateTablesIfNecessary();

            if (Database.CreateTable<Models.Deps.State>())
            {
                Database.Insert(new Models.Metric
                {
                    ID = "M_TFS_Deps_State",
                    Name = "Dependencies",
                    Type = Models.Metric.Types.Version,
                    Description = "C# code dependencies analyzer.<br />Only available for TFS repositories.",
                    Dependencies = "D_TFS_Deps_Solutions;D_TFS_Deps_Projects;D_TFS_Deps_Files;D_TFS_Deps_Members:D_TFS_Deps_Relations:D_TFS_Deps_VersionedClasses:D_TFS_Deps_VersionedMethods:D_TFS_Deps_ClassMetrics:D_TFS_Deps_MethodMetrics",
                });
            }

            Database.CreateTable<Models.Deps.Solution>();
            Database.CreateTable<Models.Deps.Project>();
            Database.CreateTable<Models.Deps.File>();
            Database.CreateTable<Models.Deps.Member>();
            Database.CreateTable<Models.Deps.Relation>();
            Database.CreateTable<Models.Deps.VersionedClass>();
            Database.CreateTable<Models.Deps.VersionedMethod>();
            Database.CreateTable<Models.Deps.ClassMetric>();
            Database.CreateTable<Models.Deps.MethodMetric>();
        }
    }
}
