﻿using System;
using System.Collections.Generic;
using Microsoft.TeamFoundation;
using Microsoft.TeamFoundation.WorkItemTracking.Client;

namespace SolidTA.SCMImporter.Commands
{
    class ExtractWorkItemMetric : TfsCommand
    {
        public ExtractWorkItemMetric()
        {
            IsCommand("work-items", "Imports version associated Work Items.");
        }

        protected override void Execute(string[] args, string[] input)
        {
            TFSServer tfsServer = null;
            if(Server is TFSServer)
            {
                tfsServer = (TFSServer)Server;
            } else
            {
                throw new ArgumentException("This metric is only supported for TFS repositories");
            }


            var store = tfsServer.GetService<WorkItemStore>();

            var items = store.Query(@"
                SELECT [System.Id], [System.CreatedBy], [System.State],
                    [System.CreatedDate], [System.Description],
                    [System.Reason], [System.Title], [System.WorkItemType]
                FROM WorkItems
                WHERE [System.ExternalLinkCount] > 0
                ORDER BY [System.Id] ASC");

            Database.Table<Models.Metrics.VersionToWorkItem>().Delete();
            Database.Table<Models.Metrics.WorkItem>().Delete();

            InitProgress(items.Count, 2);

            Database.BeginTransaction();

            foreach (WorkItem item in items)
            {
                ExitIfCancelled();

                var changesets = GetChangesets(item);

                if (changesets.Count > 0)
                {
                    var model = new Models.Metrics.WorkItem
                    {
                        ID = item.Id,
                        Author = item.CreatedBy,
                        CreationDate = item.CreatedDate,
                        Title = item.Title,
                        Description = item.Description,
                        State = item.State,
                        Reason = item.Reason,
                        Type = item.Type.Name,
                    };

                    Database.Insert(model);

                    foreach (var changeset in changesets)
                    {
                        Database.Insert(new Models.Metrics.VersionToWorkItem
                        {
                            Version = changeset,
                            WorkItem = model.ID,
                        });
                    }
                }

                UpdateProgress(2);
            }

            Database.Commit();
        }

        protected List<int> GetChangesets(WorkItem item)
        {
            var changesets = new List<int>();

            foreach (Link link in item.Links)
            {
                ExitIfCancelled();

                ExternalLink extLink = link as ExternalLink;

                if (extLink != null)
                {
                    ArtifactId artifact = LinkingUtilities.DecodeUri(extLink.LinkedArtifactUri);

                    if (String.Equals(artifact.ArtifactType, "Changeset", StringComparison.Ordinal))
                    {
                        changesets.Add(int.Parse(artifact.ToolSpecificId));
                    }
                }
            }

            return changesets;
        }

        protected override void CreateTablesIfNecessary()
        {
            base.CreateTablesIfNecessary();

            Database.CreateTable<Models.Metrics.WorkItem>();

            if (Database.CreateTable<Models.Metrics.VersionToWorkItem>())
            {
                Database.Insert(new Models.Metric
                {
                    ID = "M_JK2013051501",
                    Name = "TFS Work Items",
                    Type = Models.Metric.Types.Version,
                    Description = "Version associated Work Items from a TFS repository",
                    Dependencies = "D_JK2013051501",
                });
            }
        }
    }
}
