﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using Ionic.Zip;
using static SolidTA.SCMImporter.SCMServer;

namespace SolidTA.SCMImporter.Commands
{
    class ImportFileContents : TfsCommand
    {
        private static string TMP_VERSION_PATH = "./extra/files";
        private static int COMMIT_CHUNK_SIZE = 100;

        public ImportFileContents()
        {
            IsCommand("contents", "Imports the contents of the specified files at the given snapshot times");

            RequiresAdditionalArgumentsViaInput();
        }

        protected override void Execute(string[] args, string[] ids)
        {
            InitProgress(ids.Length);

            if (!Directory.Exists(TMP_VERSION_PATH))
            {
                Directory.CreateDirectory(TMP_VERSION_PATH);
            }

            List<Models.File> files = new List<Models.File>();
            foreach (var id in ids)
            {
                ExitIfCancelled();

                var file = Database.Find<Models.File>(id);
                if (file != null)
                {
                    files.Add(file);
                }
                else
                {
                    Console.WriteLine("File with id [{0}] does not exist in the database.", id);
                }
            }

            ImportVersions(files);
        }

        protected void ImportVersions(List<Models.File> files)
        {
            if (Server.GetRepoMode().Equals(RepoOperationMode.HORIZONTAL))
            {
                ImportVersionsHorizontal(files);
            } else {
                ImportVersionsVertical(files);
            }
        }

        protected void ImportVersionsHorizontal(List<Models.File> files)
        {
            //Query the changesets for a specific file and then process the affected changesets
            foreach (var file in files)
            {
                var path = ArchivePathForFile(file);

                using (var zipFile = new ZipFile(path))
                {
                    var begin = LatestImportedChangelog(zipFile) + 1;

                    var changesets = Server.QueryFileHistory(file.Path, RecursionType.None, new VersionSpec(begin), int.MaxValue, true, true);
                    foreach (ChangeSet changeset in changesets)
                    {
                        ExitIfCancelled();

                        PutChangesetContents(changeset, zipFile);
                    }

                    zipFile.Save();
                }
                UpdateProgress();
            }
        }

        protected void ImportVersionsVertical(List<Models.File> files)
        {
            int latestChangeSet = Server.GetLatestChangesetId();
            int numberOfUIUpdates = latestChangeSet / COMMIT_CHUNK_SIZE;
            InitProgress(numberOfUIUpdates);

            //Process the data in chunks
            for (int startingCommit = 0; startingCommit < latestChangeSet; startingCommit += COMMIT_CHUNK_SIZE)
            {
                //Query changesets for all files (path="")
                var changesets = Server.QueryFileHistory("", RecursionType.Full, new VersionSpec(startingCommit), COMMIT_CHUNK_SIZE, true, true);

                //process each database file
                foreach (var file in files)
                {
                    var path = ArchivePathForFile(file);
                    using (var zipFile = new ZipFile(path))
                    {
                        //Loop through the changesets
                        foreach (ChangeSet changeset in changesets)
                        {
                            ExitIfCancelled();

                            foreach (var change in changeset.Changes)
                            {
                                //Check weather the database file is in the changeset
                                if (file.Path.Equals(change.Item.ServerItem))
                                {
                                    ExitIfCancelled();
                                    PutHorizontalContents(changeset, change, zipFile);
                                    break;
                                }
                            }
                        }
                        zipFile.Save();
                    }
                }
                UpdateProgress();
            }
        }

        protected void PutHorizontalContents(ChangeSet changeset, ChangeSet.Change change, ZipFile zipFile)
        {
            if (zipFile.ContainsEntry(changeset.ChangesetId.ToString()))
            {
                return;
            }
            if (change.Item != null)
            {
                using (Stream data = change.Item.fileStream)
                {
                    if (data != null)
                    {
                        zipFile.AddEntry(changeset.ChangesetId.ToString(), GetBytes(data));
                    }
                }                
            }
        }

        protected void PutChangesetContents(ChangeSet changeset, ZipFile zipFile)
        {
            if(zipFile.ContainsEntry(changeset.ChangesetId.ToString()))
            {
                return;
            }

            if(0<changeset.Changes.Count()) { 
                ChangeSet.Change change = changeset.Changes.FirstOrDefault(null);
                if (change.Item != null)
                {
                    using(Stream data = change.Item.fileStream)
                    {
                        if (data != null)
                        {
                            zipFile.AddEntry(changeset.ChangesetId.ToString(), GetBytes(data));
                        }
                    }
                }
            }
        }

        protected byte[] GetBytes(Stream input)
        {
            using (var output = new MemoryStream())
            {
                byte[] buffer = new byte[32768];
                int read;

                while ((read = input.Read(buffer, 0, buffer.Length)) > 0)
                {
                    output.Write(buffer, 0, read);
                }

                return output.ToArray();
            }
        }

        protected int LatestImportedChangelog(ZipFile zipFile)
        {
            return zipFile.EntryFileNames.Count > 0 ? zipFile.EntryFileNames.Select(n => int.Parse(n)).Max() : 0;
        }

        protected string ArchivePathForFile(Models.File file)
        {
            return string.Format(TMP_VERSION_PATH + "/{0}.zip", file.ID);
        }
    }
}
