﻿using System;
using System.Collections.Generic;
using SolidTA.SCMImporter.Models;
using static SolidTA.SCMImporter.SCMServer;

namespace SolidTA.SCMImporter.Commands
{
    class ImportFileVersions : TfsCommand
    {
        private static int COMMIT_CHUNK_SIZE = 100;

        public ImportFileVersions()
        {
            IsCommand("versions", "Imports all version history of the specified files into the database.");

            RequiresAdditionalArgumentsViaInput();
        }

        protected override void Execute(string[] args, string[] ids)
        {
            InitProgress(ids.Length);

            Database.BeginTransaction();

            List<Models.File> files = new List<Models.File>();
            foreach (var id in ids)
            {
                var file = Database.Find<Models.File>(id);

                if (file != null)
                {
                    files.Add(file);
                }
                else
                {
                    Console.WriteLine("File with id [{0}] does not exist in the database.", id);
                }
            }

            ImportVersions(files);

            Database.Commit();
        }

        protected void ImportVersions(List<Models.File> files)
        {
            if (Server.GetRepoMode().Equals(RepoOperationMode.HORIZONTAL))
            {
                ImportVersionsHorizontal(files);
            } else
            {
                ImportVersionsVertical(files);
            }
        }

        private void ImportVersionsHorizontal(List<File> files)
        {
            foreach (var file in files)
            {
                ExitIfCancelled();

                int begin = LatestImportedChangelog(file);
                var changesets = Server.QueryFileHistory(file.Path, RecursionType.Full, new VersionSpec(begin), int.MaxValue, true, false);

                foreach (ChangeSet changeset in changesets)
                {
                    ExitIfCancelled();

                    InsertChangeset(file, changeset);
                }

                UpdateProgress();
            }
        }

        private void ImportVersionsVertical(List<File> files)
        {
            int latestChangeSet = Server.GetLatestChangesetId();
            int numberOfUIUpdates = latestChangeSet / COMMIT_CHUNK_SIZE;
            InitProgress(numberOfUIUpdates);

            //Process the data in chunks
            for (int startingCommit = 0; startingCommit < latestChangeSet; startingCommit += COMMIT_CHUNK_SIZE)
            {
                GC.Collect(); // FORCE GARBAGE COLLECTION
                var changesets = Server.QueryFileHistory("", RecursionType.Full, new VersionSpec(startingCommit), COMMIT_CHUNK_SIZE, true, false);
                foreach (var file in files)
                {
                    ExitIfCancelled();

                    int begin = LatestImportedChangelog(file);
                    foreach (ChangeSet changeset in changesets)
                    {
                        // Should be <= probably, might introduce a bug when continueing a previous aborted run, -1 instead of 0 in
                        // LatestImportedChangelog
                        if (changeset.ChangesetId < begin)
                        {
                            continue;
                        }

                        foreach (var change in changeset.Changes)
                        {
                            if (file.Path.Equals(change.Item.ServerItem))
                            {
                                InsertChangeset(file, changeset);
                                break;
                            }
                        }
                    }
                }
                UpdateProgress(); // Update for each chunk
            }
        }

        private void InsertChangeset(Models.File file, ChangeSet changeset)
        {
            var version = new Models.Version
            {
                File = file.ID,
                Name = changeset.ChangesetId.ToString(),
                Time = changeset.CreationDate,
            };

            Database.Insert(version);

            Database.Insert(new Models.Metrics.Author
            {
                ID = version.ID,
                Name = changeset.Committer,
            });

            Database.Insert(new Models.Metrics.Comment
            {
                ID = version.ID,
                Name = changeset.Comment,
            });
        }

        protected int LatestImportedChangelog(Models.File file)
        {
            return Database.Table<Models.Version>().Where(v => v.File == file.ID).Max("Name") ?? 0;
        }
    }
}
