﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using Microsoft.TeamFoundation.VersionControl.Client;

namespace SolidTA.SCMImporter
{
    class TFSServer : SCMServer
    {
        private Microsoft.TeamFoundation.Client.TfsTeamProjectCollection Server;

        public override void Connect(string url, string directory, string user, string password)
        {
            var provider = new Microsoft.TeamFoundation.Client.UICredentialsProvider();

            System.Diagnostics.Debug.WriteLine("Url " + url);
            Uri uri = ParseUrl(url);

            if (String.IsNullOrEmpty(user))
            {
                Server = new Microsoft.TeamFoundation.Client.TfsTeamProjectCollection(uri, provider);
            }
            else
            {
                var credentials = GetCredentials(user, password);

                Server = new Microsoft.TeamFoundation.Client.TfsTeamProjectCollection(uri, credentials, provider);
            }

            Server.EnsureAuthenticated();
        }

        protected Uri ParseUrl(string url)
        {
            if (!Uri.IsWellFormedUriString(url, UriKind.Absolute))
            {
                var servers = Microsoft.TeamFoundation.Client.RegisteredTfsConnections.GetConfigurationServers();
                var registered = servers.FirstOrDefault(s => String.Equals(s.Name, url, StringComparison.OrdinalIgnoreCase));

                if (registered == null)
                {
                    throw new ArgumentException("The given URL is invalid and could not be found as a TFS instance.");
                }

                return registered.Uri;
            }
            else
            {
                return new Uri(url);
            }
        }

        protected ICredentials GetCredentials(string user, string password)
        {
            int split = user.IndexOf('\\');
            if (split >= 0)
            {
                string domain = user.Substring(0, split);

                return new NetworkCredential(user.Substring(split + 1), password, domain);
            }
            else
            {
                return new NetworkCredential(user, password);
            }
        }

        public override string PreparePath(string path)
        {
            if (string.IsNullOrEmpty(path))
            {
                return "$/";
            }
            else if (path.StartsWith("$/"))
            {
                return path;
            }
            else if (path.StartsWith("/"))
            {
                return "$" + path;
            }
            else if (path.StartsWith("./"))
            {
                return "$" + path.Substring(1);
            }
            else
            {
                return "$/" + path;
            }
        }

        public override VersionSpec GetFirstVersionAfter(DateTime dt)
        {
            return new DateVersionSpec(dt);
        }

        public override IEnumerable<ChangeSet> QueryFileHistory(string path, RecursionType recursion, VersionSpec versionFrom, int maxCount, bool includeChanges, bool includeDownloadInfo)
        {
            //Converting types to TFS types
            var recursionConv = ConvertHelper.RecursionType(recursion);
            var versionFromConv = ConvertHelper.VersionSpec(versionFrom);

            //Run the command
            var changesetRaw = GetService<Microsoft.TeamFoundation.VersionControl.Client.VersionControlServer>().QueryHistory(path, versionFromConv, 0, recursionConv,
                    null, versionFromConv, Microsoft.TeamFoundation.VersionControl.Client.VersionSpec.Latest, maxCount, includeChanges, true, includeDownloadInfo, true);

            //rewrap the result in a list
            List<ChangeSet> changeSetList = new List<ChangeSet>();
            foreach (Microsoft.TeamFoundation.VersionControl.Client.Changeset changeset in changesetRaw)
            {
                changeSetList.Add(ConvertHelper.FromChangeSet(changeset));
            }
            return changeSetList;
        }

        public override Item[] GetItems(string path, VersionSpec version, RecursionType recursion, DeletedState deletedState, Item.ItemType itemType, bool includeDownloadInfo)
        {
            //Converting types to TFS types
            var versionConv = ConvertHelper.VersionSpec(version);
            var recursionConv = ConvertHelper.RecursionType(recursion);
            Microsoft.TeamFoundation.VersionControl.Client.DeletedState deletedStateConv = ConvertHelper.DeletedState(deletedState);
            Microsoft.TeamFoundation.VersionControl.Client.ItemType itemTypeConv = ConvertHelper.ItemType(itemType);

            //Run the command
            Microsoft.TeamFoundation.VersionControl.Client.ItemSet itemSetRaw;
            itemSetRaw = GetService<Microsoft.TeamFoundation.VersionControl.Client.VersionControlServer>().GetItems(path, versionConv, recursionConv, deletedStateConv, itemTypeConv, includeDownloadInfo);

            //rewrap the result in a list
            List<Item> items = new List<Item>();
            foreach (Microsoft.TeamFoundation.VersionControl.Client.Item item in itemSetRaw.Items)
            {
                items.Add(new Item(itemType, item.ServerItem, item.DownloadFile()));
            }
            return items.ToArray();
        }

        public override int GetLatestChangesetId()
        {
            return GetService<Microsoft.TeamFoundation.VersionControl.Client.VersionControlServer>().GetLatestChangesetId();
        }

        public T GetService<T>()
        {
            return Server.GetService<T>();
        }

        class DateVersionSpec : VersionSpec
        {
            public DateTime dt { get; private set; }
            public DateVersionSpec(DateTime dt) : base(-1)
            {
                this.dt = dt;
            }
        }

        public override RepoOperationMode GetRepoMode()
        {
            return RepoOperationMode.HORIZONTAL;
        }

        /**
         * Static class, which contains wrapper function to convert from the SCMServer types to the TFS specific classes
         **/
        static class ConvertHelper
        {

            public static Microsoft.TeamFoundation.VersionControl.Client.DeletedState DeletedState(DeletedState deletedState)
            {
                switch (deletedState)
                {
                    case SCMServer.DeletedState.Any:
                        return Microsoft.TeamFoundation.VersionControl.Client.DeletedState.Any;
                    case SCMServer.DeletedState.Deleted:
                        return Microsoft.TeamFoundation.VersionControl.Client.DeletedState.Deleted;
                    case SCMServer.DeletedState.NonDeleted:
                        return Microsoft.TeamFoundation.VersionControl.Client.DeletedState.NonDeleted;
                    default:
                        throw new NotImplementedException();
                }
            }

            public static Microsoft.TeamFoundation.VersionControl.Client.ItemType ItemType(Item.ItemType itemType)
            {
                switch (itemType)
                {
                    case Item.ItemType.Any:
                        return Microsoft.TeamFoundation.VersionControl.Client.ItemType.Any;
                    case Item.ItemType.File:
                        return Microsoft.TeamFoundation.VersionControl.Client.ItemType.File;
                    case Item.ItemType.Folder:
                        return Microsoft.TeamFoundation.VersionControl.Client.ItemType.Folder;
                    default:
                        throw new NotImplementedException();
                }
            }

            public static Microsoft.TeamFoundation.VersionControl.Client.RecursionType RecursionType(RecursionType recursionType)
            {
                switch (recursionType)
                {
                    case SCMServer.RecursionType.Full:
                        return Microsoft.TeamFoundation.VersionControl.Client.RecursionType.Full;
                    case SCMServer.RecursionType.None:
                        return Microsoft.TeamFoundation.VersionControl.Client.RecursionType.None;
                    default:
                        throw new NotImplementedException();
                }
            }

            public static Microsoft.TeamFoundation.VersionControl.Client.VersionSpec VersionSpec(VersionSpec versionSpec)
            {
                if (versionSpec is DateVersionSpec)
                {
                    var dt = ((DateVersionSpec)versionSpec).dt;
                    return new Microsoft.TeamFoundation.VersionControl.Client.DateVersionSpec(dt);
                }
                return new ChangesetVersionSpec(versionSpec.version);
            }

            public static ChangeSet FromChangeSet(Changeset changeset)
            {
                List<ChangeSet.Change> changes = new List<ChangeSet.Change>();
                foreach (Change change in changeset.Changes)
                {
                    Item item = new Item(FromItemType(change.Item.ItemType), change.Item.ServerItem, change.Item.DownloadFile());
                    ChangeSet.Change scmChange = new ChangeSet.Change(item, FromChangeType(change.ChangeType));
                    changes.Add(scmChange);
                }

                return new ChangeSet(changeset.ChangesetId, changeset.CreationDate, changeset.Committer, changeset.Comment, changes);
            }

            private static Item.ItemType FromItemType(ItemType itemType)
            {
                switch (itemType)
                {
                    case Microsoft.TeamFoundation.VersionControl.Client.ItemType.Any:
                        return Item.ItemType.Any;
                    case Microsoft.TeamFoundation.VersionControl.Client.ItemType.File:
                        return Item.ItemType.File;
                    case Microsoft.TeamFoundation.VersionControl.Client.ItemType.Folder:
                        return Item.ItemType.Folder;
                    default:
                        throw new NotImplementedException();
                }
            }

            private static ChangeSet.ChangeType FromChangeType(ChangeType changeType)
            {
                switch(changeType)
                {
                    case ChangeType.Add:
                    case ChangeType.Undelete:
                        return ChangeSet.ChangeType.Add;
                    case ChangeType.Edit:
                    case ChangeType.Encoding:
                    case ChangeType.Merge:
                    case ChangeType.Rename:
                        return ChangeSet.ChangeType.Edit;
                    case ChangeType.Delete:
                        return ChangeSet.ChangeType.Delete;
                    case ChangeType.None:
                        return ChangeSet.ChangeType.None;
                    default:
                        throw new NotImplementedException();
                }
            }
        }
    }
}
