#*****************************************************************************************
# Name			: Author ID
# File			: pextAuthor.py
# Description	: A plugin for visualizing author ID information.
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

#------------------------------------------------------------ Global imports ---
import wx
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GLU import *

import pysqlite2.dbapi2 as sqlite

#------------------------------------------------------------- Local imports ---
import sta_globals
import sta_plugin
import sta_utils
import sta_gl_list
import sta_gl_mixerunit

#===============================================================================
#													Configuration dialog box
#===============================================================================

class CSettingsDlg(wx.Dialog):
	def __init__(self, pParent, *args, **kwds):

		self.pParent = pParent

		kwds["style"] = wx.DEFAULT_DIALOG_STYLE

		wx.Dialog.__init__(self, *args, **kwds)
		self.txt1 = wx.StaticText(self, -1, "   Look-up period")

		self.comboTimeRef	= wx.ComboBox(self, -1, choices=['3 months','6 months','12 months','18 months','2 years','3 years','4 years','5 years','10 years'], style=wx.CB_DROPDOWN|wx.CB_READONLY)

		self.bOK			= wx.Button(self, -1, "Ok")
		self.bCancel		= wx.Button(self, -1, "Cancel")

		self.__set_properties()
		self.__do_layout()
		self.__get_values()

		self.bOK.Bind(wx.EVT_BUTTON,self.OnOK)
		self.bCancel.Bind(wx.EVT_BUTTON,self.OnCancel)

	def __set_properties(self):
		self.SetTitle('Author metric settings')
		self.comboTimeRef.SetMinSize((150, -1))
		self.bOK.SetMinSize((50, -1))
		self.bCancel.SetMinSize((50, -1))

	def __do_layout(self):

		self.sizerMain = wx.GridSizer(2,2,0,0)
		self.sizerMain.Add(self.txt1, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.comboTimeRef, 0, wx.ALIGN_CENTER_VERTICAL|wx.ADJUST_MINSIZE, 0)
		self.sizerMain.Add(self.bOK, 0, wx.ALL|wx.ALIGN_RIGHT|wx.ADJUST_MINSIZE, 10)
		self.sizerMain.Add(self.bCancel, 0, wx.ALL|wx.ALIGN_LEFT|wx.ADJUST_MINSIZE, 10)

		self.SetAutoLayout(True)
		self.SetSizer(self.sizerMain)
		self.sizerMain.Fit(self)
		self.sizerMain.SetSizeHints(self)
		self.Layout()

	def __get_values(self):
		self.comboTimeRef.SetSelection(self.pParent.iPastRef_GUI)

	def OnOK(self, event):
		try:
			self.pParent.iPastRef_GUI 		= self.comboTimeRef.GetSelection()
			self.pParent.iPastRef			= self.pParent.iStepRef * self.pParent.iPastRefList[self.pParent.iPastRef_GUI]

			self.pParent.cGUIList.RefreshMetric()

			self.EndModal(wx.ID_OK)
			event.Skip()
		except:
			dlg = wx.MessageDialog(sta_globals.GUI.mainFrame,'Look-up period','Incorrect value',style=wx.OK|wx.ICON_EXCLAMATION)
			dlg.ShowModal()
			dlg.Destroy()

	def OnCancel(self, event):
		self.EndModal(wx.ID_CANCEL)
		event.Skip()

#===============================================================================
#																Authors plugin
#===============================================================================
class Plugin(sta_plugin.Plugin):

	def __init__(self,*args, **kwds):
		sta_plugin.Plugin.__init__(self,*args, **kwds)
		self.sClass				= 'pextAuthor'		# Name of the plugin file
		self.sName				= 'Authors'	 		# Name to display in a list
		self.sVersion			= '1.0'		 		# Version
		self.sMetricID			= 'author_name'

		self.sTable				= 'M_SS2007010101'	# Name of the table where metric is stored
		self.lTables.append(self.sTable)			# Append to this list all tables that are used

		#----------------------------------------------------- Specific processing

		self.gTitle				= ''				# GUI title
		self.bHasDecay          = True		        # True if commit influence is decaying
		self.iDecayFactor		= 2
		self.iNrDecayLevels		= 10
		self.lDecayValues		= [[0,0.8],[0,0.6],[0,0.4],[0,0.2],[0,0.1],[0,0.05],[0,0.04],[0,0.03],[0,0.02],[0,0.01],[0,0.01],[0,0.01]] # (time,value)
		for i in range(12):
			self.lDecayValues[i][0] = (i+1)*604800-1

		self.iStepRef				= 7884000			# Reference step
		self.iPastRef				= self.iStepRef * 4	# Past reference to look for activity (1 year)
		self.iPastRefList			= [1,2,4,6,8,12,16,20,40]
		self.iPastRef_GUI			= 2

#-------------------------------------------------------------------------------
#																 Filter GUI
#-------------------------------------------------------------------------------

#----------------------------------------------------------------- Build GUI ---
	def buildFilterGUI(self,pMasterGUI):

		self.cGUIMaster = pMasterGUI

		self.cSizer = wx.BoxSizer(wx.VERTICAL)
		l_cConfigSizer = wx.BoxSizer(wx.HORIZONTAL)

		l_cLabelSizer = wx.BoxSizer(wx.HORIZONTAL)
		l_sLabel = wx.StaticText(pMasterGUI.project_filters, -1, '  '+self.sName)
		self.g_txtMetric = wx.StaticText(pMasterGUI.project_filters, -1, style = wx.ALIGN_RIGHT )
		self.g_txtMetric.SetLabel('')
		l_cLabelSizer.Add(l_sLabel, 0, wx.ADJUST_MINSIZE, 0)
		l_cLabelSizer.Add(self.g_txtMetric, 0, wx.ALIGN_RIGHT, 0)

		self.cGUIList = sta_gl_list.wxColorCheckListBox(self.sName,self.sClass,pMasterGUI.project_filters, style=wx.BORDER_SUNKEN|wx.LB_MULTIPLE|wx.LB_SORT)
		self.cGUIList.lObservers.append(self)

		self.cGUIScale = wx.Slider(pMasterGUI.project_filters, -1, self.iDecayFactor, 1, self.iNrDecayLevels)
		self.cGUIConfig = wx.Button(pMasterGUI.project_filters, -1, 'cfg')
		self.cGUIScale.SetOwnBackgroundColour(wx.Colour(150,150,180))

		self.cGUIList.SetData(sta_globals.lIDColors)
		self.cGUIList.AddMetric("revisions",self.MetricAuthorContribution)
		self.cGUIList.AddMetric("text hits",self.MetricAuthorTextHits)
		self.cGUIList.AddMetric("age",self.MetricAuthorAge)

		self.cGUIList.Bind(wx.EVT_LISTBOX,self.cbListSelect)
		self.cGUIScale.Bind(wx.EVT_SLIDER,self.cbLevelSelect)
		self.cGUIConfig.Bind(wx.EVT_BUTTON,self.cbConfig)

		self.cGUIList.SetMinSize((160,-1))
		self.cGUIConfig.SetMinSize((30,15))
		self.cGUIScale.SetMinSize((-1,15))

		self.cSizer.Add(l_cLabelSizer, 0, wx.EXPAND, 0)
		self.cSizer.Add(self.cGUIList, 1, wx.EXPAND|wx.ADJUST_MINSIZE)
		l_cConfigSizer.Add(self.cGUIScale, 1, wx.EXPAND, 0)
		l_cConfigSizer.Add(self.cGUIConfig, 0, wx.ADJUST_MINSIZE, 0)
		pMasterGUI.cListsSizer_Project.Add(self.cSizer, 0, wx.EXPAND, 0)
		self.cSizer.Add(l_cConfigSizer, 0, wx.EXPAND, 0)

		self.showFilterGUI(False)
		self.doBindings()

	def cbConfig(self,evt):
		dlg = CSettingsDlg(self,sta_globals.GUI.mainFrame)
		val = dlg.ShowModal()
		dlg.Destroy()

#-------------------------------------------------------------------------------
#								 Filter GUI callback
#-------------------------------------------------------------------------------

	def cbListSelect_aux(self, files):
		if len(self.lIdx) == 0:
			for i in files:
				for j in i.lRevs:
					try:
						l_iIdx = self.lData.index(j.lValues[self.sMetricID])
						j.lValues[self.sClass] = l_iIdx
					except:
						j.lValues[self.sClass]=-1
		else:
			for i in files:
				for j in i.lRevs:
					try:
						l_iIdx = self.lData.index(j.lValues[self.sMetricID])
						j.lValues[self.sClass] = -1
						if l_iIdx in self.lIdx:
							j.lValues[self.sClass] = l_iIdx
					except:
						j.lValues[self.sClass]=-1

	def getSelectedIndexes(self):
		l_lsUsers = self.cGUIList.GetSelections()
		self.clearSelection()
		self.lIdx = sta_utils.getSelectionIndex(l_lsUsers,self.lData)
		self.bSelected = len(self.lIdx)>0

	def cbLevelSelect(self,evt):
		self.iDecayFactor = self.cGUIScale.GetValue()
		sta_plugin.Plugin.updateCanvases(self)

#-------------------------------------------------------------------------------
#									Filter GUI metrics
#-------------------------------------------------------------------------------
#--------------------------------------------------------- Number of commits ---
	def MetricAuthorContribution(self,sName):
		l_iNr = 0

		l_iStartTime = sta_globals.iStartFocusInterval
		if (l_iStartTime < 0):
			l_iStartTime = sta_globals.iSelectedFilesStartTime

		l_iEndTime = sta_globals.iEndFocusInterval
		if (l_iEndTime < 0):
			l_iEndTime = sta_globals.iSelectedFilesEndTime

		for i in sta_globals.lSelectedFiles:
			for j in i.lRevs:
				if (j.iTime >= l_iStartTime) and (j.iTime <= l_iEndTime):
					if j.lValues[self.sMetricID] == sName:
						l_iNr = l_iNr+1
		return l_iNr
#--------------------------------------------------------- Number of commits ---
	def MetricAuthorTextHits(self,sName):
		l_iNr = 0

		l_iStartTime = sta_globals.iStartFocusInterval
		if (l_iStartTime < 0):
			l_iStartTime = sta_globals.iSelectedFilesStartTime

		l_iEndTime = sta_globals.iEndFocusInterval
		if (l_iEndTime < 0):
			l_iEndTime = sta_globals.iSelectedFilesEndTime

		for i in sta_globals.lSelectedFiles:
			for j in i.lRevs:
				try:
					if (j.iTime >= l_iStartTime) and (j.iTime <= l_iEndTime) and (j.lValues['pextFind'] != -1):
						if j.lValues[self.sMetricID] == sName:
							l_iNr = l_iNr+1
				except:
					pass
		return l_iNr
#----------------------------------------------------------------------- Age ---
	def MetricAuthorAge(self,sName):
		l_iAge = sta_globals.iProjectEndTime - 1
		bIsPresent = False
		for i in sta_globals.lSelectedFiles:
			for j in i.lRevs:
				if j.lValues[self.sMetricID] == sName:
					if (l_iAge < sta_globals.iProjectStartTime) or (l_iAge > j.iTime):
						l_iAge = j.iTime
						bIsPresent = True
		if bIsPresent:
			return (sta_globals.iProjectEndTime - l_iAge)
		else:
			return 0
#-------------------------------------------------------------------------------
#																Data management
#-------------------------------------------------------------------------------
#--------------------------------------------------------------- Load metric ---
	def loadMetric(self):
		l_sDBPath = sta_globals.cSTAPrj.storagePath()+"/history.db"

		if not sta_utils.isValidDB(l_sDBPath):
			print 'DATA: no history information found'
			return

		cDB = sqlite.connect(l_sDBPath)
		cCursorDB = cDB.cursor()

		if not sta_utils.isValidTable(cCursorDB,self.sTable):
			print 'DATA: no Author information found'
			cCursorDB.close()
			cDB.close()
			return

		# Load list of author names
		sCmd = "select distinct(Name) from "+self.sTable
		cCursorDB.execute(sCmd)
		for i in cCursorDB:
			self.lData.append(unicode(i[0]))
			self.cGUIList.Append(unicode(i[0]))
		self.clearSelection()

		# Set the author name and ID for each version
		l_lMetric = {}
		sCmd = "select ID,Name from "+self.sTable
		cCursorDB.execute(sCmd)
		for i in cCursorDB:
			l_lMetric[i[0]]=i[1]

		for i in sta_globals.lFiles:
			for j in i.lRevs:
				#TODO what if this does not exist
				try:
					l_sAuthor = l_lMetric[j.iDBid]
					j.lValues[self.sMetricID]=l_sAuthor
					j.lValues[self.sClass]=self.lData.index(l_sAuthor)
				except:
					j.lValues[self.sMetricID]='VEA_other'
					j.lValues[self.sClass]=-1

		l_lMetric = {}

		cCursorDB.close()
		cDB.close()

#--------------------------------------------------------------- Decay value ---
	def getDecayedValue(self,p_iTime):
		if p_iTime>self.lDecayValues[-1][0]*self.iDecayFactor:
			return 0
		else:
			l_iIdx = (int(p_iTime) / (604800*self.iDecayFactor))
			return self.lDecayValues[l_iIdx][1]

#===============================================================================
#						Authors filter mixer unit
#===============================================================================
class MixerUnit(sta_gl_mixerunit.MixerUnit):

	def __init__(self, *args, **kwds):
		sta_gl_mixerunit.MixerUnit.__init__(self, *args, **kwds)
		self.iAttrType		= 0
		self.iDrawType		= 0
