#*****************************************************************************************
# Name			: Modularity metrics (fan-in/out, cohesion)
# File			: pextModularity.py
# Description	: A plugin for visualizing modularity of files
# Version		: 1.0
# Copyright SolidSource B.V.
#*****************************************************************************************

#------------------------------------------------------------ Global imports ---
import wx
from OpenGL.GL import *
from OpenGL.GLUT import *
from OpenGL.GLU import *

import pysqlite2.dbapi2 as sqlite

#------------------------------------------------------------- Local imports ---
import sta_globals
import sta_plugin
import sta_utils
import sta_gl_list
import sta_gl_mixerunit

#===============================================================================
#                                             Modularity plugin
#===============================================================================
class Plugin(sta_plugin.Plugin):

   def __init__(self,*args, **kwds):
      sta_plugin.Plugin.__init__(self,*args, **kwds)
      self.sClass                   = 'pextModularity' # Name of the plugin file
      self.sName                    = 'Modularity'     # Name to display in a list
      self.sVersion                 = '1.0'
      self.sMetricID                = 'MOD'
      self.bHasMetricValue          = True

      self.sTable1                   = 'D_TFS_Deps_VersionedClasses'  # Name of the table where metric is stored
      self.sTable2                   = 'D_TFS_Deps_VersionedMethods'
      self.lTables.append(self.sTable1)          # Append to this list all tables that are used
      self.lTables.append(self.sTable2)

      #----------------------------------------------------- Specific processing

      self.iLevel                   = 1   # The interval level
      self.iLevelFactor             = 10  # The level size factor (CFG)
      self.iIntervals               = 10  # Number of intervals (CFG)
      self.iNrLines                 = 30  # Line group size (CFG)
      self.iNrLevels                = 10  # Number of slider levels (CFG)
      self.undefinedText            = None

      self.iIntervalSize            = (self.iLevel * self.iLevelFactor) / self.iIntervals

      self.lMcCabesColors           = []  # Color map (CFG)

#-------------------------------------------------------------------------------
#                                                                 Filter GUI
#-------------------------------------------------------------------------------

#----------------------------------------------------------------- Build GUI ---
   def buildFilterGUI(self,pMasterGUI):

      self.cGUIMaster = pMasterGUI

      self.cSizer = wx.BoxSizer(wx.VERTICAL)
      l_cConfigSizer = wx.BoxSizer(wx.HORIZONTAL)

      l_cLabelSizer = wx.BoxSizer(wx.HORIZONTAL)
      l_sLabel = wx.StaticText(pMasterGUI.project_filters, -1, '  '+self.sName)
      self.g_txtMetric = wx.StaticText(pMasterGUI.project_filters, -1, style = wx.ALIGN_RIGHT )
      l_cLabelSizer.Add(l_sLabel, 0, wx.ADJUST_MINSIZE, 0)
      l_cLabelSizer.Add(self.g_txtMetric, 0, wx.ALIGN_RIGHT, 0)

      self.cGUIMode = wx.ComboBox(pMasterGUI.project_filters, -1, choices=[], style=wx.CB_DROPDOWN|wx.CB_READONLY)
      self.cGUIMode.Append('Fan-in (class sum)')
      self.cGUIMode.Append('Fan-in (class avg)')
      self.cGUIMode.Append('Fan-in (method sum)')
      self.cGUIMode.Append('Fan-in (method avg)')
      self.cGUIMode.Append('Fan-out (class sum)')
      self.cGUIMode.Append('Fan-out (class avg)')
      self.cGUIMode.Append('Fan-out (method sum)')
      self.cGUIMode.Append('Fan-out (method avg)')
      self.cGUIMode.Append('Cohesion (class max)')
      self.cGUIMode.Append('Cohesion (class min)')
      self.cGUIMode.Append('Cohesion (class avg)')
      self.cGUIMode.Append('Cohesion (method max)')
      self.cGUIMode.Append('Cohesion (method min)')
      self.cGUIMode.Append('Cohesion (method avg)')
      self.cGUIMode.SetSelection(0)

      self.cGUIList = sta_gl_list.wxColorCheckListBox(self.sName,self.sClass,pMasterGUI.project_filters, style=wx.BORDER_SUNKEN|wx.LB_MULTIPLE|wx.LB_SORT)
      self.cGUIList.lObservers.append(self)

      for i in range(self.iIntervals):
         self.lMcCabesColors.append(sta_utils.getRainbowColor(float(i+1)/self.iIntervals))
      self.lMcCabesColors.append((0.2,0.2,0.2))
      self.cGUIList.SetData(self.lMcCabesColors)

      self.cGUIScale = wx.Slider(pMasterGUI.project_filters, -1, self.iLevel, 1,self.iNrLevels)
      self.cGUIConfig = wx.Button(pMasterGUI.project_filters, -1, 'cfg')
      self.cGUIConfig.Disable()
      self.cGUIScale.SetOwnBackgroundColour(wx.Colour(150,150,180))

      self.cGUIList.AddMetric("# files",self.MetricIntervalFrequency)

      self.cGUIList.Bind(wx.EVT_LISTBOX,self.cbListSelect)
      self.cGUIScale.Bind(wx.EVT_SLIDER,self.cbLevelSelect)
      self.cGUIMode.Bind(wx.EVT_COMBOBOX,self.cbModeSelect)

      self.cGUIList.SetMinSize((140,-1))
      self.cGUIConfig.SetMinSize((30,15))
      self.cGUIScale.SetMinSize((-1,15))

      self.cSizer.Add(l_cLabelSizer, 0, wx.EXPAND, 0)
      self.cSizer.Add(self.cGUIMode, 0, wx.EXPAND, 0)
      self.cSizer.Add(self.cGUIList, 1, wx.EXPAND|wx.ADJUST_MINSIZE, 0)
      l_cConfigSizer.Add(self.cGUIScale, 1, wx.EXPAND, 0)
      l_cConfigSizer.Add(self.cGUIConfig, 0, wx.ADJUST_MINSIZE, 0)
      self.cSizer.Add(l_cConfigSizer, 0, wx.EXPAND, 0)

      pMasterGUI.cListsSizer_Project.Add(self.cSizer, 0, wx.EXPAND, 0)

      self.showFilterGUI(False)
      self.doBindings()

#-------------------------------------------------------------------------------
#						         Filter GUI callback
#-------------------------------------------------------------------------------

   def cbListSelect_aux(self, files):

      if len(self.lIdx) == 0:
        for i in files:
           for j in i.lRevs:
               if j.lValues[self.sMetricID] is None:
                  j.lValues[self.sClass] = self.iIntervals
               else:
                  j.lValues[self.sClass] = int(j.lValues[self.sMetricID] / self.iIntervalSize)
                  if j.lValues[self.sClass] < 0:
                     j.lValues[self.sClass] = -1
                  if j.lValues[self.sClass] >= self.iIntervals:
                        j.lValues[self.sClass] = self.iIntervals-1
      else:
        for i in files:
           for j in i.lRevs:
               if j.lValues[self.sMetricID] is None:
                  l_iIdx = self.iIntervals
               else:
                  l_iIdx = int(j.lValues[self.sMetricID] / self.iIntervalSize)
                  if l_iIdx >= self.iIntervals:
                     l_iIdx = self.iIntervals-1
               j.lValues[self.sClass] = -1
               if l_iIdx in self.lIdx:
                 j.lValues[self.sClass] = l_iIdx

   def getSelectedIndexes(self):
      l_lsMcCabe = self.cGUIList.GetSelections()
      self.clearSelection()
      for i in range(self.iIntervals):
         if i<self.iIntervals-1:
            l_sItem = str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1)
         else:
            l_sItem = str(i*self.iIntervalSize)+' - ...'
         if l_sItem in l_lsMcCabe:
            self.lIdx.append(i)
      if self.undefinedText in l_lsMcCabe:
         self.lIdx.append(self.iIntervals)
      self.bSelected = len(self.lIdx)>0

   def cbLevelSelect(self,evt):
      self.unloadMetric()
      self.clearFilterGUI()
      self.iLevel = self.cGUIScale.GetValue()
      self.iIntervalSize = (self.iLevel * self.iLevelFactor) / self.iIntervals

      for i in range(self.iIntervals):
         if (i < self.iIntervals-1):
            self.cGUIList.Append(str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1),i)
         else:
            self.cGUIList.Append(str(i*self.iIntervalSize)+' - ...',i)
         self.lData.append(i)

      if self.undefinedText is not None:
         self.cGUIList.Append(self.undefinedText, self.iIntervals);
         self.lData.append(self.iIntervals)

      self.loadMetric()

      sta_plugin.Plugin.updateCanvases(self)


   def cbModeSelect(self,evt):
      self.cGUIList.DeselectAll()
      self.unloadMetric()
      self.loadMetric()

      sta_plugin.Plugin.updateCanvases(self)

#-------------------------------------------------------------------------------
#                             Filter GUI metrics
#-------------------------------------------------------------------------------
#--------------------------------------------------------- Number of commits ---
   def MetricIntervalFrequency(self,sName):

      l_iLevel = -1
      if sName == self.undefinedText:
         l_iLevel = self.iIntervals
      else:
         for i in range(self.iIntervals):
            if i<self.iIntervals-1:
               l_sItem = str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1)
            else:
               l_sItem = str(i*self.iIntervalSize)+' - ...'
            if l_sItem == sName:
               l_iLevel = i
               break

      l_iNr = 0
      if (l_iLevel >=0):
         if (l_iLevel <self.iIntervals-1):
            for i in sta_globals.lSelectedFiles:
               for j in i.lRevs:
                  if (j.lValues[self.sMetricID] >= l_iLevel*self.iIntervalSize) and (j.lValues[self.sMetricID] < (l_iLevel+1)*self.iIntervalSize):
                     l_iNr = l_iNr+1
                     break
         elif l_iLevel == self.iIntervals:
            for i in sta_globals.lSelectedFiles:
               for j in i.lRevs:
                  if j.lValues[self.sMetricID] is None:
                     l_iNr = l_iNr+1
                     break
         else:
            for i in sta_globals.lSelectedFiles:
               for j in i.lRevs:
                  if (j.lValues[self.sMetricID] >= l_iLevel*self.iIntervalSize):
                     l_iNr = l_iNr+1
                     break
      return l_iNr

#-------------------------------------------------------------------------------
#                                                                Data management
#-------------------------------------------------------------------------------
#--------------------------------------------------------------- Load metric ---
   def loadMetric(self):

      l_sDBPath = sta_globals.cSTAPrj.storagePath()+"/history.db"

      if not sta_utils.isValidDB(l_sDBPath):
         print 'DATA: no history information found'
         return

      cDB = sqlite.connect(l_sDBPath)
      cCursorDB = cDB.cursor()

      if not sta_utils.isValidTable(cCursorDB,self.sTable1):
         print 'DATA: no modularity information found'
         cCursorDB.close()
         cDB.close()
         return

      self.clearSelection()
      self.undefinedText = None
      l_lMetric = {}

      #------------------------------------------ Fan-in (class sum)
      if (self.cGUIMode.GetSelection() == 0):

         sCmd = "select ID, SUM(FI) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-in (class avg)
      if (self.cGUIMode.GetSelection() == 1):

         sCmd = "select ID, AVG(FI) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-in (method sum)
      if (self.cGUIMode.GetSelection() == 2):

         sCmd = "select ID, SUM(FI) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-in (method avg)
      if (self.cGUIMode.GetSelection() == 3):

         sCmd = "select ID, AVG(FI) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-out (class sum)
      if (self.cGUIMode.GetSelection() == 4):

         sCmd = "select ID, SUM(FO) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-out (class avg)
      if (self.cGUIMode.GetSelection() == 5):

         sCmd = "select ID, AVG(FO) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-out (method sum)
      if (self.cGUIMode.GetSelection() == 6):

         sCmd = "select ID, SUM(FO) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Fan-out (method avg)
      if (self.cGUIMode.GetSelection() == 7):

         sCmd = "select ID, AVG(FO) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Cohesion (class max)
      if (self.cGUIMode.GetSelection() == 8):

         sCmd = "select ID, MAX(Cohesion) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Cohesion (class min)
      if (self.cGUIMode.GetSelection() == 9):

         sCmd = "select ID, MIN(Cohesion) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Cohesion (class avg)
      if (self.cGUIMode.GetSelection() == 10):

         sCmd = "select ID, AVG(Cohesion) from "+self.sTable1+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Cohesion (method max)
      if (self.cGUIMode.GetSelection() == 11):

         sCmd = "select ID, MAX(Cohesion) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Cohesion (method min)
      if (self.cGUIMode.GetSelection() == 12):

         sCmd = "select ID, MIN(Cohesion) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #------------------------------------------ Cohesion (method avg)
      if (self.cGUIMode.GetSelection() == 13):

         sCmd = "select ID, AVG(Cohesion) from "+self.sTable2+" group by ID"
         cCursorDB.execute(sCmd)
         for i in cCursorDB:
            l_lMetric[i[0]]=i[1]

      #---------------------------------------- General ---
      for i in sta_globals.lFiles:
         for j in i.lRevs:
            try:
               l_iMCB = l_lMetric[j.iDBid]
               j.lValues[self.sMetricID]=l_iMCB

               if l_iMCB is None:
                  j.lValues[self.sClass] = self.iIntervals
               else:
                  j.lValues[self.sClass]= int(l_iMCB / self.iIntervalSize)
                  if j.lValues[self.sClass] < 0:
                     j.lValues[self.sClass] = -1
                  if j.lValues[self.sClass] >= self.iIntervals:
                     j.lValues[self.sClass] = self.iIntervals-1
            except:
               j.lValues[self.sMetricID]= -1
               j.lValues[self.sClass]= -1
      l_lMetric = {}

      # Display intervals
      for i in range(self.iIntervals):
         if (i < self.iIntervals-1):
            self.cGUIList.Append(str(i*self.iIntervalSize)+' - '+str((i+1)*self.iIntervalSize-1),i)
         else:
            self.cGUIList.Append(str(i*self.iIntervalSize)+' - ...',i)
         self.lData.append(i)

      if self.undefinedText is not None:
         self.cGUIList.Append(self.undefinedText, self.iIntervals);
         self.lData.append(self.iIntervals)
      #----------------------------------------------------

      cCursorDB.close()
      cDB.close()

#===============================================================================
#						Authors filter mixer unit
#===============================================================================
class MixerUnit(sta_gl_mixerunit.MixerUnit):

	def __init__(self, *args, **kwds):
		sta_gl_mixerunit.MixerUnit.__init__(self, *args, **kwds)

		self.iAttrType		= 0
		self.iDrawType		= 1
