import sta_globals
import os
import os.path
import sta_utils
import math
import sys
import time
import pysqlite2.dbapi2 as sqlite

#//////////////////////////////////////////////////////////////////////////////////////////
#														Cluster Class
#//////////////////////////////////////////////////////////////////////////////////////////

class CCluster:
	def __init__(self,*args,**kwds):
		self.iType			= 1				# 0 - node, 1 - cluster
		self.lData			= []			# Clusters value list
		self.fMeasure		= 0.0			# The distance characteristic to this cluster
		self.pElement1		= 0				# Child 1 (agglomerative clustering)
		self.pElement2		= 0				# Child 2 (agglomerative clustering)
		self.pNode			= 0				# Link to file object
		self.iFileSize		= 1				# Number of clustered files (child leafs)
		self.bSkip			= False			# Skip from clustering

#//////////////////////////////////////////////////////////////////////////////////////////
#												Cluster Layout Class
#//////////////////////////////////////////////////////////////////////////////////////////
class CClusterLayout:			   
	def __init__(self,*args,**kwds): 
		self.iLayers	= 0
		self.lLayer		= []
	
	def SetUpLayers(self,p_iNrLayers):		
		self.iLayers	= p_iNrLayers
		self.lLayer		= []
		for i in range(p_iNrLayers):
			self.lLayer.append([])

#//////////////////////////////////////////////////////////////////////////////////////////
#												Clusters Engine Class
#//////////////////////////////////////////////////////////////////////////////////////////

class CClusterEngine:
#==========================================================================================
#																		Construction       
#==========================================================================================
	def __init__(self,*args,**kwds):
		self.sVersion			= '1.0'					# The current version of the engine
		self.sName				= 'My_name'				# Cluster specific name (descriptive name)
		self.sID				= 'My_ID'				# Cluster specific name (file name)
		self.cClusterRoot		= 0						# Root of the computed cluster hierarchy
		self.cLayout			= CClusterLayout()		# Layout dendogram		
		self.lLayout			= []					# List of cushion heights lists
		self.lLevelLayout		= []					# List of cushion heights per detail level
		self.fMaxDistance		= 0						# Maximum distance between two clusters		
		self.fMaxValue			= 0						# Maximum value of a cluster
		self.fMinValue			= -1					# Minimum value of a cluster
		self.fDeltaValue		= 0						# Interval for the cluster values
		self.fRefValue			= 0						# Reference value for normalization (0 if none available)
		
		self.bHasGUI			= False					# True if it allows settings GUI
		self.cGUI				= None					# The GUI
		self.cParentGUI			= None					# The parent GUI
		
		#----------------------------------------------------------------------------------
		self.sClusterSave		= ""					# Used to save clusters
		
		self.lFiles				= []					# List of files
		self.iFiles				= 0						# Number of files clustered
		self.lClusterAux		= []					# Tmp store clustered files
		self.dClusters			= {}					# Set of work clusters
		self.dSkipClusters		= {}					# Set of clusters to skip from the distance calculation
		self.iClusterID			= 0						# Cluster ID
		
		self.iSelectedCLevel	= 0						# Global selected output cluster level (by normal layout)
		self.iLayoutLevel		= 0						# Used to compute layer cushions in normal mode
		
		self.bIsClustering		= False					# True when clustering is in progress
		self.bDoClustering		= True					# False when clustering operation is canceled
		self.gProgress			= 0						# Used when a progress bar can be provided
		self.iClusterPos		 = 0					# Current position in the progrress bar
		
		self.fResolution		= 0						# Current resolution
		self.iStartTime			= 0
		self.iEndTime			= 0
		
	#------------------------------------------------------------------ Det GUI ---
	def SetGUI(self,p_cGUI):
		if (p_cGUI != None) and (self.bHasGUI):
			self.cParentGUI = p_cGUI
	#-------------------------------------------------------------------- Reset ---
	def Reset(self):
		self.cClusterRoot		= 0
		
	#-------------------------------------------------------------------- Clear ---
	def Clear(self,p_sPath):
		if (sta_globals.bFolderSelection):
			l_sPath = p_sPath+"\\.STA.clusters."+self.sID+".dat"
			if os.path.exists(l_sPath):
				os.remove(l_sPath) 
				
		if self.HasClusters():
			self.cClusterRoot	= 0
			
	#------------------------------------------------------------------- Cancel ---
	def Cancel(self):
		self.bDoClustering = False
	
#==========================================================================================
#																		Print/Save clusters
#==========================================================================================
	def Print(self,p_cCluster):
		
		l_lStack			= list()
		l_lStackPost		= list()
		
		l_lStack.append((p_cCluster,0))
		
		while len(l_lStack) > 0:
			
			l_cCluster,l_iLevel = l_lStack.pop()
			
			# Do post-processing
			if len(l_lStackPost)>0:
				while l_lStackPost[-1][1] >= l_iLevel:
					l_cPostCluster,l_iPostLevel = l_lStackPost.pop()
					self.sClusterSave =  self.sClusterSave + " " +str(l_cPostCluster.fMeasure) + ")"
			
			# Do pre-processing
			l_lStackPost.append((l_cCluster,l_iLevel))
			self.sClusterSave =  self.sClusterSave + "("
			
			# Do processing
			if (l_cCluster.iType == 1):
				l_lStack.append((l_cCluster.pElement1,l_iLevel+1))
				l_lStack.append((l_cCluster.pElement2,l_iLevel+1))
			else:
				self.sClusterSave =  self.sClusterSave + l_cCluster.pNode.sPath
				
		#-- Finish up post stack
		while len(l_lStackPost)>0:
			l_cPostCluster,l_iPostLevel = l_lStackPost.pop()
			self.sClusterSave =  self.sClusterSave + " " +str(l_cPostCluster.fMeasure) + ")"
		
	def Save(self,p_sPath):
		
		if self.cClusterRoot == 0:
			return
		l_sPath = p_sPath+"\\.STA.clusters."+self.sID+".dat"
		
		if not(os.path.exists(p_sPath)):
			os.makedirs(p_sPath)
		
		self.sClusterSave = ""
		self.Print(self.cClusterRoot)
		l_cFileHandle=open(l_sPath,"w")
		l_cFileHandle.write(self.sClusterSave)
		l_cFileHandle.close()
		self.sClusterSave = ""

#==========================================================================================
#																		Load clusters      
#==========================================================================================
	
	#------------------------------------------ Load cluster ---
	def pLoadCluster(self,p_sLoad,p_lFiles):
		
		l_lStackInput	= list()
		l_lStackInput.append(p_sLoad)
		
		# Stack
		l_cCluster		= CCluster()
		l_lStack		= list()
		l_lStack.append((l_cCluster,0))
		
		# Post stack
		l_lStackPost	= list()
		
		l_cRootCluster	= l_cCluster
		
		while len(l_lStackInput) > 0:
			
			l_cCluster,l_iLevel		= l_lStack.pop()
			l_sLoad					= l_lStackInput.pop()
			
			#--- Do post-processing
			if len(l_lStackPost)>0:
				while l_lStackPost[-1][1] >= l_iLevel:
					l_cPostCluster,l_iPostLevel = l_lStackPost.pop()
					if l_cPostCluster.iType == 1:
						l_cPostCluster.iFileSize = l_cPostCluster.pElement1.iFileSize + l_cPostCluster.pElement2.iFileSize
			
			#--- Do pre-processing
			l_lStackPost.append((l_cCluster,l_iLevel))
			
			#--- Do processing
			(l_sElement1,l_sElement2,l_fMeasure) = sta_utils.splitInput(l_sLoad[1:-1])
			
			l_cCluster.fMeasure = float(l_fMeasure)
			
			if len(l_sElement2) == 0:
				l_cCluster.iType = 0
				for i in p_lFiles:
					if i.sPath == l_sElement1:
						l_cCluster.pNode = i
						break
			else:
				l_cCluster.iType		= 1
				
				l_cCluster.pElement1	= CCluster()
				l_lStack.append((l_cCluster.pElement1,l_iLevel+1))
				l_lStackInput.append(l_sElement1)
				
				l_cCluster.pElement2	= CCluster()
				l_lStack.append((l_cCluster.pElement2,l_iLevel+1))
				l_lStackInput.append(l_sElement2)
			
		#-- Finish up post stack
		while len(l_lStackPost)>0:
			l_cPostCluster,l_iPostLevel = l_lStackPost.pop()
			if l_cPostCluster.iType == 1:
				l_cPostCluster.iFileSize = l_cPostCluster.pElement1.iFileSize + l_cPostCluster.pElement2.iFileSize
		
		return l_cRootCluster
	
	#------------------------------------- Load clustering ---
	def Load(self,p_sPath,p_lFiles):
		
		self.iFiles = len(p_lFiles)
		l_sPath = p_sPath+"\\.STA.clusters."+self.sID+".dat"
		if os.path.exists(l_sPath):
			l_cFileHandle=open(l_sPath,"r")
			l_sLine = l_cFileHandle.readline()
			l_cFileHandle.close()
			self.cClusterRoot = self.pLoadCluster(l_sLine,p_lFiles)
			self.GetLayout()
			return True
		return False
#==========================================================================================
#																				Access     
#==========================================================================================
	#------------------------------------------------------------- Has clusters ---
	def HasClusters(self):
		return self.cClusterRoot != 0
	#---------------------------------------------------------------- Get Files ---
	def GetFiles(self):
		self.lFiles = []
		self.pSortCluster(self.cClusterRoot)
		return self.lFiles
#==========================================================================================
#																				Compute
#==========================================================================================
	
	#------------------------------------------------------------------------------
	#																Compute value  
	#------------------------------------------------------------------------------
	def pComputeMinMaxValues(self,p_cCluster):
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			l_cCluster = l_lStack.pop()
			
			if (l_cCluster.iType == 1): 
				l_lStack.append(l_cCluster.pElement1)
				l_lStack.append(l_cCluster.pElement2)
			if (self.fMaxValue < l_cCluster.fMeasure) and (l_cCluster.fMeasure != sys.maxint):
				self.fMaxValue = l_cCluster.fMeasure
			if (self.fMinValue < 0) or (self.fMinValue > l_cCluster.fMeasure):
				self.fMinValue = l_cCluster.fMeasure
			
	def ComputeValues(self):
		if (self.fRefValue == 0):
			self.fMaxValue = 0
			self.fMinValue = -1
			self.pComputeMinMaxValues(self.cClusterRoot)
			self.fDeltaValue = self.fMaxValue
		else:
			self.fDeltaValue = self.fRefValue
		self.pNormalizeValue(self.cClusterRoot)
		self.GetLayout()
	
	#------------------------------------------------------------------------------
	#																Normalize value
	#------------------------------------------------------------------------------
	def pNormalizeValue(self,p_cCluster):
		
		if self.fDeltaValue == 0:
			return
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			l_cCluster = l_lStack.pop()
			l_cCluster.fMeasure = float(l_cCluster.fMeasure) / self.fDeltaValue
			if (l_cCluster.fMeasure > 1):
				l_cCluster.fMeasure = 1.0
			if (l_cCluster.iType == 1):
				l_lStack.append(l_cCluster.pElement1)
				l_lStack.append(l_cCluster.pElement2)
		
	#------------------------------------------------------------------------------
	#															Prepare clusters   
	#------------------------------------------------------------------------------
	def pPrepareClusters(self):
		
		#TODO: Implement specific cluster preparation if necessary
		pass
		
	#------------------------------------------------------------------------------
	#																Compact cluster
	#------------------------------------------------------------------------------
	def pCompactCluster(self,p_cCluster):			
		
		#TODO: Implement specific cluster compactation if necessary
		pass
		
	#------------------------------------------------------------------------------
	#														Construct new cluster  
	#------------------------------------------------------------------------------
	def pCheckSkip(self,p_cCluster):
		#TODO: Implement psecific cluster skip criteria
		
		p_cCluster.bSkip = False
		
	def pAppendSkipedClusters(self):
		for l_cCluster in self.dSkipClusters.values():
			l_cNew				= CCluster()
			l_cNew.pElement1	= self.cClusterRoot
			l_cNew.pElement2	= l_cCluster
			l_cNew.fMeasure		= self.cClusterRoot.fMeasure
			l_cNew.iFileSize	= self.cClusterRoot.iFileSize + 1
			self.cClusterRoot	= l_cNew
		
		self.dSkipClusters.clear()
		
	def pConstructCluster(self,p_cFile):
		
		#TODO: Implement specific cluster creation
		
		l_cNew	= CCluster()
		self.pCheckSkip(l_cNew)
		self.pAppendElement(l_cNew)
		
	#------------------------------------------------------------------------------
	#																Merge clusters 
	#------------------------------------------------------------------------------
	def pMergeClusters(self,p_fMeasure,p_iP1,p_iP2):
		
		#TODO: Implement specific merge clustering
		
		l_cNew	= CCluster()
		return l_cNew
		
	#------------------------------------------------------------------------------
	#																Remove element 
	#------------------------------------------------------------------------------
	def pRemoveElement(self,p_iPi,p_cDB,p_cCursorDB):
		
		del self.dClusters[p_iPi]
		
		p_cCursorDB.execute("delete from Dist where ID1=?", (p_iPi,) )
		p_cCursorDB.execute("delete from Dist where ID2=?", (p_iPi,) )
		
	def pAppendElement(self,p_cNew):
		self.iClusterID +=1
		if p_cNew.bSkip:
			self.dSkipClusters[self.iClusterID] = p_cNew
		else:
			self.dClusters[self.iClusterID] = p_cNew
	#------------------------------------------------------------------------------
	#													 	The distance function  
	#------------------------------------------------------------------------------
	def pDelta(self,p_cCluster1,p_cCluster2):
		# TODO: Implement specific distance
		pass
	
	#------------------------------------------------------------------------------
	#													 Find maximum similarity   
	#------------------------------------------------------------------------------
	
	def pFindDistances(self,p_iCluster,p_cDB,p_cCursorDB):
		
		l_cCluster1 = self.dClusters[p_iCluster]
		sCmd = "insert into Dist(ID1,ID2,Dst) values (?,?,?)"
		
		l = list()
		for j in self.dClusters.keys():
			if (p_iCluster <= j):
				continue
			
			l_cCluster2 = self.dClusters[j]
			l_fDist = self.pDelta(l_cCluster1,l_cCluster2)
			l.append( (p_iCluster,j,l_fDist,) )
		p_cCursorDB.executemany(sCmd, l )
			
		if self.gProgress != 0:
			self.iClusterPos = self.iClusterPos + 1
			sta_globals.GUI.MainThreadExec('sta_globals.cClusterEngine.gProgress.SetValue($STA$)',lParams=[self.iClusterPos])
	
	def pFindMinDistance(self,p_cDB,p_cCursorDB):
		
		l_fMin	= -1
		l_iP1	= 0
		l_iP2	= 0
		
		sCmd = "select ID1,ID2,Dst from Dist where Dst=(select min(Dst) from Dist)"
		p_cCursorDB.execute(sCmd)
		
		for i in p_cCursorDB:
			l_iP1	= i[0]
			l_iP2	= i[1]
			l_fMin	= i[2]
			break
		
		return l_fMin,l_iP1,l_iP2
		
	#------------------------------------------------------------------------------
	#														Compute Agglomerative  
	#------------------------------------------------------------------------------
	def pComputeAgglomerative(self,p_lFiles):   
		if (self.bIsClustering):
			return
		
		self.bIsClustering = True
		l_cDB,l_cCursorDB = DBStart(':memory:')
		
		#------------- Create initial work set (lClusters) ---
		self.dClusters.clear()
		self.dSkipClusters.clear()
		
		for i in p_lFiles:
			self.pConstructCluster(i)
		self.pPrepareClusters()
		
		#--------------------------- Initiate progress bar ---
		if self.gProgress != 0:
			l_iRange = 2*len(self.dClusters.keys())
			sta_globals.GUI.MainThreadExec('sta_globals.cClusterEngine.gProgress.SetRange($STA$)',lParams=[l_iRange])
			self.iClusterPos = 0
			sta_globals.GUI.MainThreadExec('sta_globals.cClusterEngine.gProgress.SetValue($STA$)',lParams=[self.iClusterPos])
			
		#===========================================================================
		l_bInit = True
		while (len(self.dClusters.keys())>1) and (self.bDoClustering):
			
			#------------------------------ first iteration compute all distances ---
			if l_bInit:
				l_bInit = False
				for i in self.dClusters.keys():				# Find all distances
					self.pFindDistances(i,l_cDB,l_cCursorDB)
					if not (self.bDoClustering):
						break
			#------------------------------------------- Generate the new cluster ---
			sta_utils.dprint('----')
			l_iTime1 = time.time()
			l_fMin,l_iP1,l_iP2 = self.pFindMinDistance(l_cDB,l_cCursorDB)
			l_iTime2 = time.time()
			sta_utils.dprint(l_iTime2-l_iTime1)
			
			l_cNew = self.pMergeClusters(l_fMin,l_iP1,l_iP2)# generate new element from element 1 and 2
			self.pAppendElement(l_cNew)						# add new element
			l_iTime3 = time.time()
			sta_utils.dprint(l_iTime3-l_iTime2)
			
			self.pRemoveElement(l_iP1,l_cDB,l_cCursorDB)	# remove element 1
			self.pRemoveElement(l_iP2,l_cDB,l_cCursorDB)	# remove element 2
			l_iTime4 = time.time()
			sta_utils.dprint(l_iTime4-l_iTime3)
			
			self.pFindDistances(self.iClusterID,l_cDB,l_cCursorDB) # compute distances for the new element
			l_iTime5 = time.time()
			sta_utils.dprint(l_iTime5-l_iTime4)
			
			if not (self.bDoClustering):
				break
				
		#===========================================================================
		sta_utils.dprint('Done')
		
		if self.gProgress != 0:
			sta_globals.GUI.MainThreadExec('sta_globals.cClusterEngine.gProgress.SetValue(0)')
			
		self.bIsClustering = False
		if (self.bDoClustering):
			self.cClusterRoot = self.dClusters.values()[0]
			self.pAppendSkipedClusters()
			self.pClean(self.cClusterRoot)
		else:
			self.cClusterRoot = 0
			self.bDoClustering = True
		
		self.dClusters.clear()
		self.dSkipClusters.clear()
		
		DBStop(l_cDB,l_cCursorDB,':memory:')
		
	#------------------------------------------------------------------------------
	#																		Compute
	#------------------------------------------------------------------------------
	def Compute(self,p_lFiles):
		if (sta_globals.iStartFocusInterval < 0):
			self.iStartTime = sta_globals.iSelectedFilesStartTime
		else:
			self.iStartTime = sta_globals.iStartFocusInterval
		
		if (sta_globals.iEndFocusInterval < 0):
			self.iEndTime = sta_globals.iSelectedFilesEndTime
		else:
			self.iEndTime = sta_globals.iEndFocusInterval
		
		self.iFiles = len(p_lFiles)
		self.bDoClustering = True
		self.pComputeAgglomerative(p_lFiles)
		self.ComputeValues()
		sta_utils.dprint(self.pScore())
		
	def pScore(self):
		# TODO: Implement specific distance
		return 'Not available'
		
#==========================================================================================
#																				Layout
#==========================================================================================
# EXL
#-----------
	#---------------------------------------- Get cluster level display cushions ---
	def pGetLayerLayout(self,p_cCluster,p_iLevel):
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			l_cCluster = l_lStack.pop()
			
			# Compute condititon to descend on the tree
			l_bCondition = l_cCluster.fMeasure > p_iLevel
			
			if (l_bCondition):
				if (l_cCluster.iType == 1):
					l_lStack.append(l_cCluster.pElement1)
					l_lStack.append(l_cCluster.pElement2)
				else:							
					self.lLevelLayout.append((self.pGetClusterDrawSize(l_cCluster),l_cCluster.fMeasure,l_cCluster.bSkip))
			else:					  
				self.lLevelLayout.append((self.pGetClusterDrawSize(l_cCluster),l_cCluster.fMeasure,l_cCluster.bSkip)) 
		
	#---------------------------------------------- Get cluster display cushions ---
	def GetLayout(self):
		self.lLayout = []
		for i in range(1,101):
			self.lLevelLayout = []  
			l_fValue = float(101-i)/100
			self.pGetLayerLayout(self.cClusterRoot,l_fValue)	  
			self.lLayout.append(self.lLevelLayout)
	#---------------------------------------------------------------------- Sort ---
	def Sort(self):
		self.lFiles = []
		self.pSortCluster(self.cClusterRoot)
		return self.lFiles
						  
#==========================================================================================
#																		Sort Preserve
#==========================================================================================
# EXL
#---------	
	def pSortPreserve(self,iLevel,p_cCluster,funcSort):
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			
			l_cCluster = l_lStack.pop()
			
			bCondition = (l_cCluster.fMeasure > iLevel)
			
			if (bCondition):
				if (l_cCluster.iType == 1):
					l_lStack.append(l_cCluster.pElement1)
					l_lStack.append(l_cCluster.pElement2)
				else:
					self.lFiles.append(l_cCluster.pNode)	# Append the file the result
			else:
				self.lClusterAux = []
				self.pGetFiles(l_cCluster)					# Find list of files on a level
				self.lClusterAux.sort(funcSort)				# Sort these list
				if (sta_globals.bSortInvert):
					self.lClusterAux.reverse()
				self.lFiles.extend(self.lClusterAux)		# Append it to the result
			
	def SortPreserve(self,funcSort):
		
		self.lFiles = []
		self.pSortPreserve(1-self.fResolution,self.cClusterRoot,funcSort)
		
		return self.lFiles

#==========================================================================================
#																				Utils
#==========================================================================================
	
	#--------------------------------------------------- Get files in a cluster ---	
	def pGetFiles(self,p_cCluster):
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			
			l_cCluster = l_lStack.pop()
			
			if (l_cCluster.iType != 1):
				self.lClusterAux.append(l_cCluster.pNode)
			else:
				l_lStack.append(l_cCluster.pElement1)
				l_lStack.append(l_cCluster.pElement2)
			
	#----------------------------------------- Get number of files in a cluster ---
	def pGetClusterSize(self,cCluster):
		
		l_iReturn = 0
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			
			l_cCluster = l_lStack.pop()
			
			if (cCluster.iType != 1):
				l_iReturn += 1
			else:
				l_lStack.append(l_cCluster.pElement1)
				l_lStack.append(l_cCluster.pElement1)
		return l_iReturn
			
   #------------ Get number of files on a level up to the global selected level ---
	def pGetLevelSize(self,p_cCluster,p_iLevel):
		
		l_iReturn = 0
		l_lStack = list()
		l_lStack.append(p_cCluster,p_iLevel)
		while len(l_lStack) > 0:
			
			l_cCluster,l_iLevel = l_lStack.pop()
			
			if (l_iLevel < self.iSelectedCLevel):
				if (l_cCluster.iType == 1):			  
					l_lStack.append((l_cCluster.pElement1,l_iLevel+1))
					l_lStack.append((l_cCluster.pElement2,l_iLevel+1))
				else:
					l_iReturn += 1
			else:
				l_iReturn += 1
				
		return l_iReturn
			
   #---- Get number of files on a level up to the global selected level (Layout version) ---
	def pGetLayoutLevelSize(self,p_cCluster,p_iLevel):
		
		l_iReturn = 0
		l_lStack = list()
		l_lStack.append(p_cCluster,p_iLevel)
		while len(l_lStack) > 0:
			
			l_cCluster,l_iLevel = l_lStack.pop()
			
			if (l_iLevel < self.iLayoutLevel):
				if (l_cCluster.iType == 1):
					l_lStack.append((l_cCluster.pElement1,l_iLevel+1))
					l_lStack.append((l_cCluster.pElement2,l_iLevel+1))
				else:
					l_iReturn +=1
			else:
				l_iReturn += 1 
		return l_iReturn
	#--------------------------------------------- Get the Y size of a cluster ---
	def pGetClusterDrawSize(self,p_cCluster):
		if (p_cCluster.iType == 1):
			return p_cCluster.iFileSize
		else:
			return 1
			
	#-------------------------------------------------------------------- Clean ---
	def pClean(self,p_cCluster):
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			l_cCluster = l_lStack.pop()
			l_cCluster.lData = []
			if (l_cCluster.iType == 1):
				l_lStack.append(l_cCluster.pElement1)
				l_lStack.append(l_cCluster.pElement2)
				
	#----------------------------------------------------------- Sort the files ---
	def pSortCluster(self,p_cCluster):
		
		l_lStack = list()
		l_lStack.append(p_cCluster)
		while len(l_lStack) > 0:
			l_cCluster = l_lStack.pop()
			
			if (l_cCluster.iType != 1):
				self.lFiles.append(l_cCluster.pNode)
			else:
				l_lStack.append(l_cCluster.pElement1)
				l_lStack.append(l_cCluster.pElement2)   
		
#//////////////////////////////////////////////////////////////////////////////////////////
#														Utils
#//////////////////////////////////////////////////////////////////////////////////////////

def DBStart(p_sPath):
	
	if os.path.exists(p_sPath):
		os.remove(p_sPath)
	
	l_cDB = sqlite.connect(p_sPath)
	l_cCursorDB = l_cDB.cursor()
	
	sCmd = "create table Dist(ID1 integer, ID2 integer, Dst real)"
	l_cCursorDB.execute(sCmd)
	sCmd = "CREATE INDEX index_dst ON Dist (Dst ASC)"
	l_cCursorDB.execute(sCmd)
	sCmd = "CREATE INDEX index_ID1 ON Dist (ID1)"
	l_cCursorDB.execute(sCmd)
	sCmd = "CREATE INDEX index_ID2 ON Dist (ID2)"
	l_cCursorDB.execute(sCmd)
	
	l_cCursorDB.execute( "PRAGMA journal_mode = OFF")
	l_cCursorDB.execute( "PRAGMA synchronous = 0")
	l_cCursorDB.execute( "PRAGMA temp_store = 2") # store temp tables in memory
	l_cCursorDB.execute( "PRAGMA legacy_file_format = OFF")
	l_cCursorDB.execute( "PRAGMA locking_mode = EXCLUSIVE")
	
	l_cDB.commit()
	
	return l_cDB,l_cCursorDB
	
def DBStop(p_cDB,p_cCursorDB,p_sPath):
	p_cCursorDB.close()
	p_cDB.close()
	if os.path.exists(p_sPath):
		os.remove(p_sPath)
