﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using ICSharpCode.NRefactory.Semantics;
using ICSharpCode.NRefactory.TypeSystem;

namespace CSharpDeps.Scopes
{
    public class Class
    {
        public enum Types
        {
            Class,
            Struct,
            Interface,
            Enum,
        }

        public Types Type { get; set; }

        public Namespace Namespace { get; private set; }

        protected Dictionary<string, Class> children = new Dictionary<string, Class>();

        public Dictionary<string, Class> Children
        {
            get { return children; }
        }

        protected List<Method> methods = new List<Method>();

        public List<Method> Methods
        {
            get { return methods; }
        }

        protected List<Property> properties = new List<Property>();

        public List<Property> Properties
        {
            get { return properties; }
        }

        public ClassReference Inherits { get; set; }

        public List<ClassReference> implements = new List<ClassReference>();

        public List<ClassReference> Implements
        {
            get { return implements; }
        }

        public Class Parent { get; private set; }

        public string Name { get; private set; }

        public string FullName
        {
            get { return Namespace.IsGlobal() ? Name : string.Format("{0}.{1}", Namespace.FullName, Name); }
        }

        public Class(Types type, Namespace ns, Class parent, string name)
        {
            Type = type;
            Namespace = ns;
            Parent = parent;
            Name = name;
        }

        public void AddChild(Class child)
        {
            if (Children.ContainsKey(child.Name))
            {
                Children[child.Name].MergeClass(child);
            }
            else
            {
                Children[child.Name] = child;
            }
        }

        public void MergeClass(Class other)
        {
            foreach (Class klass in other.Children.Values)
            {
                AddChild(klass);
            }

            Methods.AddRange(other.Methods);
        }

        internal void AddMethod(Method method)
        {
            Methods.Add(method);
        }

        internal void AddProperty(MemberResolveResult property)
        {
            properties.Add(new Property
            {
                Name = property.Member.Name,
                Type = new ClassReference
                {
                    Namespace = property.Member.ReturnType.Namespace,
                    Name = property.Member.ReturnType.Name,
                    Type = TranslateKind(property.Member.ReturnType.Kind),
                },
            });
        }

        internal void SetInherits(IType type)
        {
            Inherits = new ClassReference
            {
                Type = TranslateKind(type.Kind),
                Namespace = type.Namespace,
                Name = type.Name,
            };
        }

        internal void AddImplements(IType type)
        {
            implements.Add(new ClassReference
            {
                Type = TranslateKind(type.Kind),
                Namespace = type.Namespace,
                Name = type.Name,
            });
        }

        public static Class.Types TranslateKind(TypeKind type)
        {
            switch (type)
            {
                case TypeKind.Class:
                    return Class.Types.Class;
                case TypeKind.Struct:
                    return Class.Types.Struct;
                case TypeKind.Enum:
                    return Class.Types.Enum;
                case TypeKind.Interface:
                    return Class.Types.Interface;
            }

            return Class.Types.Class;
        }
    }

    public class ClassReference
    {
        public Class.Types Type { get; set; }

        public string Namespace { get; set; }

        public string Name { get; set; }

        public string FullName
        {
            get { return string.Format("{0}.{1}", Namespace, Name).Trim('.'); }
        }
    }

    public class Property
    {
        public string Name { get; set; }

        public ClassReference Type { get; set; }
    }
}
