﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using LibGit2Sharp;
using LibGit2Sharp.Handlers;

namespace SolidTA.SCMImporter
{
    class GitServer : SCMServer
    {
        private static string REPO_DIR_PREFIX = "/extra/git_repo/";

        public string Url { get; private set; }
        public string User { get; private set; }
        public string Password { get; private set; }

        //Git repository
        private Repository repo;

        public override void Connect(string url, string directory, string user, string password)
        {
            // get the name of the repository, and check if it already exists, otherwise, clone it
            int idx1 = url.LastIndexOf("/");
            int idx2 = url.IndexOf(".git");
            if (idx2 == -1)
            {
                idx2 = url.Length;
            }

            string name = url.Substring(idx1, idx2 - idx1);
            string localRepoPath = directory + REPO_DIR_PREFIX;

            if (Directory.Exists(localRepoPath))
            {
                //repo was already created
                //update it if possible
                repo = new Repository(localRepoPath);
                FetchOptions options = new FetchOptions();
                options.CredentialsProvider = new CredentialsHandler(
                    (urli, usernameFromUrl, types) =>
                        new UsernamePasswordCredentials()
                        {
                            Username = user,
                            Password = password
                        });

                // It needs a signature for possible merges, this should not happen, since we don't change code
                Remote remote = repo.Network.Remotes["origin"];
                repo.Network.Fetch(remote, options);
            }
            else
            {
                //repo does not exist
                //clone it first
                var co = new CloneOptions();
                co.IsBare = true;
                
                co.CredentialsProvider = (_url, _user, _cred) => new UsernamePasswordCredentials {
                    Username = user,
                    Password = password
                };
                Repository.Clone(url, localRepoPath, co);
                Directory.CreateDirectory(localRepoPath);
                repo = new Repository(localRepoPath);
            }
        }

        public override string PreparePath(string path)
        {
            if (string.IsNullOrEmpty(path))
                return "";

            return path;
        }

        public override VersionSpec GetFirstVersionAfter(DateTime dt)
        {
            int revisionCounter = 0;
            foreach (Commit c in repo.Commits)
            {
                if (0 < c.Author.When.CompareTo(dt))
                {
                    break;
                }

                revisionCounter++;
            }
            return new VersionSpec(repo.Commits.Count() - revisionCounter);
        }

        /**
         * param includeChanges Doesnt matter for git. Changes are always included
         * param includeDownloadInfo Also includes the file content in the Item object of each file. Attention: The files are copied into memory.
         **/
        public override IEnumerable<ChangeSet> QueryFileHistory(string path, RecursionType recursion, VersionSpec versionFrom, int maxCount, bool includeChanges, bool includeDownloadInfo)
        {
            //Since this is a local git repo, we collected everything already recursively and have all downloadInfo ready
            List<ChangeSet> changeSetList = new List<ChangeSet>();

            VersionSpec from = new VersionSpec(versionFrom.version - 1);
            VersionSpec to = new VersionSpec(versionFrom.version + maxCount);



            List <Commit> commits = commitsFromVersion(from, to);
            Commit[] commit_arr = commits.ToArray();

            //Special handling of the first commit
            if (versionFrom.version == 0)
            {
                Console.WriteLine(string.Format("{0:HH:mm:ss tt}: ", DateTime.Now) + "QueryFileHistory:" + "ProcessCommit:" + 1);

                ChangeSet changeSet = getChangeSet(null, commit_arr[0], includeDownloadInfo, 0);
                if (0 < changeSet.Changes.Count())
                {
                    changeSetList.Add(changeSet);
                }
            }

            // Convert each commit to a changeset
            for (int i = 1; i < commit_arr.Count(); i++)
            {
                int realCommitNumber = from.version+i;
                Console.WriteLine(string.Format("{0:HH:mm:ss tt}: ", DateTime.Now) + "QueryFileHistory:" + "ProcessCommit:"+realCommitNumber);
                
                ChangeSet changeSet = getChangeSet(commit_arr[i - 1], commit_arr[i], includeDownloadInfo, realCommitNumber);
                if (0 < changeSet.Changes.Count())
                {
                    changeSetList.Add(changeSet);
                }
            }

            return changeSetList;
        }

        private int getCommitNumber(Commit c)
        {
            int counter = 0;
            foreach (Commit compare in repo.Commits)
            {
                if (compare.Equals(c))
                {
                    return repo.Commits.Count() - 1 - counter;
                }
                counter++;
            }
            return -1;
        }

        public override int GetLatestChangesetId()
        {
            return repo.Commits.Count()-1;
        }

        /**
         * param includeDownloadInfo Also includes the file content in the Item object of each file. Attention: The files are copied into memory.
         **/
        public override Item[] GetItems(string path, VersionSpec version, RecursionType recursion, DeletedState deletedState, Item.ItemType itemType, bool includeDownloadInfo)
        {
            List<Item> items = new List<Item>();
            Boolean excludeDeletedFiles = deletedState.Equals(DeletedState.NonDeleted);

            ChangeSet cs = getChangeSet(null,commitFromVersion(version), includeDownloadInfo, - 1);
            foreach (ChangeSet.Change change in cs.Changes)
            {
                if (excludeDeletedFiles && change.ChangeType.Equals(ChangeSet.ChangeType.Delete))
                {
                    Item currentItem = change.Item;
                    foreach (Item item in items)
                    {
                        if (Item.ItemType.Any.Equals(itemType) || item.itemType.Equals(itemType))
                        {
                            //The Equals method of Item also includes the fileStream, so items.Remove can not be used,
                            //because here we want to only differentiate between different locations, but not the complete Item
                            if (currentItem.ServerItem.Equals(item.ServerItem))
                            {
                                items.Remove(item);
                                break;
                            }
                        }
                    }
                }
                else
                {
                    items.Add(change.Item);
                }
            }

            return items.ToArray();
        }

        /*
         * gets the changeset between a earlier/previous commit and another Commit c
         * If commitNumber is unknown, pass -1 and this function will find it out (not so good performance)
         */
        private ChangeSet getChangeSet(Commit previous, Commit current, Boolean includeFileContent, int commitNumber)
        {
            Tree previousTree = (previous != null ? previous.Tree : null);
            Tree currentTree = (current != null ? current.Tree : null);
            //Build the diff between two commits
            TreeChanges diff = repo.Diff.Compare<TreeChanges>(previousTree, currentTree);

            //Process the diff
            //Dictionary<Item, ChangeSet.ChangeType> diff_d = new Dictionary<Item, ChangeSet.ChangeType>();
            IEnumerator<TreeEntryChanges> num = diff.GetEnumerator();
            List<ChangeSet.Change> changes = new List<ChangeSet.Change>();
            while (num.MoveNext())
            {
                TreeEntryChanges tec = num.Current;

                Item.ItemType itemType = ConvertHelper.FromMode(tec.Mode);
                string itemPath = "/" + tec.Path.Replace("\\", "/");
                Stream stream = includeFileContent ? getFileStreamFromOid(tec.Oid) : null;
                Item item = new Item(itemType, itemPath, stream);
                //diff_d.Add(item, ConvertHelper.FromChangeKind(tec.Status));
                ChangeSet.Change change = new ChangeSet.Change(item, ConvertHelper.FromChangeKind(tec.Status));
                changes.Add(change);
            }

            //Convert diff_d to a change list

            /*foreach (KeyValuePair<Item, ChangeSet.ChangeType> kv in diff_d)
            {
                ChangeSet.Change change = new ChangeSet.Change(item, ConvertHelper.FromChangeKind(tec.Status));
                changes.Add(change);
            }*/

            //Return changeSet
            int id = commitNumber != -1 ? commitNumber: getCommitNumber(current);
            DateTime dt = current.Author.When.DateTime;
            string author = current.Author.ToString();
            return new ChangeSet(id, dt, author, current.Message, changes);
        }

        private Stream getFileStreamFromOid(ObjectId objectid)
        {
            Blob blob = repo.Lookup<Blob>(objectid);
            return blob != null ? blob.GetContentStream() : null;
        }

        private Commit commitFromVersion(VersionSpec versionFrom)
        {
            //Commits are ordered by date with the newest/latest coming first and the oldest being last
            int revisionSkipCounter = versionFrom.Equals(VersionSpec.Latest) ? 0 : repo.Commits.Count() - versionFrom.version;
            foreach (Commit c in repo.Commits.ToArray())
            {
                if (0 < revisionSkipCounter)
                {
                    //this revision is not in the scope
                    revisionSkipCounter--;
                    continue;
                }
                return c;
            }
            throw new ArgumentException("This commit does not exist");
        }

        private List<Commit> commitsFromVersion(VersionSpec versionFrom, VersionSpec versionTo)
        {
            List<Commit> repoCommits = repo.Commits.ToList();
            //Commits are ordered by date with the newest/latest coming first and the oldest being last
            repoCommits.Reverse();

            int offsetCounter = Math.Max(versionFrom.version, 0);
            int revisionToIncludeCounter = versionTo.version - versionFrom.version;
            
            List<Commit> commits = repoCommits.Skip(offsetCounter).Take(revisionToIncludeCounter).ToList();
            return commits;
        }

        public override RepoOperationMode GetRepoMode()
        {
            return RepoOperationMode.VERTICAL;
        }
    }

    /**
    * Static class, which contains wrapper function to convert from the LibSharp2Git types to the SCMServer specific classes
    **/
    public static class ConvertHelper
    {
        public static ChangeSet.ChangeType FromChangeKind(ChangeKind changeKind)
        {
            switch (changeKind)
            {
                case ChangeKind.Added:
                case ChangeKind.Copied:
                    return ChangeSet.ChangeType.Add;
                case ChangeKind.Deleted:
                    return ChangeSet.ChangeType.Delete;
                case ChangeKind.Modified:
                case ChangeKind.Renamed:
                case ChangeKind.TypeChanged:
                    return ChangeSet.ChangeType.Edit;
                case ChangeKind.Ignored:
                case ChangeKind.Unmodified:
                case ChangeKind.Unreadable:
                case ChangeKind.Untracked:
                    return ChangeSet.ChangeType.None;
                default:
                    Console.WriteLine("Unexpected ChangeKind: " + changeKind.ToString());
                    throw new NotImplementedException();
            }
        }

        public static Item.ItemType FromMode(Mode mode)
        {
            switch (mode)
            {
                case Mode.Directory:
                case Mode.SymbolicLink:
                    return Item.ItemType.Folder;
                case Mode.ExecutableFile:
                case Mode.NonExecutableFile:
                case Mode.NonExecutableGroupWritableFile:
                    return Item.ItemType.File;
                case Mode.Nonexistent:
                case Mode.GitLink:
                    return Item.ItemType.None; // Ignore these times
                default:
                    Console.WriteLine("Unexpected Mode: " + mode.ToString());
                    throw new NotImplementedException();
            }
        }
    }
}
