import wx
import math
import random

import sta_utils
import sta_data
import sta_globals
import cushions
import sta_gl_mixerunit

from wx import glcanvas
from OpenGL.GL import *
from OpenGL.GLUT import *

#///////////////////////////////////////////////////////////////////////////////
#															OpenGL Panel
#///////////////////////////////////////////////////////////////////////////////
class glPanel(wx.Panel):
	def __init__(self, *args, **kwds):
		wx.Panel.__init__(self, *args, **kwds)
		self.Bind(wx.EVT_SIZE, self.OnResize)	

	def __do_layout(self):
		self.Layout()
	
	def OnResize(self,evt):
		self.children = self.GetChildren()
		if len(self.children)==1:
			self.size = self.GetSize()
			self.children[0].SetSize(self.size)

#///////////////////////////////////////////////////////////////////////////////
#															Base Canvas
#///////////////////////////////////////////////////////////////////////////////
class CanvasBase(glcanvas.GLCanvas):
	def __init__(self, parent):
		self.bDataInit = False
		glcanvas.GLCanvas.__init__(self, parent,-1, style=wx.NO_BORDER|wx.WANTS_CHARS,attribList=[glcanvas.WX_GL_RGBA,glcanvas.WX_GL_DOUBLEBUFFER,glcanvas.WX_GL_MIN_ALPHA,8,glcanvas.WX_GL_MIN_RED,8,glcanvas.WX_GL_MIN_BLUE,8,glcanvas.WX_GL_MIN_GREEN,8])
		self.bDataInit = True
		
		self.cGUI			= parent
		
		self.init			= False
		self.bRendering		= False
		self.cGlist			= 0
		self.lPixels		= []		#Snapshot image
		
		self.fSizeX			= 0
		self.fSizeY			= 0
		
		self.lastx = self.x = 30
		self.lasty = self.y = 30
		self.bMouseIn		= False
		
		self.Bind(wx.EVT_ERASE_BACKGROUND, self.OnEraseBackground)
		self.Bind(wx.EVT_SIZE, self.OnSize)
		self.Bind(wx.EVT_PAINT, self.OnPaint)
		self.Bind(wx.EVT_LEFT_DOWN, self.OnLeftMouseDown)
		self.Bind(wx.EVT_LEFT_DCLICK, self.OnLeftMouseDclick)
		self.Bind(wx.EVT_RIGHT_DCLICK, self.OnRightMouseDclick)
		self.Bind(wx.EVT_RIGHT_DOWN, self.OnRightMouseDown)
		self.Bind(wx.EVT_LEFT_UP, self.OnLeftMouseUp)
		self.Bind(wx.EVT_RIGHT_UP, self.OnRightMouseUp)
		self.Bind(wx.EVT_MOTION, self.OnMouseMotion)
		self.Bind(wx.EVT_LEAVE_WINDOW, self.OnMouseLeave)
		self.Bind(wx.EVT_ENTER_WINDOW, self.OnMouseEnter)
		self.Bind(wx.EVT_KEY_DOWN, self.OnKeyDown)
		self.Bind(wx.EVT_KEY_UP, self.OnKeyUp)
		self.Bind(wx.EVT_MOUSEWHEEL, self.OnMouseWheel)
		self.SetSize(parent.GetSize())
		
	#-------------------------------------- Drawing
	def InitGL(self):
		glMatrixMode(GL_MODELVIEW)
		glClearColor(1, 1, 1, 0)
	
	def OnEraseBackground(self, event):
		pass # Do nothing, to avoid flashing on MSW.
		
	def GetSize(self):
		self.size = self.GetClientSize()
		self.fSizeX = self.size.width
		self.fSizeY = self.size.height
		
	def OnSize(self, event):
		if self.bDataInit:
			self.GetSize()
		event.Skip()
		self.Refresh()
		
	def OnPaint(self, event):
		dc = wx.PaintDC(self)
		self.SetCurrent()
		if not self.init:
			self.InitGL()
			self.init = True
		self.OnDraw()
		
	def RefreshEx(self):
		self.Refresh()
		self.Update()
		
	def glListSetup(self):
		self.bRendering = True
		if (self.cGlist == 0):
			self.cGlist = glGenLists( 1 )
		glNewList( self.cGlist, GL_COMPILE)
		self.glMatrixSetup()
		
	def glListRun(self,bSnapshot = False):
		glEndList()
		glCallList(self.cGlist)
		glDeleteLists(self.cGlist,1)
		glFinish()
		
		if (bSnapshot):
			glPixelStorei(GL_PACK_ALIGNMENT, 1)
			self.lPixels = glReadPixelsb(0,0,self.size.width,self.size.height,GL_RGB)
		
		self.SwapBuffers()
		self.bRendering = False
		
	def glMatrixSetup(self):
		self.GetSize()
		glMatrixMode(GL_MODELVIEW)
		glLoadIdentity()
		glMatrixMode(GL_PROJECTION)
		glLoadIdentity()
		glOrtho(0,self.size.width,0,self.size.height,-10,10)
		glViewport(0, 0, self.size.width, self.size.height)
		glMatrixMode(GL_MODELVIEW)
		glDisable(GL_DEPTH_TEST)
		glClear(GL_COLOR_BUFFER_BIT)
		
	#------------------------------------- Draw text
	def drawString(self,x,y,p_sString):
		glRasterPos2f(x,y)
		for c in p_sString:  
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, ord(c))
			
	def drawGlowString(self,x,y,p_sString,cTextColor = (0,0,0),cGlowColor = (1.0,1.0,1.0)):
		glColor3fv(cGlowColor)
		for i in range(3):
			for j in range(3):
				self.drawString(x-1+i,y-1+j,p_sString)
		glColor3fv(cTextColor)
		self.drawString(x,y,p_sString)
	
	#------------------------------------- Save snapshot
	def cbSnapShot(self,evt):
		l_cDlg = wx.FileDialog(self.cGUI,style = wx.FD_SAVE, wildcard = "PNG files (*.png)|*.png")
		if (l_cDlg.ShowModal() == wx.ID_OK):
			sFilePath = l_cDlg.GetPath()
			
			pixels = sta_utils.flatten(self.lPixels)
			
			for i in range(len(pixels)):
				pixels[i] = int(pixels[i]*1.8)
			
			l_ImageHeight = self.size.height
			l_ImageWidth  = self.size.width
			
			l_offset 	= 0
			l_step 		= 3*l_ImageWidth
			
			l_lRawData	= range(l_ImageHeight)
			for i in range(l_ImageHeight):
				l_lRawData[i] = pixels[l_offset:l_offset+l_step]
				l_offset = l_offset+l_step
			l_lRawData.reverse()
			
			import png
			l_fileHandle = open(sFilePath, 'wb')
			l_PNGWriter = png.Writer(l_ImageWidth, l_ImageHeight)
			l_PNGWriter.write(l_fileHandle, l_lRawData) 
			l_fileHandle.close()
			
	#------------------------------------- Mouse and keys
	def OnLeftMouseDown(self, evt):
		pass
		
	def OnLeftMouseDclick(self,evt):
		pass
		
	def OnRightMouseDclick(self,evt):
		pass
		
	def OnRightMouseDown(self, evt):
		pass
		
	def OnLeftMouseUp(self, evt):        
		pass
		
	def OnRightMouseUp(self, evt):
		pass
		
	def OnMouseMotion(self, evt):
		if evt.Dragging() and evt.LeftIsDown():
			self.x, self.y = self.lastx, self.lasty
			self.x, self.y = evt.GetPosition()       
		
	def OnMouseEnter(self,evt):
		#self.CaptureMouse()
		#self.SetFocus()
		self.bMouseIn = True
		
	def OnMouseLeave(self,evt):
		#if self.HasCapture():
		#	self.ReleaseMouse()
		self.bMouseIn = False
		
	def OnKeyDown(self,evt):
		pass
		
	def OnKeyUp(self,evt):
		pass
		
	def OnMouseWheel(self,evt):
		pass
	
#///////////////////////////////////////////////////////////////////////////////
#																Custom Canvas
#///////////////////////////////////////////////////////////////////////////////
class MainCanvas(CanvasBase):

	def __init__(self, parent, *args, **kwds):
		CanvasBase.__init__(self, *args, **kwds)
		
		self.bUpdate		= True					# Reconfigure draw
		
		self.bShowClusters	= False
		
		self.cCurrentFile	= sta_data.STAFile()
		self.cNullFile		= sta_data.STAFile()
		self.cCurrentCommit	= sta_data.STACommit()
		self.bKeyCTRL 		= False
		self.bKeyALT 		= False
		self.bKeySHIFT 		= False
		
		self.cSelectedFile	= sta_data.STAFile()	# Currently clicked file
		self.cOriginFile	= sta_data.STAFile()	# Origin file for SHIFT selection
		self.lSelectedFiles	= []					# Used to construct a custom selection
		
		self.cGlist		= 0	
		self.bRendering		= False
		
		self.bDataInit		= True
		self.bShowClusterHint	= False
		self.fClusterHintY1	= 0
		self.fClusterHintY2	= 0
		
	def InitGL(self):
		
		# position viewer
		glMatrixMode(GL_MODELVIEW)
		glClearColor(1, 1, 1, 1.0)
		cushions.generateMainTextures(sta_globals.cMainCushions)
		sta_globals.fFontWidth = 6
		sta_globals.fFontHeight = sta_globals.fFontWidth * 2

#===============================================================================
# 															Key interaction
#===============================================================================
	
	#------------------------------------------------------------- Key down ---
	def OnKeyDown(self,evt):
		if (len(sta_globals.lSelectedFiles) == 0):
			return
		
		if (evt.GetKeyCode() == wx.WXK_ALT) and (not self.bKeyALT):
			self.bKeyALT = True
			if (sta_globals.cSelectedFile.sName != "-"):
				self.Refresh()
		
		if (evt.GetKeyCode() == wx.WXK_CONTROL) and (not self.bKeyCTRL):
			self.bKeyCTRL = True
			self.bKeySHIFT = False
			
		if (evt.GetKeyCode() == wx.WXK_SHIFT) and (not self.bKeySHIFT):
			self.bKeySHIFT = True
			self.bKeyCTRL = False
		
		if evt.GetKeyCode() == wx.WXK_UP:
			sta_globals.GUI.mainFrame.InfoDialog.g_tInfo.ScrollLines(-1)
		
		if evt.GetKeyCode() == wx.WXK_DOWN:
			sta_globals.GUI.mainFrame.InfoDialog.g_tInfo.ScrollLines(1)
		
	#------------------------------------------------------------- Key up ---
	def OnKeyUp(self,evt):
		if (len(sta_globals.lSelectedFiles) == 0):
			return
		
		if evt.GetKeyCode() == wx.WXK_ALT:
			self.bKeyALT = False
			if (sta_globals.cSelectedFile.sName != "-"):
				self.Refresh()
		
		if evt.GetKeyCode() == wx.WXK_CONTROL:
			if len(self.lSelectedFiles) >0:
				# Negative selection enabled
				if (self.bKeyALT):
					for i in self.lSelectedFiles:
						try:
							sta_globals.lSelectedFiles.remove(i)
						except:
							pass
				else:
					sta_globals.lSelectedFiles = self.lSelectedFiles
				sta_globals.GUI.HideFilterMetrics()
				sta_globals.GUI.mainFrame.g_listFiles.Unselect()
				sta_globals.GUI.mainFrame.g_listSelections.SetSelection(wx.NOT_FOUND)
				sta_globals.bSelectedFile = False
				sta_globals.GUI.UpdateSelectedFiles(False,False)
			
			self.bKeyCTRL = False
			self.lSelectedFiles = []
		
		if evt.GetKeyCode() == wx.WXK_SHIFT:
			if len(self.lSelectedFiles) >0:
				# Negative selection enabled
				if (self.bKeyALT):
					for i in self.lSelectedFiles:
						try:
							sta_globals.lSelectedFiles.remove(i)
						except:
							pass
				else:
					sta_globals.lSelectedFiles = self.lSelectedFiles
				sta_globals.GUI.HideFilterMetrics()
				sta_globals.GUI.mainFrame.g_listFiles.Unselect()
				sta_globals.GUI.mainFrame.g_listSelections.SetSelection(wx.NOT_FOUND)
				sta_globals.bSelectedFile = False
				sta_globals.GUI.UpdateSelectedFiles(False,False)
				
			self.bKeySHIFT = False
			self.lSelectedFiles = []
			
		sta_globals.GUI.RefreshEx()

#===============================================================================
# 															Mouse interaction
#===============================================================================
	def getTreeNode(self,l_tRootItem,l_lFile):
		
		bFound = False
		(tItem,cookie) = sta_globals.GUI.mainFrame.g_listFiles.GetFirstChild(l_tRootItem)
			
		if tItem.IsOk():
			while tItem.IsOk():
				
				sText = sta_globals.GUI.mainFrame.g_listFiles.GetItemText(tItem)
				
				if (sText == l_lFile[0]):              
					if len(l_lFile) > 0:                 
						return self.getTreeNode(tItem,l_lFile[1:])
					else:
						return tItem
				(tItem,cookie) = sta_globals.GUI.mainFrame.g_listFiles.GetNextChild(l_tRootItem,cookie)
				
		if len(l_lFile)>0:
			return self.getTreeNode(l_tRootItem,l_lFile[1:])
		else:
			return l_tRootItem

	#------------------------------------------------------ Left button down ---
	def OnLeftMouseDown(self, evt):
		if (len(sta_globals.lSelectedFiles) == 0):
			return
		
		self.cSelectedFile = self.cCurrentFile
		#-------------------------------------------------- Normal selection ---
		if (not self.bKeyCTRL) and (not self.bKeySHIFT):
			if (sta_globals.bSelectedFile) and (sta_globals.cSelectedFile.sPath == self.cCurrentFile.sPath):
				sta_globals.cSelectedFile = self.cNullFile
				sta_globals.GUI.mainFrame.g_listFiles.Unselect()
				sta_globals.bSelectedFile = False
			else:
				sta_globals.cSelectedFile = self.cCurrentFile   
				sta_globals.bSelectedFile = True
				
				l_lFile = self.cCurrentFile.sPath.split("/")
				tRoot = sta_globals.GUI.mainFrame.g_listFiles.GetRootItem()
				tItem = self.getTreeNode(tRoot,l_lFile)
				sta_globals.GUI.SelectItemAux(tItem)
		#---------------------------------------------------- CTRL selection ---
		elif (self.bKeyCTRL):
			if (sta_globals.bSelectedFile) and len(self.lSelectedFiles) == 0:
				self.lSelectedFiles.append(sta_globals.cSelectedFile)
			if self.cSelectedFile in self.lSelectedFiles:
				self.lSelectedFiles.remove(self.cSelectedFile)
			else:
				self.lSelectedFiles.append(self.cSelectedFile)
		#--------------------------------------------------- SHIFT selection ---
		elif (self.bKeySHIFT):
			l_bSw = True
			#-------------------------------- Start click ---
			if len(self.lSelectedFiles) == 0:
				if (sta_globals.bSelectedFile):
					self.cOriginFile = sta_globals.cSelectedFile                    
				else:
					self.cOriginFile = self.cSelectedFile
					l_bSw = False
				self.lSelectedFiles.append(self.cOriginFile)
			#---------------------------------- End click ---
			if l_bSw:
				self.lSelectedFiles = []
				if (self.cSelectedFile.iPos < self.cOriginFile.iPos):
					for i in sta_globals.lSelectedFiles:
						if (i.iPos >= self.cSelectedFile.iPos) and (i.iPos <= self.cOriginFile.iPos):
							self.lSelectedFiles.append(i)
				elif (self.cSelectedFile.iPos > self.cOriginFile.iPos):
					for i in sta_globals.lSelectedFiles:
						if (i.iPos >= self.cOriginFile.iPos) and (i.iPos <= self.cSelectedFile.iPos):
							self.lSelectedFiles.append(i)
			
		#-----------------------------------------------------------------------
		
		sta_globals.GUI.RefreshEx()
		
	#----------------------------------------------------- Right button down ---
	def OnRightMouseDown(self, evt):   
		if (len(sta_globals.lSelectedFiles) == 0):
			return
		
		self.PopupMenu(sta_globals.GUI.mainFrame.cMenu)       
	#----------------------------------------------------------- Mouse wheel ---
	def OnMouseWheel(self,evt):
		if (len(sta_globals.lSelectedFiles) == 0):
			return
		
		#---------------------------------------------------- Zoom ---
		if evt.m_controlDown:
			if evt.m_altDown:#------ X ---
				if evt.m_wheelRotation < 0:
					sta_globals.fZoomX = sta_globals.fZoomX + 0.25
				else:              
					sta_globals.fZoomX = sta_globals.fZoomX - 0.25
					if sta_globals.fZoomX < 1:
						sta_globals.fZoomX = 1              
				sta_globals.GUI.cbZoomX(0)
			else:#------------------ Y ---              
				if evt.m_wheelRotation < 0:                 
					sta_globals.fZoomY = sta_globals.fZoomY + 0.25
				else:
					sta_globals.fZoomY = sta_globals.fZoomY - 0.25
					if sta_globals.fZoomY < 1:
						sta_globals.fZoomY = 1              
				sta_globals.GUI.cbZoomY(0)
		#------------------------------------------------- Scroll ---
		else:                            
			if evt.m_altDown:#------ X ---     
				
				fNrPixels = 30.0
				if (sta_globals.iSelectedFilesInterval == 0):
					return                 
				l_iScrollStep = fNrPixels / (sta_globals.fRatioX * sta_globals.iSelectedFilesInterval) * sta_globals.iScrollResolution              
				
				sta_globals.iScrollPosX = sta_globals.GUI.mainFrame.g_scrollX.GetThumbPosition()           
				if evt.m_wheelRotation < 0:
					sta_globals.iScrollPosX = sta_globals.iScrollPosX + l_iScrollStep
				else:
					sta_globals.iScrollPosX = sta_globals.iScrollPosX - l_iScrollStep
				sta_globals.GUI.mainFrame.g_scrollX.SetScrollbar(sta_globals.iScrollPosX,sta_globals.iScrollSizeX,sta_globals.iScrollResolution,sta_globals.iScrollSizeX)
				sta_globals.GUI.cbScrollX(0)
			else:#------------------ Y ---
				
				fNrPixels = 20.0                 
				l_iScrollStep = fNrPixels / (sta_globals.fRatioY * sta_globals.fSizeY) * sta_globals.iScrollResolution
				
				if l_iScrollStep < 1:
					l_iScrollStep = 1
				
				sta_globals.iScrollPosY = sta_globals.GUI.mainFrame.g_scrollY.GetThumbPosition()           
				if evt.m_wheelRotation < 0:
					sta_globals.iScrollPosY = sta_globals.iScrollPosY + l_iScrollStep
				else:
					sta_globals.iScrollPosY = sta_globals.iScrollPosY - l_iScrollStep
				sta_globals.GUI.mainFrame.g_scrollY.SetScrollbar(sta_globals.iScrollPosY,sta_globals.iScrollSizeY,sta_globals.iScrollResolution,sta_globals.iScrollSizeY)
				sta_globals.GUI.cbScrollY(0)
				
	#---------------------------------------------------------- Mouse motion ---
	def OnMouseMotion(self, evt):
		
		self.SetFocus()
		
		if (len(sta_globals.lSelectedFiles) == 0):
			return
		
		self.x, self.y = evt.GetPosition()
		self.x = max(self.x,0)
		self.x = min(self.x,self.size.width)
		self.y = max(self.y,0)
		self.y = min(self.y,self.size.height)
		
		self.bFullCurrentInfo	= False		# Both file and commit are known
		self.cCurrentFile   	= 0
		self.cCurrentCommit 	= 0
		
		if len(sta_globals.lSelectedFiles)>0:
		#-------------------------------------- detect file ---        
			iVirtualY = float(self.y) / sta_globals.fRatioY + sta_globals.fOffsetY
			iYOffset  = 0
			iIdxFile  = 0
			l_iLen = len(sta_globals.lSelectedFiles)
			while (iYOffset < iVirtualY) and (iIdxFile < l_iLen):           
				iYOffset = iYOffset + 1
				iIdxFile = iIdxFile + 1
			iIdxFile = iIdxFile - 1
			self.cCurrentFile = sta_globals.lSelectedFiles[iIdxFile]
			
			if (sta_globals.iSelectedFilesInterval == 0):
				sta_globals.GUI.cbBrush(self.cCurrentFile,0,"",False)
				return
			
			# mouse date                 
			l_iDate = int(float(self.x) / sta_globals.fRatioX  + sta_globals.fOffsetX)
			l_sDate = sta_utils.getDate(l_iDate)
			
			#----------------------- detect revision ---
			iIdxCommit = 0
			
			while iIdxCommit<len(sta_globals.lSelectedFiles[iIdxFile].lRevs):
				cCommit = sta_globals.lSelectedFiles[iIdxFile].lRevs[iIdxCommit]
				if cCommit.iTime > l_iDate:
					break
				iIdxCommit = iIdxCommit + 1
			
			iIdxCommit = iIdxCommit - 1
			
			if iIdxCommit>=0:
				self.cCurrentCommit = sta_globals.lSelectedFiles[iIdxFile].lRevs[iIdxCommit]
				self.bFullCurrentInfo = True
			
			#--------------------------------- print information ---
			
			if iIdxCommit>=0:
				sta_globals.GUI.cbBrush(self.cCurrentFile,self.cCurrentCommit,l_sDate,True)
			else:
				sta_globals.GUI.cbBrush(self.cCurrentFile,0,l_sDate,False)              

#===============================================================================
#															Size related updates
#===============================================================================
	
	#---------------------------------------------------------- Update setup ---
	def updateRatios(self):
		
		#--- X axis measures		
		if (sta_globals.iSelectedFilesInterval) >0:
			sta_globals.fRatioX 	= float(sta_globals.fSizeX)/sta_globals.iSelectedFilesInterval * sta_globals.fZoomX
		else:
			sta_globals.fRatioX 	= 1.0
		sta_globals.fUnitX = 1.0 / sta_globals.fRatioX
			
		#--- Y axis measures
		if (len(sta_globals.lSelectedFiles) != 0):
			sta_globals.fRatioY = float(sta_globals.fSizeY)/len(sta_globals.lSelectedFiles) * sta_globals.fZoomY
		else:
			sta_globals.fRatioY = 1.0
		sta_globals.fUnitY = 1.0/sta_globals.fRatioY
		
		if (sta_globals.fRatioY<1):
			sta_globals.fUnitFile = int(1.0 / sta_globals.fRatioY)
		else:
			sta_globals.fUnitFile = 1
				    
	#------------------------------------------------------- Update position ---
	def updatePosition(self):
		iPosY = 0
		for i in sta_globals.lSelectedFiles:
			i.iPos	= iPosY
			iPosY	+= 1
	#------------------------------------------------------- Update Offset X ---
	def updateOffsetX(self,fRatio):
		sta_globals.fOffsetX = sta_globals.iSelectedFilesInterval * fRatio + sta_globals.iSelectedFilesStartTime 
	
	#------------------------------------------------------- Update Offset Y ---
	def updateOffsetY(self,fRatio):
		sta_globals.fOffsetY = len(sta_globals.lSelectedFiles) * fRatio
		
	#----------------------------------------------------------- Update size ---
	def OnSize(self, event):    	
		self.size = self.GetClientSize()                
		sta_globals.fSizeX = self.size.width
		sta_globals.fSizeY = self.size.height        
		
		self.updateOffsetX(sta_globals.fScrollRatioX)
		self.updateOffsetY(sta_globals.fScrollRatioY)
		
		if self.bDataInit:
			self.updateRatios()
			self.updatePosition()
		
		event.Skip()
		self.Refresh()

#===============================================================================
#															 Draw operations
#===============================================================================
	def OnDraw(self):
		if self.bRendering:
			return
		self.glListSetup()
		
		#------------------------------------------------------------------------
		# Start drawing
		#------------------------------------------------------------------------
		if len(sta_globals.lSelectedFiles)>0:
			#------------------------------------------- Update configuration ---
			if (self.bUpdate):
				
				self.updateRatios()			# fRatioX,fRatioY
				self.updatePosition()		# Position on Y axis
				
				sta_globals.GUI.cbFitToScreen_aux(0)
				
				self.bUpdate = False
			#--------------------------------------------------- matrix setup ---
			glTranslatef(0,sta_globals.fSizeY,0)	  
			glPushMatrix()
			
			glScalef(sta_globals.fRatioX,-sta_globals.fRatioY,1)
			glTranslatef(-sta_globals.fOffsetX,-sta_globals.fOffsetY,0)
			
			#---------------------------------------------------- File window ---
			
			iNrFiles = len(sta_globals.lSelectedFiles)
			iNrFilesDisplayed = int(iNrFiles / sta_globals.fZoomY) + 1
			if (iNrFilesDisplayed > iNrFiles):
				iNrFilesDisplayed = iNrFiles
			
			i = 0
			while (i < iNrFiles) and (sta_globals.fOffsetY > sta_globals.lSelectedFiles[i].iPos+1):
				i += 1
			iIdxStartFile = i
			
			iIdxEndFile = iIdxStartFile+iNrFilesDisplayed + 1
			if iIdxStartFile < 0:
				iIdxStartFile = 0
			if iIdxEndFile > iNrFiles:
				iIdxEndFile = iNrFiles
			lFilesRange = range(iIdxStartFile,iIdxEndFile)
			
			#-------------------------------------------------  Draw cushions ---
			if (sta_globals.fRatioY>3):
				for i in lFilesRange:
					l_iPosY = sta_globals.lSelectedFiles[i].iPos
					cushions.drawCushion1DH(\
							0,\
							l_iPosY,\
							sta_globals.iSelectedFilesEndTime,\
							l_iPosY+sta_globals.fUnitFile,\
							sta_globals.cMainCushions,2,True)
			
			#---------------------------------------------------- Draw colors ---
			glBlendFunc(GL_DST_ALPHA,GL_ZERO)
			glEnable(GL_BLEND)
					
			for i in lFilesRange:
				if (i% sta_globals.fUnitFile == 0):
					self.drawFileColors(sta_globals.lSelectedFiles[i])
			
			glDisable(GL_BLEND)
			#-------------------------------------------- Draw marker density ---
			glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA)
			glEnable(GL_BLEND)
			
			if (sta_globals.sMarker4 != ''):
				for i in lFilesRange:
					self.drawMarkerDensity(sta_globals.lSelectedFiles[i])
			
			glDisable(GL_BLEND)
			#--------------------------------------------- Draw marker center ---
			if (sta_globals.sMarker3 != ''):
				for i in lFilesRange:
					self.drawMarkerCenter(sta_globals.lSelectedFiles[i])
			
			#---------------------------------------------------- Draw labels ---
			if (sta_globals.bFileLabels):
				for i in lFilesRange:
					if (i% sta_globals.fUnitFile == 0):
						self.drawFileLabels(sta_globals.lSelectedFiles[i])
			
			#--------------------------------------------------------------------
			#							 Draw clusters
			#--------------------------------------------------------------------
			if (self.bShowClusters) and (sta_globals.cClusterEngine.HasClusters() != 0):
					
				l_iClusterLevelIdx = sta_globals.getClusterLevelTableIdx(sta_globals.iClusterLevel)
				#------------------------------------------------ Dendrogram ---
				l_fStartX	= sta_globals.fOffsetX
				l_fSizeX	= sta_globals.fOffsetX+sta_globals.fSizeX*sta_globals.fUnitX
				#--------------------------------------------------- level 0 ---
				lCushionMap = []
				lCushionMap.extend(sta_globals.cClusterEngine.lLayout[l_iClusterLevelIdx])
				
				#cushion clusters
				glBlendFunc(GL_ZERO,GL_SRC_ALPHA)
				glEnable(GL_BLEND)
				yPos = 0
				bSwitch = True
				for i in lCushionMap:
					cushions.drawCluster2D(\
						l_fStartX,\
						yPos,\
						l_fSizeX,\
						yPos+i[0],\
						sta_globals.fUnitX,\
						sta_globals.fUnitY,\
						sta_globals.cMainCushions)
					yPos = yPos + i[0]
				glDisable(GL_BLEND)
				
				#---------------------------------------------- cushion hint (from cluster map) ---
				if self.bShowClusterHint:
					glColor3f(0.5,0,0)
					sta_utils.drawRectangle(l_fStartX,self.fClusterHintY1,l_fSizeX,self.fClusterHintY2,6)
					glColor3f(1,1,1)
					sta_utils.drawRectangle(l_fStartX,self.fClusterHintY1,l_fSizeX,self.fClusterHintY2,2)
			#--------------------------------------------------------------------
			#						Draw selected files
			#-------------------------------------------------------------------- 
			
			if len(self.lSelectedFiles)>0:
				self. drawSelectedSet()
			
			if (sta_globals.bSelectedFile):
				self.drawSelectedFile(sta_globals.cSelectedFile)
			
			#------------
			glPopMatrix()
			
			#--------------------------------------- show ticks selected file ---
			if (self.bKeyALT) and (sta_globals.cSelectedFile.sName != "-"):
				glPushMatrix()
				#-------------
				glScalef(sta_globals.fRatioX,-self.size.height,1)
				glTranslatef(-sta_globals.fOffsetX,0,0)
				if (self.cCurrentFile.sPath == sta_globals.cSelectedFile.sPath):
					self.drawFileTick(sta_globals.cSelectedFile,self.cCurrentCommit)
				#-------------
				glPopMatrix()
			
		else:
			glColor3f(0.85,0.85,0.95)
			glRectf(0, 0, self.size.width, self.size.height)
			glColor3f(0.25,0.25,0.55)
			if (sta_globals.cSTAPrj.sName == ''):
				l_sText = "No project loaded"
			else:
				l_sText = "No data available"
			self.drawText(self.size.width/2,self.size.height/2,l_sText,1)
		#------------------------------------------------------------------------
		# End drawing
		#------------------------------------------------------------------------
		self.glListRun()
		
		# Save image
		glPixelStorei(GL_PACK_ALIGNMENT, 1)
		sta_globals.lPixelsEvolution = glReadPixels(0,0,self.size.width,self.size.height,GL_RGB,GL_UNSIGNED_BYTE)
		
	#------------------------------------------- draw selected set highlight ---
	def drawSelectedSet(self):
		
		if (sta_globals.fUnitY<0.8):
			l_width = 2
		else:
			l_width = 1
		
		l_fMinY = -1
		l_fMaxY = 0
		
		#---------------------------------------------------- CTRL selection ---
		if self.bKeyCTRL:
			for i in self.lSelectedFiles:
				
				l_fX1 = sta_globals.fOffsetX
				l_fX2 = l_fX1+sta_globals.fUnitX*10
				l_fY2 = i.iPos
				l_fY1 = i.iPos+1
				
				glColor3f(0.75,0.91,0.34)
				glRectf(l_fX1,l_fY1,l_fX2,l_fY2)
				
				glColor3f(1,1,0)	    
				glLineWidth(l_width+1)
				glBegin(GL_LINE_LOOP)
				glVertex2f(sta_globals.iSelectedFilesStartTime,i.iPos)
				glVertex2f(sta_globals.iSelectedFilesEndTime  ,i.iPos)
				glVertex2f(sta_globals.iSelectedFilesEndTime  ,i.iPos+sta_globals.fUnitFile-sta_globals.fUnitY)
				glVertex2f(sta_globals.iSelectedFilesStartTime,i.iPos+sta_globals.fUnitFile-sta_globals.fUnitY)	       
				glEnd()
				
		#--------------------------------------------------- SHIFT selection ---
		if self.bKeySHIFT:
			for i in self.lSelectedFiles:
				if (l_fMinY == -1) or (i.iPos < l_fMinY):
					l_fMinY = i.iPos
				if (i.iPos+1) > l_fMaxY:
					l_fMaxY = i.iPos+1
			
			glColor3f(0.75,0.91,0.34)
			glRectf(sta_globals.iSelectedFilesStartTime,l_fMinY,sta_globals.iSelectedFilesStartTime+10*sta_globals.fUnitX,l_fMaxY)
			
			glColor3f(1,1,0)	    
			glLineWidth(l_width+1)
			glBegin(GL_LINE_LOOP)
			glVertex2f(sta_globals.iSelectedFilesStartTime,l_fMinY)
			glVertex2f(sta_globals.iSelectedFilesEndTime  ,l_fMinY)
			glVertex2f(sta_globals.iSelectedFilesEndTime  ,l_fMaxY-sta_globals.fUnitY)
			glVertex2f(sta_globals.iSelectedFilesStartTime,l_fMaxY-sta_globals.fUnitY)	       
			glEnd()
		#-----------------------------------------------------------------------
		
		glLineWidth(1)
	#------------------------------------------------------ draw file colors ---
	def drawFileColors(self,cFile):
		
		glColor3f(1,1,1)					# draw segment
		glRectf(0,cFile.iPos,sta_globals.iSelectedFilesEndTime,cFile.iPos+sta_globals.fUnitFile)
		
		if (len(cFile.lRevs)==0):
			return
			
		if (cFile.sName != "-"):			#file information present
			
			#-------------------------------------------------- draw commits ---
			l_fMinTime = 3*sta_globals.fUnitX
			cCommit1 = cFile.lRevs[0]
			for i in range(len(cFile.lRevs)-1):	               
				cCommit2 = cFile.lRevs[i+1]
				if ((cCommit2.iTime - cCommit1.iTime) > l_fMinTime):
				
					#-------------------------- choose color ---
					glColor3fv(self.getCommitColor(cCommit1,cFile))
					
					#------------------------- draw file info ---
					
					glRectf(\
						cCommit1.iTime,\
						cFile.iPos,\
						cCommit2.iTime,\
						cFile.iPos+sta_globals.fUnitFile)
					
				#--------------------------- draw commits ---
				glColor3f(0.5,0.0,0.0)
				glRectf(\
					cCommit1.iTime-sta_globals.fUnitX,\
					cFile.iPos,\
					cCommit1.iTime + sta_globals.fUnitX,\
					cFile.iPos+sta_globals.fUnitFile)
					
				cCommit1 = cCommit2
					
			#----------------------------------------------- draw last active ---
			#if cFile.iState == 1:
			
			# segment
			if (cFile.iStatus == 0):
				glColor3fv(self.getCommitColor(cCommit1,cFile))
				glRectf(\
						cFile.iEnd,\
						cFile.iPos,\
						sta_globals.iSelectedFilesEndTime,\
						cFile.iPos+sta_globals.fUnitFile) 
			# commit
			glColor3f(0.9,0.0,0.0)
			glRectf(\
					cCommit1.iTime - sta_globals.fUnitX,\
					cFile.iPos,\
					cCommit1.iTime + sta_globals.fUnitX,\
					cFile.iPos+sta_globals.fUnitFile)
	#------------------------------------------------------ get commit color ---
	def getCommitColor(self,l_cCommit,l_cFile):
		r = 0
		g = 0
		b = 0
		
		# TODO: only compute this when modifying the preset controller; buffer the color otherwise
		
		#Compute total contribution
		for i in sta_globals.lDisplayedFilters_Project:
			l_unit = sta_globals.lcUnits_Project[i.sClass]            
			lColor = l_unit.cPlugin.getElementColor(l_cCommit,l_cFile)	       	       
			l_r = lColor[0]*l_unit.fContribution
			l_g = lColor[1]*l_unit.fContribution
			l_b = lColor[2]*l_unit.fContribution
			r = r + l_r
			g = g + l_g
			b = b + l_b	       
	    
		#Make color gray if commit out of reach
		if sta_globals.fMixRatio_Project < 1:	       
			fRatio = 1 - sta_globals.fMixRatio_Project
			r = r + 0.75 * fRatio
			g = g + 0.75 * fRatio
			b = b + 0.75 * fRatio	    
		
		return (r,g,b)
	#------------------------------------------------------ draw file labels ---
	def drawFileLabels(self,cFile):
		if sta_globals.fRatioY > sta_globals.fFontHeight:
			glColor3f(0,0,0)
			glRasterPos2f(sta_globals.fOffsetX+sta_globals.fUnitX*5,cFile.iPos+1-sta_globals.fUnitY*5)
			for c in cFile.sPath:  
				glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, ord(c))
	#--------------------------------------------------- draw marker density ---
	def drawMarkerDensity(self,cFile):
		l_cUnit = sta_globals.lcUnits_Project[sta_globals.sMarker4]
		if l_cUnit.bActive and (l_cUnit.iAttrType == 0) and l_cUnit.cPlugin.bSelected:
			for i in cFile.lRevs:
				if l_cUnit.cPlugin.isItemSelected(i):
					
					if (sta_globals.iMode4 == 0):
						l_lColor = l_cUnit.cPlugin.getElementColor(i,cFile)
					else:
						l_lColor = (1,0,0)
					
					l_lColor = l_lColor + (sta_globals.fMarker4Intensity,)
					glColor4fv(l_lColor)
					
					l_fx = i.iTime
					l_fy = cFile.iPos + 0.5
					
					l_fSizeX = sta_globals.fUnitX*sta_globals.fMarker4Detail
					l_fSizeY = sta_globals.fUnitY*sta_globals.fMarker4Detail
					
					sta_utils.drawNDisk((l_fx,l_fy),(l_fSizeX,l_fSizeY))
	#---------------------------------------------------- draw marker center ---
	def drawMarkerCenter(self,cFile):            
		l_cUnit = sta_globals.lcUnits_Project[sta_globals.sMarker3]
		if l_cUnit.bActive and (l_cUnit.iAttrType == 0) and l_cUnit.cPlugin.bSelected:
			for i in cFile.lRevs:
				if l_cUnit.cPlugin.isItemSelected(i):
					
					glColor3fv(l_cUnit.cPlugin.getElementColor(i,cFile))
					l_fx = i.iTime
					l_fy = cFile.iPos + 0.5
					l_fSizeX = sta_globals.fUnitX*sta_globals.iMarker3Size
					l_fSizeY = sta_globals.fUnitY*sta_globals.iMarker3Size
					
					cushions.drawCushion2D(l_fx-l_fSizeX,l_fy-l_fSizeY,l_fx+l_fSizeX,l_fy+l_fSizeY,sta_globals.cMainCushions,18,True)
					glBlendFunc(GL_DST_ALPHA,GL_ZERO)
					glEnable(GL_BLEND)
					glRectf(l_fx-l_fSizeX,l_fy-l_fSizeY,l_fx+l_fSizeX,l_fy+l_fSizeY)
					glDisable(GL_BLEND)
	#---------------------------------------------------- draw selected file ---
	def drawSelectedFile(self,cFile):
		
		if (sta_globals.fUnitY<0.8):
			l_width = 2
		else:
			l_width = 1
		
		fX1 = sta_globals.fOffsetX
		fX2 = fX1+sta_globals.fUnitX*(len(cFile.sPath)*sta_globals.fFontWidth)
		fY2 = cFile.iPos
		if (fY2 == 0):
			fY1 = cFile.iPos+sta_globals.fUnitFile
			fY2 = fY1+sta_globals.fUnitY*sta_globals.fFontHeight
		else:
			fY1 = fY2-sta_globals.fUnitY*sta_globals.fFontHeight
		
		glColor3f(1,0.8,0.6)
		glRectf(fX1,fY1,fX2,fY2)
		
		glColor3f(1,1,0)
		glLineWidth(l_width+1)
		glBegin(GL_LINE_LOOP)
		glVertex2f(sta_globals.iSelectedFilesStartTime,cFile.iPos )
		glVertex2f(sta_globals.iSelectedFilesEndTime,cFile.iPos  )
		glVertex2f(sta_globals.iSelectedFilesEndTime,cFile.iPos+sta_globals.fUnitFile)
		glVertex2f(sta_globals.iSelectedFilesStartTime,cFile.iPos+sta_globals.fUnitFile)	       
		glEnd()
		
		glColor3f(1,0,0)	    
		glLineWidth(l_width)
		glBegin(GL_LINE_LOOP)
		glVertex2f(sta_globals.iSelectedFilesStartTime,cFile.iPos )
		glVertex2f(sta_globals.iSelectedFilesEndTime, cFile.iPos  )
		glVertex2f(sta_globals.iSelectedFilesEndTime, cFile.iPos+sta_globals.fUnitFile)
		glVertex2f(sta_globals.iSelectedFilesStartTime,cFile.iPos+sta_globals.fUnitFile)	       
		glEnd()
		
		glBegin(GL_LINE_LOOP)
		glVertex2f(fX1,fY1)
		glVertex2f(fX2,fY1)
		glVertex2f(fX2,fY2)
		glVertex2f(fX1,fY2)
		glEnd()
		
		glLineWidth(1)
		glColor3f(0.5,0,0)
		glRasterPos2f(fX1+sta_globals.fUnitX*5,fY2-sta_globals.fUnitY*3)
		for c in cFile.sPath:  
			glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, ord(c))
	#-------------------------------------------------------- draw file tick ---
	def drawFileTick(self,cFile,cCommit):
		
		if isinstance(cCommit,int):
			return
		
		#-------------------------------------- detect mode --
		lRevs = cFile.lRevs
		
		#---------------------------------------- draw tick ---
		for i in range(len(lRevs)):
			l_cCommit = lRevs[i]	       
			if (lRevs[i].sID == cCommit.sID):
			
				glColor3f(0.2,0.2,0.0)
				glRectf(\
					l_cCommit.iTime-sta_globals.fUnitX,\
					0,\
					l_cCommit.iTime,\
					1)

				if i<len(lRevs)-1:
					l_cCommit = lRevs[i+1]
	             
					glColor3f(0.2,0.2,0.0)
					glRectf(\
						l_cCommit.iTime,\
						0,\
						l_cCommit.iTime + sta_globals.fUnitX,\
						1)
	#---------------------------------------------------------- Draw name ---
	def drawText(self,x,y,p_sText,mode):  
	
		# mode = 0 -> Left aligned
		# mode = 1 -> Center aligned
		# mode = 2 -> Right aligned
		
		l_fLen = len(p_sText)*9
		
		if mode == 1:
			glRasterPos2f(x-l_fLen/2, y)
		elif mode == 2:
			glRasterPos2f(x-l_fLen, y)
		else:
	 		 glRasterPos2f(x, y)
			 
		for c in p_sText:  
			 glutBitmapCharacter(GLUT_BITMAP_9_BY_15, ord(c))